# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import importlib
import sys

class LazyModule:
   """LazyModule behaves just like an imported module, but it does not
   actually import the requested module until some attribute of it is
   referenced.  You can use it like this:

   replace
     import rpm
   with
     rpm = LazyModule( 'rpm' )


   LazyModule forwards getattr, setattr, and hasattr to the real
   module, but with introspection (type, dir, isinstance), of course,
   the two are different.

   The importHook is a one-argument function to invoke immediately
   after the module is first imported.  The argument is the
   just-imported module.  This is useful for replacing top-level code
   that imports a module and then configures it, like this:

     import rpm
     rpm.setVerbosity( 1 )

   with

     def hook( rpm ): rpm.setVerbosity(1)
     rpm = LazyModule( 'rpm', hook )

   Note that this feature may be included in a future release of Python 3
   (https://peps.python.org/pep-0690/), which would likely obsolete this class.
   """

   __slots__ = ( '__lazy__', '__hook__', '__mod__' )

   def __init__( self, name, importHook=None ):
      object.__setattr__( self, '__lazy__', name )
      object.__setattr__( self, '__hook__', importHook )
      object.__setattr__( self, '__mod__', None )

   def _loadmod( self ):
      return importlib.import_module( self.__lazy__ )

   def _getmod( self ):
      if self.__mod__:
         # This is a bit faster than looking up in sys.modules
         return self.__mod__
      module = sys.modules.get( self.__lazy__ )
      if not module:
         # nobody has imported this module, now load it
         module = self._loadmod()
         if self.__hook__:
            self.__hook__( module )
      object.__setattr__( self, '__mod__', module )
      return module

   def __getattr__( self, x ):
      return getattr( self._getmod(), x )

   def __setattr__( self, x, value ):
      return setattr( self._getmod(), x, value )

   def __hasattr__( self, x ):
      return hasattr( self._getmod(), x )

   def __repr__( self ):
      # pylint: disable-next=consider-using-f-string
      return "<{} {} ({}loaded)>".format( self.__class__.__name__,
                                          self.__lazy__,
                                          '' if self.__mod__ else "not " )
