# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import atexit
from abc import ABCMeta
from contextlib import contextmanager
from copy import copy
from enum import Enum
from itertools import groupby

###
# Base Exception class
###

class ArException( Exception ):
   '''Useful base class exception, that allows adding any kwargs, and prints it in a
   nice easy to read, and abug format.

   Arguments:
      message - A string to be printed when the exception is thrown
      kwargs - Named arguments that will be printed after the message is printed

   Additionally, this class overrides the __str__ method so that the string rep of
   the exception consists of the message and the dictionary
   '''

   def __init__( self, message, **kwargs ):
      self.message = message
      self.messageDict = kwargs
      super().__init__()

   def __str__( self ):
      stringRep = [ self.message ]
      for key in sorted( self.messageDict ):
         value = self.messageDict[ key ]
         try:
            stringRep.append( key + " : " + str( value ) )
         except Exception as e: # pylint: disable=broad-except
            stringRep.append( f"ERROR:\n   {key}:{type( e )}{e}" )
      return '\n'.join( stringRep )

###
# Tristate type, basically an enum which cannot be evaluated as a bool.
###

class Tristate( Enum ):
   FALSE = 0
   TRUE = 1
   DONTCARE = 'X'

   def __bool__( self ):
      raise TypeError( f"Invalid conversion of {self} to boolean" )

   # Hashing and comparison follow the same design as IntEnum.
   # Tristate( True ) == True
   # hash( Tristate( True ) ) == hash( True )
   def __hash__( self ):
      # Tristate instances are immutable.
      return hash( self.value )

   def __eq__( self, thing ):
      if isinstance( thing, Tristate ):
         return thing.value == self.value
      if thing == self.value:
         return True
      return False

   def __ne__( self, thing ):
      return not self.__eq__( thing )

###
# Pretty print lists of integers
###

class PrettyIntReprList( list ):
   '''Turn large int lists into human readable reprs.'''

   def __init__( self, *args, **kwargs ):
      self.mlen = max( 2, kwargs.pop( 'mlen', 4 ) )
      self.recursive = kwargs.pop( 'recursive', True )
      super().__init__( *args, **kwargs )

   def __repr__( self ):
      # return short notation for ranges and repetitions lists
      def repr_( k, v ):
         inc = v[ 1 ] - v[ 0 ] if k == 'seq' else 0
         adj = 1 if inc > 0 else - 1
         if k == 'blob':
            return repr( v )
         if k == 'rep' or inc == 0:
            return '[%s] * %d' % ( repr( v[ 0 ] ), len( v ) )
         inc = ', %s' % inc if inc != 1 else ''
         return 'list(range(%d, %d%s))' % ( v[ 0 ], v[ -1 ] + adj, inc )

      # pylint: disable=unidiomatic-typecheck
      isType = lambda a, b: type( a ) == b

      # shorten nested lists recursively: turn off to make debugging easier
      if self.recursive:
         # pylint: disable-next=self-cls-assignment
         self = PrettyIntReprList( [ PrettyIntReprList( x, mlen=self.mlen )
            if isType( x, list ) else x for x in self ], mlen=self.mlen )

      # base case following recursion
      if len( self ) < 2:
         return super().__repr__()

      # convert non-integers to nans
      ints = [ n if isType( n, int ) else float( 'NaN' ) for n in self ]
      # calculate deltas between integer elements
      diff = [ b - a for a, b in zip( ints, ints[ 1 : ] ) ]
      # calculate a repeat count for each delta
      reps = [ ( x, len( list( y ) ) ) for x, y in groupby( diff ) ]
      # create a sequence of unique integers
      uniq = iter( range( 0, len( ints ) + 1 ) )
      # create a key generator that hands out one unique key per sequence
      keys = []
      kgen = lambda: groupby( enumerate( self ), lambda kv: keys[ kv[ 0 ] ] )

      adj = 0

      # greedy match the medians [ 1, 2, ((( 3 ))) , 5, 7 ]
      for delta, count in reps:
         [ count, adj ] = [ count + adj, 0 ]
         if not isType( delta, int ):
            keys = keys + [ -1 ] * count
         elif count + 1 >= self.mlen:
            keys = keys + [ next( uniq ) ] * ( count + 1 )
            adj = -1
         elif count != 0:
            keys = keys + [ -1 ] * count

      keys.append( -1 )

      # helper lambdas to flatten and label repetition lists and ranges
      flatten = lambda items : [ x for item in items for x in item ]
      hasReps = lambda v : ( 'blob' if len( v ) < self.mlen else 'rep', v )
      notRange = lambda v : [ hasReps( list( v ) ) for k, v in groupby( v ) ]
      isRange = lambda k, v : notRange( v ) if k < 0 else [ ( 'seq', list( v ) ) ]
      isBlob = lambda pair: pair[ 0 ] == 'blob'

      # use kgen to separate mlen integer sequences from everything else
      groups = [ ( k, list( zip( *v ) )[ 1 ] ) for k, v in kgen() ]

      # break apart everything else to look for repetitions
      groups = flatten( [ isRange( k, v ) for k, v in groups ] )
      groups = [ ( k, list( v ) ) for k, v in groupby( groups, isBlob ) ]

      flattenPair = lambda items : [ x for k, v in items for x in v ]
      groups = [ [ ( 'blob', flattenPair( v ) ) ] if k else v for k, v in groups ]

      return ' + '.join( [ repr_( k, v ) for k, v in flatten( groups ) ] )

###
# NamedLambda
###

class NamedLambda:
   '''Give a lambda a nice name.
   For example:
      fcn=NambedLambda( "TimesTwoMethod", lambda x: x*2 )
      fcn( 3 ) # returns 6
      repr( fcn ) # returns "TimesTwoMethod"
   '''
   def __init__( self, name, lambdaMethod ):
      self.lambdaMethod = lambdaMethod
      self.name = name
      self.__name__ = name
      super().__init__()

   def __call__( self, *args, **kwargs ):
      return self.lambdaMethod( *args, **kwargs )

   def __repr__( self ):
      return self.name

# Examples and useful named lambdas
returnNone = NamedLambda( "lambda: None", lambda *args, **kwargs: None )
isString = NamedLambda( "isString(x)", lambda x: isinstance( x, str ) )
isBool = NamedLambda( "isBool(x)", lambda x: x in [ True, False ] )

###
# Singleton
###

class Singleton( type ):
   '''To make a singleton class, just add the keyword metaclass=Singleton,
   eg
   class MyClass( metaclass=Singleton ):'''
   _instances = {}

   def __new__( cls, name, bases, dct ):
      # Called on class creation.  This ensures deepcopy maintains the Singleton.
      # NOTE: __new__ is a staticmethod, even though it has no @staticmethod
      assert '__deepcopy__' not in dct, "Singleton classes must not support copy."
      dct[ '__deepcopy__' ] = lambda self, memo: self
      return super().__new__( cls, name, bases, dct )

   def __call__( cls, *args, **kwargs ):
      # Called on object creation.
      if cls not in cls._instances:
         obj = super().__call__( *args, **kwargs )
         cls._instances[ cls ] = obj
      return cls._instances[ cls ]

# lessen __del__ race during shutdown by explicitly delete cache before shutdown
atexit.register( Singleton._instances.clear )  # pylint: disable=protected-access

###
# Factory class
###

class FactoryDefs:
   '''This contains all the helper class definitions for the Factory Class
   Included are
      1) All the custom exceptions which all derive from FactoryException
      2) MetaDescriptors
   This is not intended for standalone use.  The Factory class should be used.
   '''

   ### Exceptions
   class FactoryException( ArException ):
      pass

   class ExpectedValueError( FactoryException ):
      def __init__( self, fieldName, expectedVal, realVal, **kwargs ):
         message = ( "Unexpected val in {}, expected: {}, got: {}".format(
            fieldName, expectedVal, realVal ) )
         super().__init__( message, **kwargs )

   class ExtraMetaAttributesError( FactoryException ):
      def __init__( self, cls, attribute, **kwargs ):
         try:
            clsName = cls.__name__
         except AttributeError:
            clsName = str( cls )
         message = ( "Extra meta attributes: {} specified that is not required/used "
                     "by generated test class {}" ).format( attribute, clsName )
         super().__init__( message, **kwargs )

   class IncompatibleClassMetaAttributeError( FactoryException ):
      def __init__( self, attrName, attrType, objType, **kwargs ):
         message = "{} of type {} is used in classmethod on {}".format(
            attrName, attrType, objType )
         super().__init__( message, **kwargs )

   class MissingMetaAttributesError( FactoryException ):
      def __init__( self, cls, attributes, **kwargs ):
         message = ( "Required meta attributes: {} have not been set on the test "
                     "class : {}" ).format( attributes, cls.__name__ )
         super().__init__( message, **kwargs )

   class OverrideMetaAttrError( FactoryException ):
      pass

   class SettingReadOnlyMetaAttributeError( FactoryException ):
      def __init__( self, clsName, attribute, **kwargs ):
         message = ( "Setting read only meta attribute '{}' of "
                     "class '{}'" ).format( attribute, clsName )
         super().__init__( message, **kwargs )

   class SettingInvalidValue( FactoryException ):
      def __init__( self, obj, attribute, value, **kwargs ):
         message = ( "Setting invalid value '{}' on attribute '{}' on '{}'" ).format(
                     value, attribute, obj )
         super().__init__( message, **kwargs )

   class UninitializedMetaAttribute( FactoryException ):
      def __init__( self, name, cls, **kwargs ):
         message = ( "Uninitialized meta attribute: {} on class : {}" ).format(
                     name, cls.__name__ )
         super().__init__( message, **kwargs )

   class MixinError( FactoryException ):
      '''Base class for all Factory Exceptions dealing with mixins.'''

   class IncompatibleMixinError( MixinError ):
      def __init__( self, mixin, incompatibility, reason, **kwargs ):
         message = "Mixin: {} incompatible with: {} because {}".format(
            mixin, incompatibility, reason )
         super().__init__( message, **kwargs )

   class MetaAttrCollisionMixinError( MixinError ):
      def __init__( self, clsName, attr, mixin1, **kwargs ):
         message = "Class {}: Processing mixin {}: " \
                   "Already has metaAttribute {}".format( clsName, mixin1, attr )
         super().__init__( message, **kwargs )

   class NonMetaAttrSettingMixinError( MixinError ):
      def __init__( self, clsName, attr, **kwargs ):
         message = "Class {}: Meta attribute {} overriden by " \
                   "non-MetaAttribute".format( clsName, attr )
         super().__init__( message, **kwargs )

   ### Descriptors
   class MetaDescriptor( ABCMeta ):
      '''This MetaClass wraps the python class creation with extra logic for the
      MetaAttributes.'''
      # pylint: disable-next=arguments-differ
      def __new__( cls, clsname, bases, cls_kwargs ):
         baseMetaAttrs = {}
         # Find all the baseMetaAttrs & look for collisions
         for base in bases:
            if issubclass( base, FactoryDefs ):
               thisMetaAttrs = base.getMetaAttributes()
               for attr in thisMetaAttrs:
                  attrSettings = thisMetaAttrs[ attr ]
                  if isinstance( attrSettings,
                                 FactoryDefs.MetaAttributeOverrideParent ):
                     attrSettings = attrSettings.root()
                  if attr in baseMetaAttrs:
                     if baseMetaAttrs[ attr ].owner != attrSettings.owner:
                        raise FactoryDefs.MetaAttrCollisionMixinError( clsname,
                              attr, base )
                  baseMetaAttrs[ attr ] = attrSettings

         for base in bases:
            thisMetaAttrs = base.getMetaAttributes().keys() if \
                              issubclass( base, FactoryDefs ) else []
            for i in dir( base ):
               if i in baseMetaAttrs and not i in thisMetaAttrs:
                  raise FactoryDefs.NonMetaAttrSettingMixinError( clsname, i )

         # Strip out the meta attribute settings
         kwargsMetaAttrs = {}
         allAttrs = {}
         for base in bases:
            if issubclass( base, FactoryDefs ):
               allAttrs.update( base.getMetaAttributes() )
         for attrName in list( cls_kwargs.keys() ): # create list before modifying
            if attrName in allAttrs:
               attr = cls_kwargs[ attrName ]
               if not isinstance( attr, FactoryDefs.MetaAttributeOverrideParent ):
                  kwargsMetaAttrs[ attrName ] = attr
                  del cls_kwargs[ attrName ]

         # Create the new class
         newCls = super().__new__( cls, clsname, bases, cls_kwargs )

         # Since MetaAttributeOverrideParent looks for parent in the MRO, and there
         # could have been multiple mixins used, if we haven't create a new
         # MetaAttributeOverrideParent, but it exists in the parent, create a new
         # instance, so it can re-look up parent and get the proper MRO.
         for attrName, clsAttr in newCls.getMetaAttributes().items():
            if attrName not in newCls.__dict__ and \
                  isinstance( clsAttr, FactoryDefs.MetaAttributeOverrideParent ):
               clsAttr = FactoryDefs.MetaAttributeOverrideParent()
               setattr( newCls, attrName, clsAttr )
               # NOTE: PY3 doesn't call this, as this is added after class creation
               clsAttr.__set_name__( newCls, attrName )
            clsAttr.setup()

         # now set the new meta attributes for this class
         for attrName in kwargsMetaAttrs: # pylint: disable=consider-using-dict-items
            setattr( newCls, attrName, kwargsMetaAttrs[ attrName ] )

         return newCls

      def __setattr__( cls, name, value ):
         metaAttrs = cls.getMetaAttributes()
         if name in metaAttrs and \
               not isinstance( value, FactoryDefs.MetaAttributeOverrideParent ):
            metaAttrs[ name ].__set__( cls, value )
         else:
            super().__setattr__( name, value )

   class MetaAttribute:
      '''This is a python descriptor that controls the setters and getters of all
      MetaAttributes of the Factory Class.  The convience classes that derive from
      this, allow a slightly more user friendly API with less arguments needed.
      '''
      class attrTypes( Enum ):
         CLASS = 1 # attribute needed for @classmethod
         OBJECT = 2 # standard attribute. needed by __init__ time
         RUNTIME = 3 # attribute is created at runtime IE in a mixin

      def __init__( self, attrType=attrTypes.OBJECT, readOnly=False, valueCheck=None,
                    optional=False, defaultOption=None, options=None,
                    valueConversion=lambda x: x ):
         ''' Initialization of the MetaAttribute class.
         Args:
            attrType ( attrTypes ): storage type for the meta attribute.
            readOnly ( bool ): allow __set__ on the object's attribute.
            valueCheck ( lambda( x ) ): is x a valid value for the attribute
            optional ( bool ): does the attribute have a default value if none is
                               provided.
            defaultOption ( any ): the default value of an optional attribute
            options ( iterable ): container that a set attribute must be in
            valueConversion ( lambda( value ) ): does type or value conversion during
                                                 setting of the attribute.
         '''
         super().__init__()
         assert attrType in self.attrTypes
         self.attrType = attrType
         self.readOnly = readOnly
         if options:
            assert valueCheck is None
            self.valueCheck = NamedLambda( 'Option valueCheck',
                                           lambda val: val in options )
         else:
            self.valueCheck = valueCheck if valueCheck else lambda val: True
         self.valueConversion = valueConversion
         self.optional = optional
         if optional:
            self.defaultOption = self.valueConversion( defaultOption )
         else:
            assert defaultOption is None
            self.defaultOption = None
         self.name = None # Will be overriden shortly.  This keeps pylint happy
         self.owner = None # Will be overriden shortly.  This keeps pylint happy

      def __set_name__(self, owner, name ):
         self.name = name
         self.owner = owner

      def setup( self ):
         assert self.name
         assert self.owner

      def prvName( self ):
         '''This is where the actual data is stored on the instatiated child of the
         Factory class.  It shouldn't be used directly, and only within this class,
         and in the creation of a child class of Factory.
         '''
         return '_' + self.name

      def reprArgs( self ):
         return [ 'attrType', 'readOnly', 'valueCheck', 'optional', 'defaultOption' ]

      def __repr__( self ):
         try:
            args = [ f"{arg}={getattr( self, arg )}" for
                     arg in self.reprArgs() ]
            return "{}( {} ) # name={}\n".format( self.__class__.__name__,
                                                  ', '.join( args ),
                                                  self.name )
         except AttributeError: # Early startup debug
            return super().__repr__()

      def __get__( self, obj, objType ):
         thing = obj if obj else objType
         if ( obj is None ) and ( self.attrType != self.attrTypes.CLASS ):
            raise FactoryDefs.IncompatibleClassMetaAttributeError( self.name,
                                                                   self.attrType,
                                                                   objType )
         if not hasattr(thing, self.prvName() ):
            if self.optional:
               setattr(thing, self.prvName(), self.defaultOption )
            else:
               raise FactoryDefs.UninitializedMetaAttribute( self.name, objType )
         return getattr( thing, self.prvName() )

      def __set__( self, obj, value ):
         # valueConversion brings the literal value provided, into the right type
         # to have a consistent type( obj.prvName ) for example:
         # flag = MetaAttributeOptional( False, options=[ True, False ] )
         # and then:
         # self.flag = 1
         # We should store 1 as bool( 1 ) so that we have consistency in the type
         # and id of self.flag whether its set as = 1, = "True" or = True. So we
         # would write:
         # flag = MetaAttributeOptional( False, options=[ True, False ],
         #                               valueConversion=lambda x: bool( x ) )
         value = self.valueConversion( value )
         if isinstance( value, FactoryDefs.MetaAttribute ):
            # This is non-sensical, and 99.99% chance a bug.
            # If a valid use case comes up later, we'll revisit then.
            raise FactoryDefs.SettingInvalidValue( obj, self.name, value, debug=obj )
         if self.readOnly:
            if hasattr( obj, self.prvName() ) or (
                  not isinstance( obj, FactoryDefs ) and
                  hasattr( type( obj ), self.prvName() ) ):
               raise FactoryDefs.SettingReadOnlyMetaAttributeError( str(obj),
                                                                    self.name,
                                                                    debug=obj )
         if not self.valueCheck( value ):
            raise FactoryDefs.SettingInvalidValue( str(obj), self.name, value,
                                                   debug=obj )
         if callable( value ) and not isinstance( obj, FactoryDefs ):
            # If we don't do this, then there will be an inconsistency between
            # generateChildClass() and __init__()
            # Also this is a MetaAttribute, not a mixin
            # The use case is simply lambda expressions, not new class methods
            value = staticmethod( value )
         setattr( obj, self.prvName(), value )

   # The following are wrappers for convience
   class MetaAttributeRequired( MetaAttribute ):
      def __init__( self, **kwargs ):
         super().__init__(
            attrType=self.attrTypes.OBJECT, optional=False, **kwargs )

   class MetaAttributeOptional( MetaAttribute ):
      def __init__( self, defaultOption=None, **kwargs ):
         super().__init__(
            attrType=self.attrTypes.OBJECT, optional=True,
            defaultOption=defaultOption, **kwargs )

   class MetaAttributeRequiredClass( MetaAttribute ):
      def __init__( self, **kwargs ):
         super().__init__(
            attrType=self.attrTypes.CLASS, optional=False, **kwargs )

   class MetaAttributeOptionalClass( MetaAttribute ):
      def __init__( self, defaultOption=None, **kwargs ):
         super().__init__(
            attrType=self.attrTypes.CLASS, optional=True,
            defaultOption=defaultOption, **kwargs )

   class MetaAttributeRuntime( MetaAttribute ):
      class NullType:
         '''Simple wrapper to give a default type a user would never pass in.'''

      def __init__( self, default=NullType(), **kwargs ):
         assert 'optional' not in kwargs and 'defaultOption' not in kwargs
         if isinstance( default, FactoryDefs.MetaAttributeRuntime.NullType ):
            pass # No default was passed in
         else:
            kwargs[ 'optional' ] = True
            kwargs[ 'defaultOption' ] = default
         super().__init__(
            attrType=self.attrTypes.RUNTIME, **kwargs )

   # And the MetaAttribute to override select parts of the parent MetaAttribute
   class MetaAttributeOverrideParent( MetaAttribute ):
      def __init__( self, valueCheckChained=False, **kwargs ):
         super().__init__()
         self.parent = None
         self.valueCheckChained = valueCheckChained
         self.overrideParentArgs = kwargs

      def root( self ):
         ret = self
         while isinstance( ret, FactoryDefs.MetaAttributeOverrideParent ) and \
               ret.parent:
            ret = ret.parent
         return ret

      def setup( self ):
         super().setup()

         # set Parent Instance
         if not self.parent:
            parent = self
            for cls in self.owner.__mro__:
               if self.name in cls.__dict__:
                  clsAttr = cls.__dict__[ self.name ]
                  assert isinstance( clsAttr, FactoryDefs.MetaAttribute )
                  if clsAttr is not parent.owner.__dict__[ self.name ]:
                     parent.parent = copy( clsAttr )
                     parent = parent.parent
                     if not isinstance( clsAttr,
                                        FactoryDefs.MetaAttributeOverrideParent ):
                        break
            assert self.parent is not None
            assert self.parent is not self
         self.parent.setup()

         # Copy parent attrs
         for key in [ 'attrType', 'readOnly', 'optional', 'defaultOption',
                      'valueCheck' ]:
            setattr( self, key, getattr( self.parent, key ) )

         # Now override
         for key, value in self.overrideParentArgs.items():
            if key in [ 'attrType', 'readOnly', 'optional', 'defaultOption' ]:
               setattr( self, key, value )
            elif key == 'valueCheck':
               assert 'options' not in self.overrideParentArgs
               self.valueCheck =  value
            elif key == 'options':
               assert 'valueCheck' not in self.overrideParentArgs
               self.valueCheck = NamedLambda( 'Option valueCheck(Overriding Parent)',
                  lambda val: val in self.overrideParentArgs[ 'options' ] )
            else:
               assert False, f"unknown arg {key}"
         if self.valueCheckChained:
            captureValueCheck = self.valueCheck
            self.valueCheck = NamedLambda( 'Chained value check',
                                          lambda val: captureValueCheck( val ) and
                                                      self.parent.valueCheck( val ) )

      def reprArgs( self ):
         ret = super().reprArgs()
         ret.append( 'valueCheckChained' )
         return ret

   @classmethod
   def getMetaAttributes( cls, attrTypes=None ):
      '''This method returns a dict of every valid MetaAttribute defined on the class
      or its bases or a subset specified by an optional list arg of types.

      The returned dict follows the format:
         attributeName : attributeCls

      Even if meta attributes have been overwritten with actual values this function
      will discover them.

      Additionally, the returned MetaAttributes are guaranteed to follow python's
      MRO. Thus, if cls subclasses from B which subclasses from A and both these
      bases define the same meta attribute with different restrictions, B's
      attribute will be returned.
      '''
      return { attrName : attrCls for base in cls.__mro__[ : : -1 ]
                                  for attrName, attrCls
                                  in base.__dict__.items()
                                  if isinstance( attrCls, cls.MetaAttribute ) and (
                                     attrTypes is None or
                                     attrCls.attrType in attrTypes ) }

class Factory( FactoryDefs, metaclass=FactoryDefs.MetaDescriptor ):
   '''A factory class to create new classes.

   This class provides a set of routines that are useful for generating classes at
   runtime.  The generateChildClass is the method that does this.  The MetaAttributes
   allow the mixins and base class to define a set of requirements for how the final
   class should be generated, and all MetaAttributes are passed as kwargs in the
   generateChildClass method.  Generated classes can then be used as a new base
   class, and this can be called recursively.

   The use case for this is testing.  Tests tend to poke in different ways at the
   system they are testing.  As more tests are added with more variants, many tests
   may end up identical with the exception of only small bits of code.  This Factory
   class allows for mixins to override only a small directed part of the test class,
   while keeping the base class structure of how the testing should work.  Compared
   to just using base python multiple inheritance the MetaAttribute system allows for
   nice reprs, settings of defaults, restricting settings, debuggable exceptions and
   some protection from conflicting mixins.

   The original use case at Arista, is to generate a unique test class for parity
   testing on each table on the forwarding ASIC, we can also then add in mixins for
   different methods of triggering that parity error.
   For example the base class may trigger by reading the table through the control
   path, while a traffic mixin could create a UDP packet that would cause a lookup in
   that table to trigger the parity error.

   Meta Attributes on base Factory class:
      name ( optional ) - A simple print friendly name for the class/object.
      defaultAttrTrace ( optional ) - Print traces in overrideAttrs? Defaults to True
      t0 ( optional ) - A lambda to call for tracing level 0
      t1 ( optional ) - A lambda to call for tracing level 1
   '''

   # Meta Attribute
   name = FactoryDefs.MetaAttributeOptionalClass( valueCheck=isString )
   defaultAttrTrace = FactoryDefs.MetaAttributeOptional( True, valueCheck=isBool )

   # Set in a derived class for tracing
   t0 = FactoryDefs.MetaAttributeOptionalClass( returnNone )
   t1 = FactoryDefs.MetaAttributeOptionalClass( returnNone )

   def __str__( self ):
      if self.name:
         return str( self.name )
      else:
         return self.__class__.__name__

   def __repr__( self ):
      indent = '   ' # 3 space indent
      ret = []
      mro = type( self ).mro()
      # ignore this class (1st item in MRO), and object (last item in MRO)
      mro = mro[ 1 : -1 ]

      ret.append( f'Class "{self.__class__.__name__}" Info:' )
      ret.append( "MRO:" )
      for cls in mro:
         ret.append( f"{indent}{cls}" )
      ret.append( "Meta:" )
      for attr in sorted( self.getMetaAttributes() ):
         try:
            val = repr( getattr( self, attr ) )
         except Exception as ex: # pylint: disable=broad-except
            val = f'RAISES: {repr( ex )}'
         valList = val.split( '\n' )
         valList = [ i if len( i ) < 300
                     else i[ 0 : 300 ] + " **** VALUE TOO LONG TO DISPLAY ****"
                     for i in valList ]
         val = f'\n{indent}{indent}'.join( valList )
         ret.append( f'{indent}{attr}: {val}' )
      return '\n'.join( ret )

   @contextmanager
   def overrideAttrs( self, trace=None, unsetAttrs=None, **kwargs ):
      '''Used in a with: block to override specific attributes temporarily.
      To temporarily unset attributes place the attr name in a list in unsetAttrs.
      NOTE: Removing optional attrs will mean that the getter will return the default
      value.
      '''
      backup = {}
      uninit = []
      traceThis = trace or ( trace is None and self.defaultAttrTrace )
      metaAttrs = self.getMetaAttributes()

      def backUpAttr( attrName ):
         try:
            backup[ attrName ] = getattr( self, attrName )
         except FactoryDefs.UninitializedMetaAttribute:
            uninit.append( attrName )

      def removeAttr( attrName ):
         # There is no support for: del self[ attrName ]
         # So this goes through the private name in the metaAttribute itself
         prvName = metaAttrs[ attrName ].prvName()
         if hasattr( self, prvName ):
            delattr( self, prvName )

      def traceOverride( attrName, attrVal ):
         if traceThis:
            if attrName in backup:
               self.t1( "Overriding attr {} from {} to {}".format(
                        attrName, backup[ attrName ], attrVal ) )
            else:
               self.t1( "Overriding uninitialized attr {} to {}".format(
                        attrName, attrVal ) )

      try:
         for attrName, attrVal in kwargs.items():
            backUpAttr( attrName )
            setattr( self, attrName, attrVal )
            traceOverride( attrName, attrVal )
         if unsetAttrs is not None:
            for attrName in unsetAttrs:
               if attrName in kwargs:
                  raise self.OverrideMetaAttrError(
                     "overrideAttrs(..) is trying to both set and remove {}".format(
                        attrName ) )
               backUpAttr( attrName )
               removeAttr( attrName )
               traceOverride( attrName, 'uninitialized' )
         yield
      finally:
         for attrName, attrVal in backup.items():
            setattr( self, attrName, attrVal )
            if traceThis:
               self.t1( f"Restoring attr {attrName} to {attrVal}" )
         if uninit:
            for attrName in uninit:
               if traceThis:
                  self.t1( f"Removing attr {attrName}" )
               removeAttr( attrName )

   @classmethod
   def generateChildClass( cls, mixins=None, **kwargs ):
      '''Generates a subclass of the base class cls with the mixins applied and
      the contents of kwargs being added as attributes on the resultant object.
      '''
      if mixins is None:
         mixins = []

      # If we are applying a mixin that already exists in the class hierarchy then we
      # can cause all sorts of issues with python's MRO
      if any( issubclass( cls, mixin ) for mixin in mixins ):
         raise cls.IncompatibleMixinError( mixins, cls,
            "One or more mixins already exist on the class" )

      # For debugging purposes we need to generate a usefull class name
      generatedClassName = getattr( cls, "prettyName", cls.__name__ ) + "_"
      for mixin in mixins:
         generatedClassName += getattr( mixin, "prettyName", mixin.__name__ ) + "_"

      newCls = type( generatedClassName, tuple( mixins + [ cls ] ), {} )

      # Because this function is meant to support consecutive calls it is possible
      # that the caller wishes to override a meta attribute that has already been
      # set. Thus we have to query the python type system to check what was a meta
      # attribute and what was not.
      # It is important to pay attention to MRO because a mixin might have overridden
      # the optional meta attribute of a base to restrict its options, we need to
      # respect this ordering
      metaAttributes = newCls.getMetaAttributes()

      # Set all meta attributes from the supplied dict on the generated class
      # ensuring that each attribute has already been defined as a meta attribute
      # on the class. This enforces a certain "typeness" to the tests
      extraAttributes = []
      for key, value in kwargs.items():
         metaAttr = metaAttributes.get( key )
         if metaAttr is None:
            extraAttributes.append( key )
            continue
         setattr( newCls, key, value )
      if extraAttributes:
         raise cls.ExtraMetaAttributesError( newCls, extraAttributes, **kwargs )

     # Check attributes
      missingAttributes = []
      for attrName, attrCls in newCls.getMetaAttributes().items():
         assert attrName == attrCls.name
         if not attrCls.optional and ( attrCls.attrType == attrCls.attrTypes.CLASS ):
            try:
               getattr( newCls, attrCls.prvName() )
            except AttributeError:
               missingAttributes.append( attrName )
      if missingAttributes:
         raise cls.MissingMetaAttributesError( newCls, missingAttributes )

      return newCls

   def __init__( self, *args, **kwargs ):
      assert not args
      super().__init__()
      metaAttributes = self.getMetaAttributes()

      # Set all meta attributes from the supplied dict on the generated object
      # ensuring that each attribute has already been defined as a meta attribute
      # on the class. This enforces a certain "typeness" to the tests
      extraAttributes = []
      for key, value in kwargs.items():
         metaAttr = metaAttributes.get( key )
         if metaAttr is None:
            extraAttributes.append( key )
            continue
         setattr( self, key, value )
      if extraAttributes:
         raise self.ExtraMetaAttributesError( self, extraAttributes,
                                              **kwargs )

     # Check attributes
      missingAttributes = []
      for attrName, attrCls in self.getMetaAttributes().items():
         assert attrName == attrCls.name
         # pylint: disable-next=consider-using-in
         if ( attrCls.attrType == attrCls.attrTypes.CLASS or
              attrCls.attrType == attrCls.attrTypes.OBJECT ):
            if attrCls.optional:
               # Force all optional args to set.  So now readOnly=True will work.
               getattr( self, attrName )
            else: # Not optional
               try:
                  getattr( self, attrName )
               except self.UninitializedMetaAttribute:
                  missingAttributes.append( attrName )
      if missingAttributes:
         raise self.MissingMetaAttributesError( type( self ), missingAttributes )

# Deprecated Factory references.  Just use the direct ones inside the class
MetaAttributeOptional = FactoryDefs.MetaAttributeOptional
MetaAttributeRequired = FactoryDefs.MetaAttributeRequired
ExpectedValueError = FactoryDefs.ExpectedValueError
ExtraMetaAttributesError = FactoryDefs.ExtraMetaAttributesError
IncompatibleMixinError = FactoryDefs.IncompatibleMixinError
MissingMetaAttributesError = FactoryDefs.MissingMetaAttributesError
