#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import pytz
import time

def _nameLocalTime():
   """ Try and find a reasonable name for the timezone stored in
   /etc/localtime. Ideally, this should be a symlink, but isn't on many
   systems. We start with a list of commonly used zones, and see if the files
   in /usr/share/zoneinfo match /etc/localtime exactly. If not, we do an
   exhaustive search. This mostly means we stat each file in the
   subdirectories, and occasionally open and compare with the data in
   /etc/localtime if their sizes match. It's exhaustive, but fast. """

   wellKnownZones = {
      "US/Pacific",
      "US/Eastern",
      "Asia/Kolkata",
      "Europe/London",
      "Europe/Dublin",
      "Etc/Universal", # this is the same as posix/UTC
   }

   def fileContentsEqual( fname, data ):
      if not os.path.getsize( fname ) == len( data ):
         return False
      with open( fname, "rb" ) as f:
         return f.read() == data

   if not os.path.exists( "/etc/localtime" ):
      return None
   searchdir = "/usr/share/zoneinfo"
   if os.path.islink( "/etc/localtime" ):
      realpath = os.path.realpath( "/etc/localtime" )
      if realpath.startswith( searchdir + "/" ) and "posix/" not in realpath:
         return realpath[ len( searchdir ) + 1: ]

   with open( "/etc/localtime", "rb" ) as info:
      localData = info.read()

   for zone in wellKnownZones:
      if fileContentsEqual( searchdir + "/" + zone, localData ):
         return zone

   # No exact match for the common TZ's we use: try all of them, excluding
   # "posix" subdir and "localtime" files.
   best = None
   for ( parent, dirs, files ) in os.walk( searchdir ):
      if "posix" in dirs:
         dirs.remove( "posix" )
      if "localtime" in files:
         files.remove( "localtime" )
      for f in files:
         name = parent + "/" + f
         zoneName = name[ len( searchdir ) + 1: ]
         # prefer longer, more descriptive names
         if best is None or len( best ) < len( zoneName ):
            if fileContentsEqual( name, localData ):
               best = zoneName
   return best

def _getLocalPyTimeZone():
   """pytz module does NOT accept the timezone names from time.tzname: 'time'
   module gives a tuple of standard and day-light saving timezone names in
   abbreviated forms. Linux provides timezone data in the "Olson Database" in
   /usr/share/zoneinfo , which provide a separate timezone for a number of
   locations around the world. "/etc/localtime" is generally a copy of one of
   these files (or better, a symlink), and part of the information contained
   therein feeds the data in Python's time.tzname. The TZ environment variable
   can be used to override the /etc/localtime file by specifying where you are
   actually located: eg, TZ="Europe/Dublin" means that the time functions use
   the data in /usr/share/zoneinfo/Europe/Dublin

   Note that the abbreviated names in the timezones ("PST", "GMT", etc) are not
   enough to uniquely identify your timezone: in particular, "IST" is
   ambiguous, meaning either Indian or Irish Standard Time.

   If we can't make sense of the data in TZ env or /etc/localtime file, we fall
   back to the previous style guesswork.
   """

   tzName = None
   if 'TZ' in os.environ:
      # Local TZ setting means don't use /etc/localtime
      # pylint: disable-next=consider-using-f-string
      if os.path.exists( "/usr/share/zoneinfo/%s" % os.environ[ 'TZ' ] ):
         # Local TZ environment looks reasonable (exists in timezone database)
         tzName = os.getenv( 'TZ' )
   else:
      # Try and make sense of /etc/localtime
      tzName = _nameLocalTime()

   if not tzName:
      localTimezones = time.tzname
      if 'EST' in localTimezones:
         # for Bill
         tzName = 'US/Eastern'
      elif 'BST' in localTimezones:
         tzName = 'Europe/London'
      elif 'IST' in localTimezones:
         tzName = 'Asia/Kolkata'
      else:
         tzName = 'US/Pacific'
   try:
      tz = pytz.timezone( tzName )
   except pytz.exceptions.UnknownTimeZoneError:
      tz = pytz.timezone( "US/Pacific" )

   return ( tzName, tz )

( _localTzName, _localTzInfo ) = _getLocalPyTimeZone()

def localTzName():
   """Return a human-readible string representation of the user's local timezone"""
   return _localTzName

def localTzInfo():
   """Return an instance of pytz.timezone which can be consumed by
   python's datetime lib"""
   return _localTzInfo

def fromUtc( t ):
   if t.tzinfo is None:
      t = t.replace( tzinfo=pytz.timezone( 'UTC' ) )
   return t.astimezone( localTzInfo() )
