# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac

class GenerationWatcher( Tac.Notifiee ):
   """
   A watcher to keep track of and validate the changes to the generation of a
   Ark::GenerationEntity type.
   """
   notifierTypeName = "Ark::GenerationEntity"

   def __init__( self, *args, **kwargs ):
      super().__init__( *args, **kwargs )
      self.changeSequence = []

   @Tac.handler( 'generation' )
   def handleGeneration( self ):
      self.changeSequence.append( self.notifier().generation )

   def clearChanges( self ):
      self.changeSequence = []

   def checkGenerationChanges( self, expectedChanges ):
      assert len( self.changeSequence ) == len( expectedChanges ), \
            "Expected to see {} changes, but got {}".format(
                  len( expectedChanges ), len( self.changeSequence ) )
      for actual, expected in zip( self.changeSequence, expectedChanges ):
         assert actual == expected, \
               f"Expected to see generation {expected}, but got {actual}"

   def checkGenerationTransaction( self, expectedId ):
      if self.changeSequence:
         # Only want to check the last transaction
         assert len( self.changeSequence ) >= 2, "Don't have a full transaction"
         # pylint: disable=unbalanced-tuple-unpacking
         lastStartGen, lastEndGen = self.changeSequence[ -2 : ]
         assert lastStartGen.id == lastEndGen.id, \
               "Transation start id {} doesn't match end id {}".format(
                     lastStartGen.id, lastEndGen.id )
         assert not lastStartGen.valid, \
               f"Start of transaction {lastStartGen.id} is not invalid"
         assert lastEndGen.valid, \
               f"End of transaction {lastStartGen.id} is not valid"
         assert lastEndGen.id == expectedId, \
               "Expected to see transaction {}, but got {}".format( expectedId,
                                                                    lastEndGen.id )
      else:
         # We didn't see any changes, so check if we were already at the expected
         # generation. This isn't perfect since we don't know what the previous state
         # of the generation was, but we should have seen any transitions that
         # happened, so I think its pretty safe to just check this in lieu of that.
         self.checkGenerationMatches( expectedId )

   def checkGenerationMatches( self, expectedId, expectedValid=True ):
      assert self.notifier().generation.id == expectedId, \
            "Expected to see gen id {}, but got {}".format(
                  expectedId, self.notifier().generation.id )
      assert self.notifier().generation.valid == expectedValid, \
            "Expected to see gen valid {}, but got {}".format(
                  expectedValid, self.notifier().generation.valid )

