# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from TacUtils import Timeout

def waitForTaskRun( task, func, args=None, kwargs=None,
                    description=None, limit=200 ):
   """
   Runs the task @limit times until @func returns true
   @args is a list that will be passed as arguments to @func
   @kwargs is a dictionary that will be passed as keyword arguments to @func
   """
   if not isinstance( args, list ):
      if args is None:
         args = []
      else:
         args = [ args ]
   if kwargs is None:
      kwargs = {}
   if description is None:
      description = "task to run."

   oldForceYieldInTest = task.taskForceYieldInTest
   task.taskForceYieldInTest = True
   for _ in range( limit ):
      task.taskRunFromTest()
      if func( *args, **kwargs ):
         break
   task.taskForceYieldInTest = oldForceYieldInTest

   if not func( *args, **kwargs ):
      # pylint: disable-next=consider-using-f-string
      raise Timeout( "Timed out waiting for %s" % description )

def getTask( dut, agent, groupList, taskNameMatch, threadName='main',
             useDetail=False ):
   '''Get json output from agent task scheduler show command and
      descend TaskGroup hierarchy specified in groupList parameter
      to locate the task whose name matches the taskNameMatch
      parameter.'''
   detail = 'detail' if useDetail else ''
   j = dut.showCmdIs(
      f'show agent {agent} task scheduler {detail} match "{taskNameMatch}"',
      dataFormat='json' )
   # Descend the scheduler hierarchy to get to the task
   scheduler = j[ 'schedulers' ][ f'{agent}-{threadName}' ]
   for groupName in groupList:
      subSchedulers = scheduler[ 'subSchedulers' ]
      key = [ k for k in subSchedulers if groupName in k ]
      if not key:
         return None
      assert len( key ) == 1
      scheduler = subSchedulers[ key[ 0 ] ]
   tasks = scheduler[ 'tasks' ][ 'tasks' ]
   if not tasks:
      return None
   # The taskNameMatch should ensure we only get a single task from
   # the show command.
   assert len( tasks ) == 1
   return tasks[ 0 ]
