#!/usr/bin/env python3
# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import struct

import Ethernet
import random
from SysConstants.if_ether_h import ETH_FCS_LEN, ETH_P_8021Q
import Tac

EthHdrSize = 14
MacAddressSize = 6
VlanTagSize = 4
EthTypeSize = 2
EthDot1QHdrSize = 4

def _newDot1QHdr( p, currentOffset, vlanId, vlanPriority, cfiBit, ethType ):
   dot1QHdr = Tac.newInstance( "Arnet::Eth8021QHdrWrapper",
                                     p, currentOffset )
   currentOffset += EthDot1QHdrSize
   dot1QHdr.tagControlCfi = cfiBit

   if vlanPriority is None:
      vlanPriority = 0
   dot1QHdr.tagControlPriority = vlanPriority

   if vlanId is None:
      vlanId = 0
   dot1QHdr.tagControlVlanId = vlanId

   if isinstance( ethType, int ):
      dot1QHdr.typeOrLen = ethType
   else:
      assert type( ethType ) == str # pylint: disable=unidiomatic-typecheck
      dot1QHdr.ethType = ethType
   
   return( dot1QHdr, currentOffset )

# Create a packet with an ethernet header using the given values.
# Return a tuple of the pakcet, the ethernet header wrapper, the
# dot1Q header wrapper if there is one, and the current offset into
# the packet where the next header should go.
# If 'extraSize' is not specified, the returned packet is sized to hold
# exactly the needed ethernet headers.
def newEthPkt( srcMacAddr,
               dstMacAddr,
               outerVlanId=None,
               outerVlanPriority=None,
               outerCfiBit=False,
               vlanId=None,
               vlanTpid="ethTypeDot1Q",
               vlanPriority=None,
               cfiBit=False,
               ethType=0x1234,
               currentOffset=None,
               extraSize=0,
               p=None,
               outerVlanTpid="ethTypeQinQ",
               **kwargs ):

   if p is None:
      p = Tac.newInstance( "Arnet::Pkt" )
      size = EthHdrSize + extraSize
      if outerVlanId is not None or outerVlanPriority is not None:
         size += EthDot1QHdrSize
      if vlanId is not None or vlanPriority is not None:
         size += EthDot1QHdrSize
      p.newSharedHeadData = size
   if currentOffset is None:
      currentOffset = 0

   ethHdr = Tac.newInstance( "Arnet::EthHdrWrapper", p, currentOffset )
   currentOffset += EthHdrSize
   ethHdr.dst = dstMacAddr
   ethHdr.src = srcMacAddr
   if vlanId is None and vlanPriority is None:
      if isinstance( ethType, int ):
         ethHdr.typeOrLen = ethType
      else:
         assert type( ethType ) == str # pylint: disable=unidiomatic-typecheck
         ethHdr.ethType = ethType

      dot1QHdr = None
   else:
      if outerVlanId is not None or outerVlanPriority is not None:
         _outerDot1QHdr, currentOffset = \
             _newDot1QHdr( p, currentOffset, outerVlanId,
                          outerVlanPriority, outerCfiBit, vlanTpid )
         ethHdr.ethType = outerVlanTpid
      else:
         ethHdr.ethType = vlanTpid
         
      dot1QHdr, currentOffset = _newDot1QHdr( p, currentOffset,
                                             vlanId, vlanPriority,
                                             cfiBit, ethType )
         
   return (p, ethHdr, dot1QHdr, currentOffset)

Eth8022LlcStpSap = 0x42
Eth8022LlcSnapSap = 0xaa
Eth8022LlcPduControl = 3
Eth8022LlcHdrSize = 3

def newEth8022LlcPkt( ssap,
                      dsap,
                      control=Eth8022LlcPduControl,
                      extraSize=0,
                      **kwargs ):
   size = Eth8022LlcHdrSize
   (p, ethHdr, dot1qHdr, currentOffset ) = \
       newEthPkt( extraSize=(size + extraSize), **kwargs )
   llcHdr = Tac.newInstance( "Arnet::Eth8022LlcHdrWrapper", p, currentOffset )
   llcHdr.ssap = ssap
   llcHdr.dsap = dsap
   llcHdr.control = control
   currentOffset += Eth8022LlcHdrSize

   return (p, ethHdr, dot1qHdr, llcHdr, currentOffset)

Eth8022SnapPvstOrgCode = 0xc
Eth8022SnapPvstLocalCode = 0x010b
Eth8022SnapHdrSize = 5

def newEth8022SnapPkt( orgCode,
                       localCode,
                       extraSize=0,
                       **kwargs ):
   size = Eth8022SnapHdrSize
   (p, ethHdr, dot1qHdr, llcHdr, currentOffset ) = \
       newEth8022LlcPkt( ssap=Eth8022LlcSnapSap,
                         dsap=Eth8022LlcSnapSap,
                         extraSize=(size + extraSize),
                         **kwargs )
   snapHdr = Tac.newInstance( "Arnet::Eth8022SnapHdrWrapper", p, currentOffset )
   snapHdr.orgCode = orgCode
   snapHdr.localCode = localCode
   currentOffset += Eth8022SnapHdrSize

   return( p, ethHdr, dot1qHdr, llcHdr, snapHdr, currentOffset)
   
def tag( vlanId, vlanPriority, cfiBit, tpid=ETH_P_8021Q ):
   """Return an 802.1Q tag (as a 4-byte string) with the specified fields."""

   if vlanId is None:
      vlanId = 0
   if vlanPriority is None:
      vlanPriority = 0
   if cfiBit is None:
      cfiBit = 0

   assert 0 <= vlanId <= 4095
   assert 0 <= vlanPriority <= 7
   assert cfiBit in ( 0, 1 )
   vlanField = ( vlanPriority << 13 ) | ( cfiBit << 12 ) | vlanId
   return struct.pack( '!HH', tpid, vlanField )

def etherFrame( srcAddr, dstAddr, etherType=0x1234, size=96, 
                vlanId=None, vlanPriority=None, cfiBit=0, 
                innerVlanId=None, innerVlanPriority=None, innerCfiBit=0,
                content=None, contents=None, padChar=b'x', truncateContents=False,
                excludeFcsLen=False, customInnerTpid=None, customOuterTpid=None,
                **kwargs ):
   """Creates a ethernet frame with the given source and destination addresses.  If
   etherType is not specified, it defaults to 0x1234.  If size is not specified, it
   defaults to 96.  CRC is by default included in the size.  The payload of the frame
   will be set to the content specified.  The packet will be padded beyond the 
   content to the size specified if needed.
   """

   frame = b''
   frame += Ethernet.convertMacAddrToPackedString( dstAddr )
   frame += Ethernet.convertMacAddrToPackedString( srcAddr )

   # A lot of functions use "contents" instead of the original "content"
   # We now accept either
   assert content is None or contents is None
   if content is None:
      content = contents or b''
   if isinstance( content, bytearray ):
      content = bytes( content )
   elif isinstance( content, str ):
      content = content.encode()
   if vlanId is not None or vlanPriority is not None:
      if customOuterTpid is not None:
         frame += tag( vlanId, vlanPriority, cfiBit,
                       tpid=customOuterTpid )
      else:
         frame += tag( vlanId, vlanPriority, cfiBit )
      if innerVlanId is not None or innerVlanPriority is not None:
         if customInnerTpid is not None:
            frame += tag( innerVlanId, innerVlanPriority, innerCfiBit,
                          tpid=customInnerTpid )
         else:
            # Use default, which as of time of writing is 0x8100.
            frame += tag( innerVlanId, innerVlanPriority, innerCfiBit )
   else:
      assert innerVlanId is None
      assert innerVlanPriority is None

   frame += struct.pack( '!H', etherType )

   if excludeFcsLen:
      size += ETH_FCS_LEN

   if truncateContents:
      contentSize = ( size - ETH_FCS_LEN ) - len( frame )
      content = content[ : contentSize ]
   frame += content
   padSize = ( size - ETH_FCS_LEN ) - len( frame )
   assert padSize >= 0
   frame += padChar * padSize

   return frame

def randomMac( multicast=False ):
   mac = [ 0x01 if multicast else random.getrandbits( 7 ) << 1,
           random.randint( 0x00, 0xff ),
           random.randint( 0x00, 0xff ),
           random.randint( 0x00, 0xff ),
           random.randint( 0x00, 0xff ),
           random.randint( 0x00, 0xff ) ]
   # pylint: disable-next=consider-using-f-string
   return ":".join( "%02x" % x for x in mac )
