# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=singleton-comparison
# pylint: disable=chained-comparison
# pylint: disable=consider-using-f-string

from enum import IntEnum
import re
import random
from socket import AF_INET, AF_INET6

import Arnet
from Arnet import Ip6Addr
from Arnet.EthTestLib import newEthPkt
from if_ether_arista import SfeReasonCode
from IpUtils import IpAddress, Mask
from SysConstants.if_arp_h import ARPOP_REQUEST, ARPOP_REPLY
from SysConstants.if_ether_h import ETH_P_IP, ETH_P_IPV6
from SysConstants.in_h import ( IPPROTO_ROUTING, IPPROTO_FRAGMENT,
                            IPPROTO_HOPOPTS, IPPROTO_AH,
                            IPPROTO_DSTOPTS, IPPROTO_NONE )
import Tac

IpAddrWithMask = Tac.Type( 'Arnet::IpAddrWithMask' )
Ip6AddrWithMask = Tac.Type( 'Arnet::Ip6AddrWithMask' )

# note: __defaultTraceHandle__ here is 'Arnet::IpTestLib'
import Tracing # pylint: disable=wrong-import-position
t8 = Tracing.trace8

ArpHdrSize = 28
MplsHdrSize = 4
MimHdrSize = 6
EthHdrSize = 14
EthDot1QHdrSize = 4
IpHdrSize = 20
Ip6HdrSize = 40
Ip6HdrOptionCommonSize = 2
Ip6HdrRouterAlertPad = 2
IpHdrRouterAlertOptionSize = 4
IpHdrRouterAlertOptionNum = 20
IpHdrUnrecognizedOptionSize = 8
IpHdrUnrecognizedOptionNum = 26
Ip6HdrHopByHopOptionNum = 0
IpMcastMacPrefix = "01:00:5E:"
IpMcastAllHostsMac = IpMcastMacPrefix + "00:00:01"
InitialIpChecksum = 0
ValidIpChecksum = 0
UdpHdrSize = 8
TcpHdrSize = 20
SctpHdrSize = 12
IcmpHdrSize = 8
defaultTtl = 1
IPV4 = 4
IPV6 = 6
MAC = 10
# Icmp6HdrSize covers only type, code and checksum
Icmp6HdrSize = 4
Icmp6RaBasicHeaderLen = 12
Icmp6RsBasicHeaderLen = 4
Icmp6NaBasicHeaderLen = 20
Icmp6NsBasicHeaderLen = 20
Icmp6SrcLinkLayerAddressOptionLen = 8
Icmp6TgtLinkLayerAddressOptionLen = 8
Icmp6MtuOptionLen = 8
Icmp6PrefixInfoOptionLen = 32
Icmp6DnsServerInfoOptionLen = 24
Icmp6Pref64OptionLen = 16
# Icmp6PacketTooBigHeaderLen covers only the MTU part
Icmp6PacketTooBigHeaderLen = 4
# GRE header only covers protoType, not optional fields
GreHdrSize = 4
PwAchHdrSize = 4
SfeHdrRxSize = 31

class IcmpType( IntEnum ):
   echoResp = 0
   dstUnreachable = 3
   sourceQuench = 4
   redirect = 5
   echoReq = 8
   timeExceeded = 11
   parameterProblem = 12
   timestampReq = 13
   timestampResp = 14
   infoReq = 15
   infoResp = 16
   addrMaskReq = 17
   addrMaskResp = 18

def isIp6Addr( address ):
   try:
      # If the address is an IPV6 link-local BGP peering address, it is represented
      # as <addr>%ifname (e.g. fe80::1%Et1) and so this must be special-cased.
      if '%' in str( address ):
         # pylint: disable-next=use-maxsplit-arg
         address = str( address ).split( '%' )[ 0 ]
      _ = Ip6Addr( address )
      return True
   except IndexError:
      return False

# verify whether the provided string is a valid representation of an IP address
def isIpAddr( s ):
   a = s.split('.')
   if len(a) != 4:
      return False
   for x in a:
      if not x.isdigit():
         return False
      i = int(x)
      if i < 0 or i > 255:
         return False
   return True

# Convert the given multicast ip address into its corresponding multicast
# mac address.
def ipMcastMacAddr( dstIpAddr ):
   dstIp = Arnet.IpGenAddr( dstIpAddr )
   return str( dstIp.mcastEtherAddress )

# Make up a source mac address based on the given source ip address.  We use
# the ip address for the low 4 bytes and a fixed prefix for the high 2 bytes.
def macFromIpAddr( srcIp ):
   strBytes = srcIp.split( '.' )
   if len( strBytes ) == 4:
      lowBytes = [ int( b ) for b in strBytes ]
   else:
      # We use the lowest order non-zero bytes we find in the IPv6 address.
      lowBytes = [ 0, 0, 0, 0 ]
      i = 3
      for s in reversed( srcIp.split( ':' ) ):
         if s:
            x = int( s, 16 )
            lowBytes[ i ] = ( x & 0xff )
            lowBytes[ i - 1 ] = ( ( x >> 8 ) & 0xff )
            i -= 2
         if i <= 0:
            break
   macStr = "00:00:%02X:%02X:%02X:%02X" % tuple( lowBytes )
   return macStr

# Return whether the given ip address is in the multicast range.
def isIpMcastAddr( ipAddr ):
   ip = Arnet.IpGenAddr( ipAddr )
   return ip.isMulticast

def isLinkLocalMcast( dstIp ):
   ip = Arnet.IpGenAddr( dstIp )
   return ip.isLinkLocal

def netmaskFromLen( length ):
   length = 32 - length
   mask = ( 0xFFFFFFFF >> length ) << length
   maskStr = "%d.%d.%d.%d" % ( ( mask >> 24 ) & 0xFF,
                               ( mask >> 16 ) & 0xFF,
                               ( mask >> 8 ) & 0xFF,
                               mask & 0xFF )
   return maskStr

# Given a MAC : deliminted string, make a string that represents
# an IP6 link local address
def ip6AddrFromMac( mac ):
   # Turn the mac into an int
   intMacAddr = int( mac.replace( ':', '' ), 16 )
   ip6LinkLocalAddr = 'fe80::%x:%xff:fe%.2x:%x' % (
                           ( ( intMacAddr >> 32 ) & 0xffff ) ^ 0x200,
                           ( intMacAddr >> 24 ) & 0xff,
                           ( intMacAddr >> 16 ) & 0xff,
                           intMacAddr & 0xffff )
   # have the tac version deal with all cases with leading zeros
   return Arnet.Ip6Addr( ip6LinkLocalAddr ).stringValue

# Given a IP6 link local address string, make a string that represents
# a MAC address. The procedure is the reverse of 'ip6AddrFromMac' written above
def macFromIp6Addr( ip6LinkLocalAddr ):
   # ip6addr = 'fe80:0000:0000:0000:0001:00ff:fe02:0000'
   # word0 = fe800000, word1 = 00000000, word2 = 000100ff, word3 = fe020000
   ipv6Addr = Ip6Addr( ip6LinkLocalAddr )
   ip6Word2 = ipv6Addr.word2
   ip6Word3 = ipv6Addr.word3
   macAddr = '%02X:%02X:%02X:%02X:%02X:%02X' % ( ( ( ip6Word2 >> 24 ) & 0xff ) ^ 0x2,
                                                 ( ip6Word2 >> 16 ) & 0xff,
                                                 ( ip6Word2 >> 8 ) & 0xff,
                                                 ( ip6Word3 >> 16 ) & 0xff,
                                                 ( ip6Word3 >> 8 ) & 0xff,
                                                 ( ip6Word3 & 0xff ) )

   return macAddr

def ipVersionOf( version, ipOrMask1, ipOrMask2=None ):
   """
   Figure out, if necessary, what version of IP is selected by the incoming
   addresses or masks. If the version is being discerned from the addresses,
   and there are two addresses, they must both be from the same IP version.
   """
   if version is None:
      assert isinstance( ipOrMask1, str )
      assert ipOrMask2 is None or isinstance( ipOrMask2, str )

      if ipOrMask2 is not None:
         assert ( ':' in ipOrMask1 ) is ( ':' in ipOrMask2 ), "IP version mismatch."
      return IPV6 if ':' in ipOrMask1 else IPV4
   return version

# Given a MAC : deliminted string, make a string that represents
# an IP6 solicited-node mcast Ip6 address
def ip6SolicitedNodeMcastAddrFromMac( mac ):
   # Turn the mac into an int and mask out
   intMacAddr = int( mac.replace( ':', '' ), 16 ) & 0xffffff
   addr = 'ff02::1:ff%.2x:%x' % ( intMacAddr >> 16,
                                  intMacAddr & 0xffff )
   return addr

# Given a MAC : deliminted string, make a string that represents
# an IP6 solicited-node mcast MAC address
def ip6SolicitedNodeMcastMacFromMac( mac ):
   mac = mac.lower()
   b = mac.split( ':' )
   res = '33:33:ff:%s:%s:%s' % ( b[ 3 ], b[ 4 ], b[ 5 ] )
   return res

# Given a target IP, determine the solicited-node multicast address
def ip6SolicitedNodeMcastAddrFromIp( targetIp ):
   ip6 = Arnet.Ip6Addr( targetIp )
   return ip6.solicitedNodeMulticastAddr.stringValue

# See RFC2464 Section 7
def macFromIp6Multicast( ipAddr ):
   ip6 = Arnet.Ip6Addr( ipAddr )
   assert ip6.isMulticast
   return ip6.mcastEtherAddress

def newEthArpPkt( srcMacAddr, targetIpAddr, dstMacAddr='ff:ff:ff:ff:ff:ff',
                  srcIpAddr='1.1.1.1', requestOrReply='request', vlanId=None,
                  vlanPriority=None,
                  senderSrcMac=None, targetDstMac=None,
                  outerVlanId=None,
                  outerVlanTpid='ethTypeQinQ',
                  vlanTpid='ethTypeDot1Q' ):
   # Currently it creates an ARP request packet for IPv4 protocol type.
   # It can be enhanced as and when we need to make it more flexible
   # Syed: Added Vlan priority optional field for ptest in
   # IpEth/ArpWithPriorityTagForSviTest.py.
   ( p, ethHdr, _, currentOffset ) = newEthPkt( srcMacAddr=srcMacAddr,
                                                dstMacAddr=dstMacAddr,
                                                vlanId=vlanId,
                                                vlanPriority=vlanPriority,
                                                ethType="ethTypeArpIp",
                                                extraSize=ArpHdrSize,
                                                outerVlanId=outerVlanId,
                                                outerVlanTpid=outerVlanTpid,
                                                vlanTpid=vlanTpid )
   arpHdr = Tac.newInstance( "Arnet::EthArpIpHdrWrapper", p, currentOffset )
   currentOffset += ArpHdrSize
   arpHdr.hardwareAddrSpace = 1 # for Ethernet
   arpHdr.protocolAddrSpace = ETH_P_IP
   arpHdr.hardwareAddrLen = 6
   arpHdr.protocolAddrLen = 4
   if requestOrReply == 'request':
      arpHdr.opcode = ARPOP_REQUEST
   elif requestOrReply == 'reply':
      arpHdr.opcode = ARPOP_REPLY
   else:
      assert 0, "Unexpected requestOrReply argument"
   if senderSrcMac == None:
      arpHdr.senderEthAddr = srcMacAddr
   else:
      arpHdr.senderEthAddr = senderSrcMac
   arpHdr.senderIpAddr = srcIpAddr
   if targetDstMac == None:
      arpHdr.targetEthAddr = dstMacAddr
   else:
      arpHdr.targetEthAddr = targetDstMac
   arpHdr.targetIpAddr = targetIpAddr

   return (p, ethHdr, arpHdr, currentOffset)

# snd/tgt MacAddr and IpAddr are the MAC and IP addresses of the ARP packet.
# dstMacAddr and srcMacAddr are the MAC addresses of the Ethernet header.
# By default, the Ethernet header values will be the same as the ARP packet values.
def newEthArpRespPkt( sndMacAddr, sndIpAddr, tgtMacAddr, tgtIpAddr,
                      dstMacAddr=None, srcMacAddr=None ):
   if dstMacAddr is None:
      dstMacAddr = tgtMacAddr
   if srcMacAddr is None:
      srcMacAddr = sndMacAddr

   # Currently it creates an ARP request packet for IPv4 protocol type.
   # It can be enhanced as and when we need to make it more flexible
   ( p, ethHdr, _, currentOffset ) = newEthPkt( srcMacAddr=srcMacAddr,
                                                dstMacAddr=dstMacAddr,
                                                ethType="ethTypeArpIp",
                                                extraSize=ArpHdrSize )
   arpHdr = Tac.newInstance( "Arnet::EthArpIpHdrWrapper", p, currentOffset )
   currentOffset += ArpHdrSize
   arpHdr.hardwareAddrSpace = 1 # for Ethernet
   arpHdr.protocolAddrSpace = ETH_P_IP
   arpHdr.hardwareAddrLen = 6
   arpHdr.protocolAddrLen = 4
   arpHdr.opcode = ARPOP_REPLY
   arpHdr.senderEthAddr = sndMacAddr
   arpHdr.senderIpAddr = sndIpAddr
   arpHdr.targetEthAddr = tgtMacAddr
   arpHdr.targetIpAddr = tgtIpAddr

   return (p, ethHdr, arpHdr, currentOffset)

def newMimPkt(srcMacAddrPayload,
              dstMacAddrPayload,
              srcMacAddrTunnel,
              dstMacAddrTunnel,
              isid,
              ttl=None,
              vlanId=None,
              vlanPriority=None,
              cfiBit=False,
              ethTypeBackbone="ethTypeDot1ad",
              extraSize=0 ):
   '''Create a packet with ethernet and ip headers using the given
   values.  Return a tuple of the packet, the ethernet header wrapper,
   the dot1Q header wrapper, the ip header wrapper, the router alert
   option wrapper, and the current offset into the packet where the
   next header should go.
   '''
   ttl = 0xFE
   t8( "Creating MIM packet with ttl : ", ttl )
   size = MimHdrSize + extraSize + EthHdrSize + EthDot1QHdrSize
   ( p, ethHdrBackbone, qHdrBackbone,
     currentOffset ) = newEthPkt( srcMacAddr=srcMacAddrTunnel,
                                  dstMacAddr=dstMacAddrTunnel,
                                  ethType=ethTypeBackbone,
                                  cfiBit=cfiBit,
                                  extraSize=size )

   mimHdr = Tac.newInstance( "Arnet::MimHdrWrapper", p, currentOffset )
   currentOffset += MimHdrSize
   mimHdr.serviceIntfEthType = 0x88E7
   mimHdr.isid = isid

   (p, ethHdr, qHdr, currentOffset) = newEthPkt( srcMacAddr=srcMacAddrPayload,
                                                 dstMacAddr=dstMacAddrPayload,
                                                 vlanId=vlanId,
                                                 vlanPriority=vlanPriority,
                                                 currentOffset=currentOffset,
                                                 cfiBit=cfiBit,
                                                 p=p,
                                                 extraSize=0 )
   return (p, ethHdrBackbone, qHdrBackbone, mimHdr, ethHdr, qHdr, currentOffset)


def newMplsPkt(srcMacAddr,
               dstMacAddr,
               labels=None,
               srcIpAddr=None,
               dstIpAddr=None,
               srcMacAddrPayload=None,
               dstMacAddrPayload=None,
               version=None,
               headerLen=None,
               tos=0,
               flowLabel=0,
               totalLen=None,
               id=0,  # pylint: disable-msg=W0622
               fragmentOffset=0,
               moreFrag=False,
               dontFrag=False,
               exp=None,
               ttl=None,
               ipttl=0xFE,
               protocol="ipProtoUdp",
               checksum=None,
               routerAlert=False,
               vlanId=None,
               vlanTpid="ethTypeDot1Q",
               vlanPriority=None,
               cfiBit=False,
               extraSize=0,
               mplsPayloadVlan=None,
               mplsPayloadVlanPri=None,
               ethMplsEth=False,
               pwAch=False,
               **kwargs ):
   '''Create a packet with ethernet, mpls and ip headers using the given
   values.  Return a tuple of the packet, the ethernet header wrapper,
   the dot1Q header wrapper, mpls header wrapper, the ip header wrapper,
   the router alert option wrapper, and the current offset into the
   packet where the next header should go.
   '''
   if version is None and srcIpAddr is None and dstIpAddr is None:
      version = IPV4
   else:
      version = ipVersionOf( version, srcIpAddr, dstIpAddr )
   numMplsLabel = len(labels)
   if not exp:
      exp = [0, 0, 0]
   if not ttl:
      ttl = [0xFE, 0xFE, 0xFE]

   t8( "Creating MPLS packet with ttl : ", ttl )
   # ethMplsEth implies an ethernet over Mpls over Ethernet packet.
   # This is used for non-Ip Mpls packet creation.
   if ethMplsEth == True:
      size = MplsHdrSize * numMplsLabel + extraSize + EthHdrSize
   else:
      if version == 4:
         size = MplsHdrSize * numMplsLabel + extraSize + IpHdrSize
         if routerAlert:
            size += IpHdrRouterAlertOptionSize
      else:
         size = MplsHdrSize * numMplsLabel + extraSize + Ip6HdrSize
         if routerAlert:
            size += Ip6HdrRouterAlertPad + Ip6HdrOptionCommonSize

   if pwAch:
      size += PwAchHdrSize

   (p, ethHdr1, qHdr1, currentOffset) = newEthPkt( srcMacAddr=srcMacAddr,
                                                   dstMacAddr=dstMacAddr,
                                                   vlanId=vlanId,
                                                   vlanTpid=vlanTpid,
                                                   vlanPriority=vlanPriority,
                                                   cfiBit=cfiBit,
                                                   ethType="ethTypeMpls",
                                                   extraSize=size,
                                                   **kwargs )

   for i in range( numMplsLabel ):
      mplsHdr = Tac.newInstance( "Arnet::MplsHdrWrapper", p, currentOffset )
      currentOffset += MplsHdrSize
      mplsHdr.label = labels[i]
      mplsHdr.cos = exp[i]
      mplsHdr.bos = False
      mplsHdr.ttl = ttl[i]
   mplsHdr.bos = True

   if pwAch:
      pwAchHdr = Tac.newInstance( "Arnet::PwAchHdrWrapper", p, currentOffset )
      currentOffset += PwAchHdrSize
      if version == 6:
         pwAchHdr.lspPingAch = Tac.Type( 'Arnet::LspPingAch' ).cwIpv6
      else:
         pwAchHdr.lspPingAch = Tac.Type( 'Arnet::LspPingAch' ).cwIpv4

   if( srcIpAddr == None and dstIpAddr == None ):
      assert srcMacAddrPayload
      assert dstMacAddrPayload
      # In case of ethMplsEth scenario, the inner ethernet payload may be tagged.
      if mplsPayloadVlan != None:
         vlanId = mplsPayloadVlan
      if mplsPayloadVlanPri != None:
         vlanPriority = mplsPayloadVlanPri
      (p, ethHdr, qHdr, currentOffset) = newEthPkt( srcMacAddr=srcMacAddrPayload,
                                                    dstMacAddr=dstMacAddrPayload,
                                                    vlanId=vlanId,
                                                    vlanPriority=vlanPriority,
                                                    cfiBit=cfiBit,
                                                    currentOffset=currentOffset,
                                                    p=p,
                                                    extraSize=0 )
      return (p, ethHdr1, qHdr1, mplsHdr, ethHdr, qHdr, currentOffset)
   else:
      assert srcIpAddr
      assert dstIpAddr
      # Ip header will derive Ip total length from extraSize
      # This must not account Mpls header
      size -= MplsHdrSize * numMplsLabel
      ( ipHdr, raOpt, currentOffset ) = createIpHdr (p=p,
                                                     currentOffset=currentOffset,
                                                     srcIpAddr=srcIpAddr,
                                                     dstIpAddr=dstIpAddr,
                                                     ttl=ipttl,
                                                     version=version,
                                                     headerLen=None,
                                                     tos=tos,
                                                     flowLabel=flowLabel,
                                                     totalLen=None,
                                                     id=0,
                                                     fragmentOffset=0,
                                                     moreFrag=False,
                                                     dontFrag=False,
                                                     protocol=protocol,
                                                     checksum=None,
                                                     routerAlert=False,
                                                     cfiBit=False,
                                                     extraSize=size )
      return (p, ethHdr1, qHdr1, mplsHdr, ipHdr, raOpt, currentOffset)

def newMplsUdpPkt( srcMacAddr,
                   dstMacAddr,
                   labels=None,
                   srcIpAddr=None,
                   dstIpAddr=None,
                   srcMacAddrPayload=None,
                   dstMacAddrPayload=None,
                   version=None,
                   headerLen=None,
                   tos=0,
                   flowLabel=0,
                   totalLen=None,
                   id=0,  # pylint: disable-msg=W0622
                   fragmentOffset=0,
                   moreFrag=False,
                   dontFrag=False,
                   exp=None,
                   ttl=None,
                   ipttl=0xFE,
                   checksum=None,
                   routerAlert=False,
                   vlanId=None,
                   vlanPriority=None,
                   vlanTpid="ethTypeDot1Q",
                   cfiBit=False,
                   srcPort=1999,
                   dstPort=1999,
                   data='',
                   udpLen=None,
                   **kwargs ):
   '''Create a packet with ethernet, mpls, ip, and udp headers using the given
   values, Return a tuple of test packet, the ethernet header wrapper, the
   dot1Q header wrapper, mpls header wrapper, the ip header wrapper, the
   router alert option wrapper, the udp header wrapper, and the current offset
   into the packet where the next header should go. '''
   version = ipVersionOf( version, srcIpAddr, dstIpAddr )
   if udpLen is None:
      udpLen = UdpHdrSize + len( data )
   ( p, ethHdr1, qHdr1, mplsHdr, ipHdr, raOpt, currentOffset ) =\
         newMplsPkt( labels=labels,
                     srcMacAddr=srcMacAddr,
                     dstMacAddr=dstMacAddr,
                     srcIpAddr=srcIpAddr,
                     dstIpAddr=dstIpAddr,
                     srcMacAddrPayload=dstMacAddrPayload,
                     version=version,
                     headerLen=headerLen,
                     tos=tos,
                     flowLabel=flowLabel,
                     totalLen=totalLen,
                     id=id,
                     fragmentOffset=fragmentOffset,
                     moreFrag=moreFrag,
                     dontFrag=dontFrag,
                     exp=exp,
                     ttl=ttl,
                     ipttl=ipttl,
                     protocol="ipProtoUdp",
                     checksum=checksum,
                     routerAlert=routerAlert,
                     vlanId=vlanId,
                     vlanPriority=vlanPriority,
                     vlanTpid=vlanTpid,
                     cfiBit=cfiBit,
                     extraSize=udpLen,
                     **kwargs )

   ( udpHdr, currentOffset ) = \
      createUdpHdr( p=p,
                    srcIpAddr=srcIpAddr,
                    dstIpAddr=dstIpAddr,
                    srcPort=srcPort,
                    dstPort=dstPort,
                    data=data,
                    udpLen=udpLen,
                    currentOffset=currentOffset,
                    version=version)

   return ( p, ethHdr1, qHdr1, mplsHdr, ipHdr, raOpt, udpHdr, currentOffset )

def newMplsTcpPkt( srcMacAddr,
                   dstMacAddr,
                   labels=None,
                   srcIpAddr=None,
                   dstIpAddr=None,
                   srcMacAddrPayload=None,
                   dstMacAddrPayload=None,
                   version=None,
                   headerLen=None,
                   tos=0,
                   flowLabel=0,
                   totalLen=None,
                   id=0,  # pylint: disable-msg=W0622
                   fragmentOffset=0,
                   moreFrag=False,
                   dontFrag=False,
                   exp=None,
                   ttl=None,
                   ipttl=0xFE,
                   checksum=None,
                   routerAlert=False,
                   vlanId=None,
                   vlanPriority=None,
                   cfiBit=False,
                   srcPort=1999,
                   dstPort=1999,
                   sequenceNumber=0,
                   acknowledgementNumber=0,
                   dataOffset=None,
                   fin=False,
                   syn=False,
                   rst=False,
                   psh=False,
                   ack=False,
                   urg=False,
                   ece=False,
                   cwr=False,
                   ns=False,
                   window=0,
                   urgentPointer=0,
                   data='',
                   tcpLen=None,
                   **kwargs ):
   '''Create a packet with ethernet, mpls, ip, and udp headers using the given
   values, Return a tuple of test packet, the ethernet header wrapper, the
   dot1Q header wrapper, mpls header wrapper, the ip header wrapper, the
   router alert option wrapper, the tcp header wrapper, and the current offset
   into the packet where the next header should go. '''
   protocol = "ipProtoTcp"
   version = ipVersionOf( version, srcIpAddr, dstIpAddr )
   if tcpLen is None:
      tcpLen = TcpHdrSize + len( data )
   ( p, ethHdr1, qHdr1, mplsHdr, ipHdr, raOpt, currentOffset ) =\
         newMplsPkt( labels=labels,
                     srcMacAddr=srcMacAddr,
                     dstMacAddr=dstMacAddr,
                     srcIpAddr=srcIpAddr,
                     dstIpAddr=dstIpAddr,
                     srcMacAddrPayload=dstMacAddrPayload,
                     version=version,
                     headerLen=headerLen,
                     tos=tos,
                     flowLabel=flowLabel,
                     totalLen=totalLen,
                     id=id,
                     fragmentOffset=fragmentOffset,
                     moreFrag=moreFrag,
                     dontFrag=dontFrag,
                     exp=exp,
                     ttl=ttl,
                     ipttl=ipttl,
                     protocol=protocol,
                     checksum=checksum,
                     routerAlert=routerAlert,
                     vlanId=vlanId,
                     vlanPriority=vlanPriority,
                     cfiBit=cfiBit,
                     extraSize=tcpLen,
                     **kwargs )

   ( tcpHdr, currentOffset ) = \
      createTcpHdr( p=p,
                    srcIpAddr=srcIpAddr,
                    dstIpAddr=dstIpAddr,
                    srcPort=srcPort,
                    dstPort=dstPort,
                    sequenceNumber=sequenceNumber,
                    acknowledgementNumber=acknowledgementNumber,
                    dataOffset=dataOffset,
                    fin=fin,
                    syn=syn,
                    rst=rst,
                    psh=psh,
                    ack=ack,
                    urg=urg,
                    ece=ece,
                    cwr=cwr,
                    ns=ns,
                    window=window,
                    urgentPointer=urgentPointer,
                    data=data,
                    currentOffset=currentOffset,
                    version=version )

   return ( p, ethHdr1, qHdr1, mplsHdr, ipHdr, raOpt, tcpHdr, currentOffset )

def createIpHdr (p,
                 currentOffset,
                 srcIpAddr,
                 dstIpAddr,
                 ttl,
                 version=None,
                 headerLen=None,
                 tos=0,
                 flowLabel=0,
                 totalLen=None,
                 id=0,  # pylint: disable-msg=W0622
                 fragmentOffset=0,
                 moreFrag=False,
                 dontFrag=False,
                 protocol="ipProtoUdp",
                 checksum=None,
                 routerAlert=False,
                 cfiBit=False,
                 extraSize=0 ):

   size = extraSize

   version = ipVersionOf( version, srcIpAddr, dstIpAddr )
   if version == 4:
      ipHdr = Tac.newInstance( "Arnet::IpHdrWrapper", p, currentOffset )
      currentOffset += IpHdrSize
   else:
      ipHdr = Tac.newInstance( "Arnet::Ip6HdrWrapper", p, currentOffset )
      currentOffset += Ip6HdrSize

   ipHdr.version = version

   t8( 'createIpHdr: creating ip header with ttl: ', ttl )

   if version == 6:
      ipHdr.trafficClass = tos
      ipHdr.flowLabel = flowLabel
      ipHdr.payloadLen = size - Ip6HdrSize
      if isinstance( protocol, int ):
         ipHdr.nextHeader = protocol
      else:
         assert type( protocol ) == str # pylint: disable=unidiomatic-typecheck
         ipHdr.protocolNum = protocol
      ipHdr.hopLimit = ttl
      ipHdr.src = Arnet.Ip6Addr( srcIpAddr )
      ipHdr.dst = Arnet.Ip6Addr( dstIpAddr )
   else:
      if headerLen is None:
         headerLen = 5
         if routerAlert:
            headerLen += 1
      ipHdr.headerLen = headerLen

      ipHdr.tos = tos

      if totalLen is None:
         totalLen = size
      ipHdr.totalLen = totalLen
      ipHdr.id = id
      ipHdr.fragmentOffset = fragmentOffset
      ipHdr.moreFrag = moreFrag
      ipHdr.dontFrag = dontFrag
      ipHdr.reservedFragFlag = False
      ipHdr.ttl = ttl
      if isinstance( protocol, int ):
         ipHdr.protocol = protocol
      else:
         assert type( protocol ) == str # pylint: disable=unidiomatic-typecheck
         ipHdr.protocolNum = protocol
      ipHdr.src = srcIpAddr
      ipHdr.dst = dstIpAddr

   raOpt = None
   if routerAlert:
      if version == 4:
         raOpt = Tac.newInstance( "Arnet::IpHdrRouterAlertOptionWrapper",
                                  p, currentOffset )
         currentOffset += IpHdrRouterAlertOptionSize
         _unused = raOpt.raHeaderInstantiate
      else:
         raOpt = Tac.newInstance( "Arnet::Ip6HdrRouterAlertOptionWrapper",
                                  p, currentOffset )
         currentOffset += IpHdrRouterAlertOptionSize + Ip6HdrOptionCommonSize
         _unused = raOpt.raHeaderInstantiate
         raOpt.nextHeader = ipHdr.nextHeader
         raOpt.optionLength = 0
         ipHdr.nextHeader = Ip6HdrHopByHopOptionNum # Hop-by-Hop EH type

         _pad = Tac.newInstance( "Arnet::Ip6HdrPadNWrapper", p, currentOffset )
         currentOffset += Ip6HdrRouterAlertPad

   if version != 6:
      if checksum is None:
         ipHdr.checksum = InitialIpChecksum
         checksum = ipHdr.computedChecksum

      ipHdr.checksum = checksum

   return ( ipHdr, raOpt, currentOffset)

def newIpPkt( srcMacAddr,
              dstMacAddr,
              srcIpAddr,
              dstIpAddr,
              version=None,
              headerLen=None,
              tos=0,
              flowLabel=0,
              totalLen=None,
              id=0, # pylint: disable-msg=W0622
              fragmentOffset=0,
              moreFrag=False,
              dontFrag=False,
              ttl=None,
              protocol="ipProtoUdp",
              checksum=None,
              routerAlert=False,
              UnrecognizedIpOptions=False,
              vlanId=None,
              vlanPriority=None,
              vlanTpid='ethTypeDot1Q',
              cfiBit=False,
              ethType="ethTypeIp",
              extraSize=0,
              baseLevel='',
              outerVlanId=None,
              outerVlanTpid="ethTypeQinQ",
              sfeHdrRx=False,
              **kwargs ):
   '''Create a packet with ethernet and ip headers using the given
   values.  Return a tuple of the packet, the ethernet header wrapper,
   the dot1Q header wrapper, the ip header wrapper, the router alert
   option wrapper, and the current offset into the packet where the
   next header should go.

   If 'extraSize' is not specified, the returned packet is sized to
   hold exactly the needed ethernet and ip headers.

   'baseLevel' is used to specify the lowest layer of the packet that
   you wish to build from. For example, if you specify baseLevel='ICMP6'
   the IP header and Eth headers will not be added and the packet will
   start with the ICMP6 header. This feature has only been added to
   a few packet creation functions so you may need to implement it for
   the functions you are using.

   If you wish to add the 'baseLevel' functionality to your packet creation
   functions please include the baseLevel optional argument and pass it down
   the packet creation stack insuring all functions in the creation path accept
   the argument as well. Create a custom string for your 'level' and handle the
   case your function is the base level, creating the packet at that point and
   setting the size (use this function as an example).
   '''
   version = ipVersionOf( version, srcIpAddr, dstIpAddr )

   # defaultTtl may be modified in IgmpSnoopingTestLib.runProductTests, so we need to
   # dynamically assign ttl=defaultTtl here rather than in the function signature
   if ttl is None:
      ttl = defaultTtl

   t8( 'newIpPkt: creating ip packet with ttl: ', ttl )

   if version == IPV6:
      size = Ip6HdrSize + extraSize
      if ethType == "ethTypeIp":
         ethType = "ethTypeIp6"
   else:
      size = IpHdrSize + extraSize

   if routerAlert:
      size += IpHdrRouterAlertOptionSize
      if version == IPV6:
         size += Ip6HdrOptionCommonSize + Ip6HdrRouterAlertPad
   if UnrecognizedIpOptions:
      size += IpHdrUnrecognizedOptionSize

   if baseLevel == 'IP':
      (p, ethHdr, qHdr, currentOffset) = \
            ( Tac.newInstance( "Arnet::Pkt" ), None, None, 0 )
      p.newSharedHeadData = size
   else:
      (p, ethHdr, qHdr, currentOffset) = newEthPkt( srcMacAddr=srcMacAddr,
                                                    dstMacAddr=dstMacAddr,
                                                    vlanId=vlanId,
                                                    vlanPriority=vlanPriority,
                                                    vlanTpid=vlanTpid,
                                                    cfiBit=cfiBit,
                                                    ethType=ethType,
                                                    extraSize=size,
                                                    outerVlanId=outerVlanId,
                                                    outerVlanTpid=outerVlanTpid,
                                                    **kwargs )

   if sfeHdrRx:
      sfeHdr = Tac.newInstance( "Arfa::SfeHdrRxWrapper", p, currentOffset )
      currentOffset += SfeHdrRxSize + 1
      sfeHdr.proto = ETH_P_IP
      if version == IPV6:
         sfeHdr.proto = ETH_P_IPV6
      sfeHdr.reasonCode = SfeReasonCode.SFE_REASON_RX_PEER_LINK_OR_CORE.value
      sfeHdr.peerlinkOrCore = kwargs.get( 'peerlinkOrCore', False )

   if version == IPV6:
      ipHdr = Tac.newInstance( "Arnet::Ip6HdrWrapper", p, currentOffset )
      currentOffset += Ip6HdrSize
   else:
      ipHdr = Tac.newInstance( "Arnet::IpHdrWrapper", p, currentOffset )
      currentOffset += IpHdrSize

   ipHdr.version = version

   if version == IPV6:
      ipHdr.trafficClass = tos
      ipHdr.flowLabel = flowLabel
      ipHdr.payloadLen = size - Ip6HdrSize
      if isinstance( protocol, int ):
         ipHdr.nextHeader = protocol
      else:
         assert type( protocol ) == str # pylint: disable=unidiomatic-typecheck
         ipHdr.protocolNum = protocol
      ipHdr.hopLimit = ttl
      ipHdr.src = Arnet.Ip6Addr( srcIpAddr )
      ipHdr.dst = Arnet.Ip6Addr( dstIpAddr )

   else:

      if headerLen is None:
         headerLen = 5
         if routerAlert:
            headerLen += 1
         else:
            if UnrecognizedIpOptions:
               headerLen += 2
      ipHdr.headerLen = headerLen

      ipHdr.tos = tos

      if totalLen is None:
         totalLen = size
      ipHdr.totalLen = totalLen
      ipHdr.id = id
      ipHdr.fragmentOffset = fragmentOffset
      ipHdr.moreFrag = moreFrag
      ipHdr.dontFrag = dontFrag
      ipHdr.reservedFragFlag = False
      ipHdr.ttl = ttl
      if isinstance( protocol, int ):
         ipHdr.protocol = protocol
      else:
         assert type( protocol ) == str # pylint: disable=unidiomatic-typecheck
         ipHdr.protocolNum = protocol
      ipHdr.src = srcIpAddr
      ipHdr.dst = dstIpAddr

   raOpt = None

   if routerAlert:
      if version == IPV6:
         raOpt = Tac.newInstance( "Arnet::Ip6HdrRouterAlertOptionWrapper",
                                  p, currentOffset )
         currentOffset += IpHdrRouterAlertOptionSize + Ip6HdrOptionCommonSize
         _unused = raOpt.raHeaderInstantiate
         raOpt.nextHeader = ipHdr.nextHeader
         raOpt.optionLength = 0
         ipHdr.nextHeader = Ip6HdrHopByHopOptionNum # Hop-by-Hop EH type

         # Note that this instance is unused within python, but we
         # must bind it to a local variable or it gets garbage
         # collected
         _pad = Tac.newInstance( "Arnet::Ip6HdrPadNWrapper", p, currentOffset )

         currentOffset += Ip6HdrRouterAlertPad
      else:
         raOpt = Tac.newInstance( "Arnet::IpHdrRouterAlertOptionWrapper",
                                  p, currentOffset )
         currentOffset += IpHdrRouterAlertOptionSize
         _unused = raOpt.raHeaderInstantiate
   else:
      if UnrecognizedIpOptions:
         if version == IPV4:
            raOpt = Tac.newInstance( "Arnet::IpHdrUnrecognizedOptionWrapper",
                                     p, currentOffset )
            currentOffset += IpHdrUnrecognizedOptionSize
            _unused = raOpt.uoHeaderInstantiate
         else:
            assert False, "UnrecognizedIpOptions not supported for IPv6"

   if version != IPV6:
      if checksum is None:
         ipHdr.checksum = InitialIpChecksum
         checksum = ipHdr.computedChecksum
      ipHdr.checksum = checksum

   if sfeHdrRx:
      assert not raOpt # dont support raOpt for this case to avoid W0632
      return ( p, ethHdr, qHdr, sfeHdr, ipHdr, currentOffset )

   return (p, ethHdr, qHdr, ipHdr, raOpt, currentOffset)

def clearPrefixHostAddr( prefix ):
   if '/' not in str( prefix ):
      return prefix
   af = AF_INET if '.' in str( prefix ) else AF_INET6
   addr, pLen = prefix.split( '/' )
   addrNum = IpAddress( addr, af ).toNum()
   addrMask = Mask( int( pLen ), addrFamily=af ).toNum()
   newAddr = IpAddress( addrNum & addrMask, af )
   return ( '%s/%s' % ( str( newAddr ), pLen ), addrNum & ~addrMask )

def newAddrForIndex( prefixOrAddr, index ):
   isPrefix = '/' in str( prefixOrAddr )
   if isPrefix:
      m = re.match( r'(\S+)\/(32|128)', prefixOrAddr ) # host route?
      if m:
         return m.group( 1 )
   # We can't directly use Arnet.IpGenPrefix because the ip addr
   # passed to interface/vlan is in the form of x.x.x.1/x,
   # which is an invalid prefix ( lastField should be 0 ).
   prefixOrAddr, hostAddrNum = clearPrefixHostAddr( prefixOrAddr )
   addr = Arnet.IpGenPrefix( str( prefixOrAddr ) )
   if addr.af == 'ipv4':
      af = AF_INET
   else:
      af = AF_INET6
   addrNum = IpAddress( addr.ipGenAddr, af ).toNum()
   newAddr = IpAddress( addrNum + + hostAddrNum + index, af )
   if isPrefix:
      assert addr.overlaps( Arnet.IpGenPrefix( str( newAddr ) ) )
   return str( newAddr )

def getBroadCastAddrFromIpAddr( ipAddress ):
   assert '/' in str( ipAddress )
   af = AF_INET if '.' in str( ipAddress ) else AF_INET6
   addr, pLen = ipAddress.split( '/' )
   addrNum = IpAddress( addr, af ).toNum()
   addrMask = Mask( int( pLen ), addrFamily=af )
   # broadCastAddr = ipAddr | ~mask
   addrMask = IpAddress( addrMask.inverseStr(), af ).toNum()
   broadCastAddr = IpAddress( addrNum | addrMask, af )
   return str( broadCastAddr )

def getDiffIpAddrWithSameSubnet( ipAddress, prefixLen, exceptIpAddrList=None,
                                 timeout=2 ):
   if not exceptIpAddrList:
      exceptIpAddrList = []
   assert isinstance( exceptIpAddrList, list )
   exceptIpAddrList.append( ipAddress )

   if isIpAddr( ipAddress ):
      addressSize = 32
   elif isIp6Addr( ipAddress ):
      addressSize = 128
   else:
      assert False, "Invalid Ip address"

   assert 0 < prefixLen < addressSize, "prefixLen must be positive and less " \
                                       "than %d bits" % addressSize

   ( prefix, _ ) = clearPrefixHostAddr( "%s/%d" % ( ipAddress, prefixLen ) )

   for ip in exceptIpAddrList:
      ( ipPrefix, _ )  = clearPrefixHostAddr( "%s/%d" % ( ip, prefixLen ) )
      if ipPrefix != prefix:
         exceptIpAddrList.remove( ip )
   exceptIpAddrSet = set( exceptIpAddrList )

   assert len( exceptIpAddrSet ) < 2 ** ( addressSize - prefixLen ), \
          "exceptIpAddrList covers all of the subnet address space"

   # in case this function is called with a ralatively small prefixLen and a large
   # exceptIpAddrList covering most of the subnet address space, this function will
   # raise exception after execution time exceeding timeout if a new random ipaddress
   # distinct from those inside the exceptIpAddrSet cannot be generated
   startTime = Tac.now()
   newIpAddr = newAddrForIndex( prefix,
                                random.getrandbits( addressSize - prefixLen ) )
   while newIpAddr in exceptIpAddrSet:
      newIpAddr = newAddrForIndex( prefix,
                                   random.getrandbits( addressSize - prefixLen ) )
      assert Tac.now() - startTime < timeout, "Unable to find new ip address with " \
                                              "the subnet within reasonable run-time"

   return newIpAddr

def createSctpHdr( p,
                   srcIpAddr,
                   dstIpAddr,
                   srcPort,
                   dstPort,
                   data,
                   verificationTag,
                   currentOffset,
                   version ):

   sctpHdr = Tac.newInstance( "Arnet::SctpHdrWrapper",
                              p, currentOffset )
   currentOffset += SctpHdrSize

   sctpHdr.srcPort = srcPort
   sctpHdr.dstPort = dstPort
   sctpHdr.verificationTag = verificationTag
   sctpHdr.sctpChecksum = InitialIpChecksum

   if isinstance( data, str ):
      data = data.encode()
   p.rawBytesAtIdxIs( currentOffset, data )

   return ( sctpHdr, currentOffset )

def newSctpPkt( srcMacAddr=None,
                dstMacAddr=None,
                vlanId=None,
                vlanPriority=None,
                cfiBit=False,
                ethType="ethTypeIp",
                srcIpAddr="10.1.1.1",
                dstIpAddr="10.2.2.2",
                version=None,
                tos=0,
                flowLabel=0,
                ipTotalLen=None,
                ipId=0,
                fragmentOffset=0,
                moreFrag=False,
                dontFrag=False,
                ttl=None,
                protocol="ipProtoSctp",
                ipChecksum=None,
                routerAlert=False,
                srcPort=1999,
                dstPort=1999,
                data='',
                sctpLen=None,
                verificationTag=1000,
                baseLevel='',
                **kwargs ):
   version = ipVersionOf( version, srcIpAddr, dstIpAddr )

   if baseLevel != 'IP':
      if dstMacAddr is None:
         if isIpMcastAddr( dstIpAddr ):
            dstMacAddr = ipMcastMacAddr( dstIpAddr )
         else:
            dstMacAddr = macFromIpAddr( dstIpAddr )

      if srcMacAddr is None:
         srcMacAddr = macFromIpAddr( srcIpAddr )

   if sctpLen is None:
      sctpLen = SctpHdrSize + len( data )

   ( p, ethHdr, qHdr, ipHdr, raOpt, currentOffset ) = \
       newIpPkt( srcMacAddr=srcMacAddr,
                 dstMacAddr=dstMacAddr,
                 srcIpAddr=srcIpAddr,
                 dstIpAddr=dstIpAddr,
                 version=version,
                 totalLen=ipTotalLen,
                 id=ipId,
                 fragmentOffset=fragmentOffset,
                 moreFrag=moreFrag,
                 dontFrag=dontFrag,
                 ttl=ttl,
                 tos=tos,
                 flowLabel=flowLabel,
                 protocol=protocol,
                 checksum=ipChecksum,
                 routerAlert=routerAlert,
                 vlanId=vlanId,
                 vlanPriority=vlanPriority,
                 cfiBit=cfiBit,
                 ethType=ethType,
                 extraSize=sctpLen,
                 baseLevel=baseLevel,
                 **kwargs )

   ( sctpHdr, currentOffset ) = \
        createSctpHdr( p=p,
                       srcIpAddr=srcIpAddr,
                       dstIpAddr=dstIpAddr,
                       srcPort=srcPort,
                       dstPort=dstPort,
                       data=data,
                       verificationTag=verificationTag,
                       currentOffset=currentOffset,
                       version=version )

   return ( p, ethHdr, qHdr, ipHdr, raOpt, sctpHdr, currentOffset )

def createUdpHdr( p,
                  srcIpAddr,
                  dstIpAddr,
                  srcPort,
                  dstPort,
                  data,
                  udpLen,
                  currentOffset,
                  version=None ):

   version = ipVersionOf( version, srcIpAddr, dstIpAddr )
   if version == IPV6:
      udpHdr = Tac.newInstance( "Arnet::UdpHdrWrapperIpv6",
                             p, currentOffset )
   else:
      udpHdr = Tac.newInstance( "Arnet::UdpHdrWrapper",
                             p, currentOffset )
   currentOffset += UdpHdrSize

   udpHdr.srcPort = srcPort
   udpHdr.dstPort = dstPort
   udpHdr.len = udpLen
   udpHdr.checksum = InitialIpChecksum

   if isinstance( data, str ):
      data = data.encode()
   p.rawBytesAtIdxIs( currentOffset, data )

   # Now that we've copied in the user data, update the udp checksum.
   if version == IPV6:
      udpHdr.ipSrc = Arnet.Ip6Addr( srcIpAddr )
      udpHdr.ipDst = Arnet.Ip6Addr( dstIpAddr )
   else:
      udpHdr.ipSrc = srcIpAddr
      udpHdr.ipDst = dstIpAddr

   udpChecksum = udpHdr.computedChecksum
   udpHdr.checksum = udpChecksum

   return ( udpHdr, currentOffset )

def newUdpPkt( srcMacAddr=None,
               dstMacAddr=None,
               vlanId=None,
               vlanPriority=None,
               vlanTpid='ethTypeDot1Q',
               cfiBit=False,
               ethType="ethTypeIp",
               srcIpAddr="10.1.1.1",
               dstIpAddr="10.2.2.2",
               version=None,
               tos=0,
               flowLabel=0,
               ipTotalLen=None,
               ipId=0,
               fragmentOffset=0,
               moreFrag=False,
               dontFrag=False,
               ttl=None,
               protocol="ipProtoUdp",
               ipChecksum=None,
               routerAlert=False,
               UnrecognizedIpOptions=False,
               srcPort=1999,
               dstPort=1999,
               data='',
               udpLen=None,
               baseLevel='',
               outerVlanId=None,
               outerVlanTpid="ethTypeQinQ",
               **kwargs ):
   '''Create a packet with ethernet, ip, and udp headers using the
   given values.  Return a tuple of the packet, the ethernet header
   wrapper, the dot1Q header wrapper, the ip header wrapper, the
   router alert option wrapper, the udp header wrapper, and the
   current offset into the packet where the next header should go (if
   you wanted one).

   'baseLevel' is used to specify the lowest layer of the packet that
   you wish to build from. For example, if you specify baseLevel='ICMP6'
   the IP header and Eth headers will not be added and the packet will
   start with the ICMP6 header. This feature has only been added to
   a few packet creation functions so you may need to implement it for
   the functions you are using.
   '''

   version = ipVersionOf( version, srcIpAddr, dstIpAddr )
   if baseLevel != 'IP':
      if dstMacAddr is None:
         if isIpMcastAddr( dstIpAddr ):
            dstMacAddr = ipMcastMacAddr( dstIpAddr )
         else:
            dstMacAddr = macFromIpAddr( dstIpAddr )

      if srcMacAddr is None:
         srcMacAddr = macFromIpAddr( srcIpAddr )

   if udpLen is None:
      udpLen = UdpHdrSize + len( data )

   (p, ethHdr, qHdr, ipHdr, raOpt, currentOffset) = \
       newIpPkt( srcMacAddr=srcMacAddr,
                 dstMacAddr=dstMacAddr,
                 srcIpAddr=srcIpAddr,
                 dstIpAddr=dstIpAddr,
                 version=version,
                 totalLen=ipTotalLen,
                 id=ipId,
                 fragmentOffset=fragmentOffset,
                 moreFrag=moreFrag,
                 dontFrag=dontFrag,
                 ttl=ttl,
                 tos=tos,
                 flowLabel=flowLabel,
                 protocol=protocol,
                 checksum=ipChecksum,
                 routerAlert=routerAlert,
                 UnrecognizedIpOptions=UnrecognizedIpOptions,
                 vlanId=vlanId,
                 vlanPriority=vlanPriority,
                 vlanTpid=vlanTpid,
                 cfiBit=cfiBit,
                 ethType=ethType,
                 extraSize=udpLen,
                 baseLevel=baseLevel,
                 outerVlanId=outerVlanId,
                 outerVlanTpid=outerVlanTpid,
                 **kwargs )

   ( udpHdr, currentOffset ) = \
         createUdpHdr( p=p,
                       srcIpAddr=srcIpAddr,
                       dstIpAddr=dstIpAddr,
                       srcPort=srcPort,
                       dstPort=dstPort,
                       data=data,
                       udpLen=udpLen,
                       currentOffset=currentOffset,
                       version=version)

   return (p, ethHdr, qHdr, ipHdr, raOpt, udpHdr, currentOffset)

def createTcpHdr( p,
                  srcIpAddr,
                  dstIpAddr,
                  srcPort,
                  dstPort,
                  sequenceNumber,
                  acknowledgementNumber,
                  dataOffset,
                  fin,
                  syn,
                  rst,
                  psh,
                  ack,
                  urg,
                  ece,
                  cwr,
                  ns,
                  window,
                  urgentPointer,
                  data,
                  currentOffset,
                  version ):

   if version == IPV6:
      tcpHdr = Tac.newInstance( "Arnet::TcpHdrWrapperIpv6",
                             p, currentOffset )
   else:
      tcpHdr = Tac.newInstance( "Arnet::TcpHdrWrapper",
                             p, currentOffset )
   currentOffset += TcpHdrSize

   tcpHdr.srcPort = srcPort
   tcpHdr.dstPort = dstPort
   tcpHdr.sequenceNumber = sequenceNumber
   tcpHdr.acknowledgementNumber = acknowledgementNumber
   if dataOffset is None:
      dataOffset = 5
   tcpHdr.dataOffset = dataOffset
   tcpHdr.reserved = 0
   tcpHdr.fin = fin
   tcpHdr.syn = syn
   tcpHdr.rst = rst
   tcpHdr.psh = psh
   tcpHdr.ack = ack
   tcpHdr.urg = urg
   tcpHdr.ece = ece
   tcpHdr.cwr = cwr
   tcpHdr.ns = ns
   tcpHdr.window = window
   tcpHdr.checksum = InitialIpChecksum
   tcpHdr.urgentPointer = urgentPointer

   if isinstance( data, str ):
      data = data.encode()
   p.rawBytesAtIdxIs( currentOffset, data )

   # Now that we've copied in the user data, update the tcp checksum.
   if version == IPV6:
      tcpHdr.ipSrc = Arnet.Ip6Addr( srcIpAddr )
      tcpHdr.ipDst = Arnet.Ip6Addr( dstIpAddr )
   else:
      tcpHdr.ipSrc = srcIpAddr
      tcpHdr.ipDst = dstIpAddr
   tcpHdr.tcpLen = TcpHdrSize + len( data )
   tcpChecksum = tcpHdr.computedChecksum
   tcpHdr.checksum = tcpChecksum

   return ( tcpHdr, currentOffset )

# Create a packet with ethernet, ip, and tcp headers using the given values.
# Return a tuple of the packet, the ethernet header wrapper, the dot1Q
# header wrapper, the ip header wrapper, the router alert
# option wrapper, the tcp header wrapper, and the current offset into the
# packet where the next header should go (if you wanted one).

def newTcpPkt( srcMacAddr=None,
               dstMacAddr=None,
               vlanId=None,
               vlanPriority=None,
               cfiBit=False,
               ethType="ethTypeIp",
               srcIpAddr="10.1.1.1",
               dstIpAddr="10.2.2.2",
               version=None,
               tos=0,
               flowLabel=0,
               ipTotalLen=None,
               ipId=0,
               fragmentOffset=0,
               moreFrag=False,
               dontFrag=False,
               ttl=None,
               protocol="ipProtoTcp",
               ipChecksum=None,
               routerAlert=False,
               srcPort=1999,
               dstPort=1999,
               sequenceNumber=0,
               acknowledgementNumber=0,
               dataOffset=None,
               fin=False,
               syn=False,
               rst=False,
               psh=False,
               ack=False,
               urg=False,
               ece=False,
               cwr=False,
               ns=False,
               window=0,
               urgentPointer=0,
               data='',
               **kwargs ):
   version = ipVersionOf( version, srcIpAddr, dstIpAddr )

   if dstMacAddr is None:
      if isIpMcastAddr( dstIpAddr ):
         dstMacAddr = ipMcastMacAddr( dstIpAddr )
      else:
         dstMacAddr = macFromIpAddr( dstIpAddr )

   if srcMacAddr is None:
      srcMacAddr = macFromIpAddr( srcIpAddr )

   dataLen = len( data )

   (p, ethHdr, qHdr, ipHdr, raOpt, currentOffset) = \
       newIpPkt( srcMacAddr=srcMacAddr,
                 dstMacAddr=dstMacAddr,
                 srcIpAddr=srcIpAddr,
                 dstIpAddr=dstIpAddr,
                 version=version,
                 totalLen=ipTotalLen,
                 id=ipId,
                 fragmentOffset=fragmentOffset,
                 moreFrag=moreFrag,
                 dontFrag=dontFrag,
                 ttl=ttl,
                 tos=tos,
                 flowLabel=flowLabel,
                 protocol=protocol,
                 checksum=ipChecksum,
                 routerAlert=routerAlert,
                 vlanId=vlanId,
                 vlanPriority=vlanPriority,
                 cfiBit=cfiBit,
                 ethType=ethType,
                 extraSize=(TcpHdrSize + dataLen),
                 **kwargs )

   ( tcpHdr, currentOffset ) = \
      createTcpHdr( p=p,
                    srcIpAddr=srcIpAddr,
                    dstIpAddr=dstIpAddr,
                    srcPort=srcPort,
                    dstPort=dstPort,
                    sequenceNumber=sequenceNumber,
                    acknowledgementNumber=acknowledgementNumber,
                    dataOffset=dataOffset,
                    fin=fin,
                    syn=syn,
                    rst=rst,
                    psh=psh,
                    ack=ack,
                    urg=urg,
                    ece=ece,
                    cwr=cwr,
                    ns=ns,
                    window=window,
                    urgentPointer=urgentPointer,
                    data=data,
                    currentOffset=currentOffset,
                    version=version )

   return (p, ethHdr, qHdr, ipHdr, raOpt, tcpHdr, currentOffset)

# Create a packet with ethernet, ip, and icmp headers using the given values.
# Return a tuple of the packet, the ethernet header wrapper, the dot1Q
# header wrapper, the ip header wrapper, the router alert
# option wrapper, the icmp header wrapper, and the current offset into the
# packet where the next header should go (if you wanted one).

def newIcmpPkt( srcMacAddr=None,
                dstMacAddr=None,
                vlanId=None,
                vlanPriority=None,
                cfiBit=False,
                ethType="ethTypeIp",
                srcIpAddr="10.1.1.1",
                dstIpAddr="10.2.2.2",
                tos=0,
                ipTotalLen=None,
                ipId=0,
                fragmentOffset=0,
                moreFrag=False,
                dontFrag=False,
                ttl=None,
                protocol="ipProtoIcmp",
                ipChecksum=None,
                routerAlert=False,
                icmpType=123,
                icmpCode=234,
                restOfHeader=0,
                data='' ):

   if dstMacAddr is None:
      if isIpMcastAddr( dstIpAddr ):
         dstMacAddr = ipMcastMacAddr( dstIpAddr )
      else:
         dstMacAddr = macFromIpAddr( dstIpAddr )

   if srcMacAddr is None:
      srcMacAddr = macFromIpAddr( srcIpAddr )

   if isinstance( data, str ):
      data = data.encode()
   dataLen = len( data )

   (p, ethHdr, qHdr, ipHdr, raOpt, currentOffset) = \
       newIpPkt( srcMacAddr=srcMacAddr,
                 dstMacAddr=dstMacAddr,
                 srcIpAddr=srcIpAddr,
                 dstIpAddr=dstIpAddr,
                 tos=tos,
                 totalLen=ipTotalLen,
                 id=ipId,
                 fragmentOffset=fragmentOffset,
                 moreFrag=moreFrag,
                 dontFrag=dontFrag,
                 ttl=ttl,
                 protocol=protocol,
                 checksum=ipChecksum,
                 routerAlert=routerAlert,
                 vlanId=vlanId,
                 vlanPriority=vlanPriority,
                 cfiBit=cfiBit,
                 ethType=ethType,
                 extraSize=(IcmpHdrSize + dataLen) )

   icmpHdr = Tac.newInstance( "Arnet::IcmpHdrWrapper",
                              p, currentOffset )
   currentOffset += IcmpHdrSize

   icmpHdr.type = icmpType
   icmpHdr.code = icmpCode
   icmpHdr.checksum = InitialIpChecksum
   icmpHdr.restOfHeader = restOfHeader

   p.rawBytesAtIdxIs( currentOffset, data )

   # Now that we've copied in the user data, update the icmp checksum.
   icmpChecksum = icmpHdr.computedChecksum[ IcmpHdrSize + dataLen ]
   icmpHdr.checksum = icmpChecksum

   return (p, ethHdr, qHdr, ipHdr, raOpt, icmpHdr, currentOffset)

def newIcmp6Pkt( srcMacAddr,
                 dstMacAddr,
                 srcIpAddr,
                 dstIpAddr,
                 icmp6Type,
                 icmp6Code,
                 ipHopLimit=255,
                 vlanId=None,
                 vlanPriority=None,
                 cfiBit=False,
                 ethType="ethTypeIp6",
                 extraSize=0,
                 icmp6Len=4,
                 baseLevel='',
                 tos=0,
                 flowLabel=0,
                 data='',
                 outerVlanId=None,
                 outerVlanTpid='ethTypeQinQ',
                 vlanTpid='ethTypeDot1Q' ):
   icmp6HdrSize = icmp6Len
   dataLen = len( data )
   icmp6Size = icmp6HdrSize + dataLen + extraSize

   if baseLevel == 'ICMP6':
      (p, ethHdr, qHdr, ipHdr, raOpt, currentOffset) = \
            ( Tac.newInstance( "Arnet::Pkt" ), None, None, None, None, 0 )
      p.newSharedHeadData = icmp6Size
   else:
      (p, ethHdr, qHdr, ipHdr, raOpt, currentOffset) = \
            newIpPkt( srcMacAddr=srcMacAddr,
                 dstMacAddr=dstMacAddr,
                 srcIpAddr=srcIpAddr,
                 dstIpAddr=dstIpAddr,
                 version=IPV6,
                 tos=tos,
                 flowLabel=flowLabel,
                 ttl=ipHopLimit,
                 protocol="ipProtoIcmpv6",
                 vlanId=vlanId,
                 vlanPriority=vlanPriority,
                 cfiBit=cfiBit,
                 ethType=ethType,
                 extraSize=( icmp6Size + dataLen ),
                 outerVlanId=outerVlanId,
                 outerVlanTpid=outerVlanTpid,
                 vlanTpid=vlanTpid )

   if isinstance( data, str ):
      data = data.encode()
   p.rawBytesAtIdxIs( currentOffset, data )

   icmp6Hdr = Tac.newInstance( "Arnet::Icmp6HdrWrapper", p, currentOffset )
   currentOffset += icmp6HdrSize
   icmp6Hdr.typeNum = icmp6Type
   icmp6Hdr.code = icmp6Code
   icmp6Hdr.checksum = InitialIpChecksum

   # Update the ICMP checksum
   icmp6Hdr.ipSrc = Arnet.Ip6Addr( srcIpAddr )
   icmp6Hdr.ipDst = Arnet.Ip6Addr( dstIpAddr )
   icmp6Hdr.icmp6LenU32 = icmp6Size
   icmp6Checksum = icmp6Hdr.computedChecksum
   icmp6Hdr.checksum = icmp6Checksum

   return( p, ethHdr, qHdr, ipHdr, raOpt, icmp6Hdr, currentOffset)

def Icmp6FillSrcLinkLayerOption( p, currentOffset, srcMacAddr ):
   srcLinkLayerAddrHdr = Tac.newInstance(
                           "Arnet::Icmp6HdrSrcLinkLayerAddrOptionWrapper",
                           p, currentOffset )
   currentOffset += Icmp6SrcLinkLayerAddressOptionLen
   srcLinkLayerAddrHdr.typeNum = 'icmp6OptionTypeSrcLinkLayerAddr'
   srcLinkLayerAddrHdr.length = 1
   srcLinkLayerAddrHdr.linkLayerAddress = srcMacAddr
   return ( srcLinkLayerAddrHdr, currentOffset )

def Icmp6FillTgtLinkLayerOption( p, currentOffset, srcMacAddr ):
   tgtLinkLayerAddrHdr = Tac.newInstance(
                           "Arnet::Icmp6HdrTgtLinkLayerAddrOptionWrapper",
                           p, currentOffset )
   currentOffset += Icmp6TgtLinkLayerAddressOptionLen
   tgtLinkLayerAddrHdr.typeNum = 'icmp6OptionTypeTgtLinkLayerAddr'
   tgtLinkLayerAddrHdr.length = 1
   tgtLinkLayerAddrHdr.linkLayerAddress = srcMacAddr
   return ( tgtLinkLayerAddrHdr, currentOffset )

def Icmp6FillPrefixInfoOption( p, currentOffset, prefixInfoList ):
   if not prefixInfoList:
      return ( None, currentOffset )

   prefixInfoHdrList = []
   for prefixInfo in prefixInfoList:
      prefixHdr = Tac.newInstance( "Arnet::Icmp6HdrPrefixInfoOptionWrapper",
                                   p, currentOffset )
      currentOffset += Icmp6PrefixInfoOptionLen
      prefixHdr.typeNum = "icmp6OptionTypePrefixInformation"
      prefixHdr.length = 4
      prefixHdr.reserved1Flag = 0
      prefixHdr.reserved2 = 0

      (prefixIp6AddrWithMask, prefixHdr.onLinkFlag, prefixHdr.autoAddressFlag, \
         prefixHdr.validLifetimeU32, prefixHdr.preferredLifetimeU32 ) = prefixInfo

      prefixHdr.prefix = prefixIp6AddrWithMask.address
      prefixHdr.prefixLength = prefixIp6AddrWithMask.len
      prefixInfoHdrList.append( prefixHdr )

      assert prefixHdr.reserved1Flag == 0
      assert prefixHdr.reserved2 == 0

   return ( prefixInfoHdrList, currentOffset )

def Icmp6FillMtuOption( p, currentOffset, mtu ):
   mtuHdr = Tac.newInstance( "Arnet::Icmp6HdrMtuOptionWrapper",
                             p, currentOffset )
   currentOffset += Icmp6MtuOptionLen
   mtuHdr.typeNum = "icmp6OptionTypeMtu"
   mtuHdr.length = 1
   mtuHdr.reserved = 0
   mtuHdr.mtuU32 = mtu
   return ( mtuHdr, currentOffset )


def Icmp6FillDnsServerInfoOption( p, currentOffset,  dnsServerInfoList ):

   if not dnsServerInfoList:
      return ( None, currentOffset )
   dnsServerHdr = Tac.newInstance( "Arnet::Icmp6HdrDnsServerInfoOptionWrapper",
                                   p, currentOffset )
   currentOffset += IcmpDnsServerInfoOptionLen( dnsServerInfoList )
   return ( dnsServerHdr, currentOffset )

def IcmpDnsServerInfoOptionLen( dnsServerInfoList ):

   numDnsServersDict = {}
   for dnsServerInfo in dnsServerInfoList:
      ( _, EntryLifetime ) = dnsServerInfo
      if EntryLifetime in numDnsServersDict:
         numDnsServersDict[ EntryLifetime ] += 1
      else:
         numDnsServersDict[ EntryLifetime ] = 1

   totalLen = 0
   for numDnsServers in numDnsServersDict.values():
      totalLen += 8 + numDnsServers * 16
   return totalLen


def Icmp6FillDnsSuffixInfoOption( p, currentOffset,  dnsSuffixInfoList ):

   if not dnsSuffixInfoList:
      return ( None, currentOffset )
   dnsSuffixHdr = Tac.newInstance( "Arnet::Icmp6HdrDnsSuffixInfoOptionWrapper",
                                   p, currentOffset )
   currentOffset += IcmpDnsSuffixInfoOptionLen( dnsSuffixInfoList )
   return ( dnsSuffixHdr, currentOffset )

def IcmpDnsSuffixInfoOptionLen( dnsSuffixInfoList ):

   totalLen = 0
   dnsSuffixesLenDict = {}
   for dnsSuffixInfo in dnsSuffixInfoList:
      ( EntryDnsSuffixStr, EntryLifetime ) = dnsSuffixInfo
      suffixLenInOption =  len(EntryDnsSuffixStr) + 1
      if EntryLifetime in dnsSuffixesLenDict:
         dnsSuffixesLenDict[ EntryLifetime ] += suffixLenInOption
      else:
         dnsSuffixesLenDict[ EntryLifetime ] = suffixLenInOption

   totalLen = 0
   for dnsSuffixLen in dnsSuffixesLenDict.values():
      # Add 8 bytes for type, len, reserved and lifetime to SuffixLenInOption
      totalLen += 8 + dnsSuffixLen
      # pad bytes to make length multiple of 8
      if totalLen % 8:
         totalLen += 8 - totalLen % 8
   return totalLen

def Icmp6FillNetworkBootOption( p, currentOffset, url, mtu ):
   networkBootHdr = Tac.newInstance( "Arnet::Icmp6HdrNetworkBootOptionWrapper",
                                     p, currentOffset )
   currentOffset += IcmpNetworkBootOptionLen( currentOffset, url, mtu )
   return ( networkBootHdr, currentOffset )

def IcmpNetworkBootOptionLen( currentOffset, url, mtu ):
   pktBufSize = 1536
   if mtu:
      pktBufSize = 1536 if mtu > 1536 else mtu
   payloadLen = len( url )
   # Add 8 bytes for type, length, paddingLen, reserved1 and reserved2
   networkBootOptionLen = 8 + payloadLen
   blen = ( pktBufSize - currentOffset ) & ~0x7
   if networkBootOptionLen > blen: # pylint: disable=consider-using-min-builtin
      networkBootOptionLen = blen
   remainder = networkBootOptionLen & 0x7
   if remainder:
      networkBootOptionLen += ( 8 - remainder )
   return networkBootOptionLen

def Icmp6RouteInformationLengthForMasklen( masklen ):
   if masklen > 64:
      return 3
   if masklen > 0:
      return 2
   return 1

def Icmp6RouteInformationOptionLen( raRouteInformationDict ):
   if not raRouteInformationDict:
      return 0
   length = 0
   for pfx in raRouteInformationDict:
      length += Icmp6RouteInformationLengthForMasklen( pfx.len )
   return length

def Icmp6FillRouteInformationOption( p, currentOffset, raRouteInformationDict ):
   if not raRouteInformationDict:
      return ( None, currentOffset )

   routeInformationHdrList = []
   for pfx, ( preference, lifetime ) in raRouteInformationDict.items():
      routeInformationHdr = Tac.newInstance(
            "Arnet::Icmp6HdrRouteInformationOptionWrapper", p, currentOffset )
      routeInformationHdr.typeNum = 'icmp6OptionTypeRouteInformation'
      length = Icmp6RouteInformationLengthForMasklen( pfx.len )
      # length is number of 8-octet chunks
      currentOffset += length * 8
      routeInformationHdr.length = length
      routeInformationHdr.reserved1 = 0
      routeInformationHdr.preference = preference
      routeInformationHdr.reserved2 = 0
      routeInformationHdr.routeLifetimeU32 = lifetime
      routeInformationHdr.prefixLength = pfx.len
      routeInformationHdr.prefix = pfx.address
      routeInformationHdrList.append( routeInformationHdr )

      assert routeInformationHdr.reserved1 == 0
      assert routeInformationHdr.reserved2 == 0

   return ( routeInformationHdrList, currentOffset )

def Icmp6FillPref64Option( p, currentOffset, pref64OptionInfo ):
   if not pref64OptionInfo:
      return ( None, currentOffset )
   ip6Prefix, lifetimeSec = pref64OptionInfo

   pref64OptionHdr = Tac.newInstance( "Arnet::Icmp6HdrPref64OptionWrapper",
                                      p, currentOffset )
   currentOffset += Icmp6Pref64OptionLen # in bytes
   pref64OptionHdr.length = 2 # in units of 8 bytes
   pref64OptionHdr.typeNum = 'icmp6OptionTypePref64'
   Pref64 = Tac.Type( "RouterAdvt::Pref64" )
   pref64OptionHdr.scaledLifetime = Pref64.scaledLifetime( lifetimeSec )
   pref64OptionHdr.plc = Pref64.plc( ip6Prefix.len )
   pref64OptionHdr.word0U32 = ip6Prefix.address.word0
   pref64OptionHdr.word1U32 = ip6Prefix.address.word1
   pref64OptionHdr.word2U32 = ip6Prefix.address.word2
   return ( pref64OptionHdr, currentOffset )

# This function is a wrapper around newIpPkt that uses an invalid next-header
#value to generate a malformed IPv6 packet.
def newIp6MalformedPkt( srcMacAddr,
                        srcIpAddr,
                        dstMacAddr,
                        dstIpAddr,
                        moreFrag=False,
                        protocol=0 ) :

   return newIpPkt( srcMacAddr=srcMacAddr,
                    dstMacAddr=dstMacAddr,
                    srcIpAddr=srcIpAddr,
                    dstIpAddr=dstIpAddr,
                    version=IPV6,
                    moreFrag=moreFrag,
                    ttl=255,
                    protocol=protocol )


# Create a Router Advt Packet with ethernet, Ip6 and ICMPv6 headers
# using the given values.
# Return a tuple of the packet, the ethernet header wrapper, the dot1Q
# header wrapper, the ip6 header wrapper, icmpv6 header wrapper,
# Router Advt header wrapper and the current offset into the packet where the next
# header should go.
# This function does not support regular expressions for DNS options
def newIcmp6RaPkt( srcMacAddr,
                   srcIpAddr,
                   dstMacAddr='33:33:00:00:00:01',
                   dstIpAddr='ff02::1',
                   raCurHopLimit=64,
                   raManagedConfigFlag=False,
                   raOtherConfigFlag=False,
                   raRouterPreference='medium',
                   raRouterLifetime=1800,
                   raRouterReachableTime=0,
                   raRetransTimer=0,
                   # raPrefixInfoListEntry -> [ prefixAddrWithMaskStr, onLink,
                   #                         autoAddress, validLifetime,
                   #                         preferredLifetime ]
                   raPrefixInfoList=None,
                   raSrcLinkLayerOptionAddress=None,
                   raMtuOption=None,
                   mtuOptionLengthZero=False,
                   raDnsServerInfoList=None,
                   raDnsServerLifetime = None,
                   raDnsSuffixInfoList=None,
                   raDnsSuffixLifetime = None,
                   raNetworkBootFileUrl = None,
                   # raRouteInformationDict ->
                   # { Ip6AddrMask : ( preference, lifetime, ... }
                   raRouteInformationDict=None,
                   raPref64OptionInfo=None, # tuple ( Ip6Prefix, lifetimeSec )
                   vlanId=None,
                   vlanPriority=None,
                   cfiBit=False,
                   ipHopLimit=255,
                   icmp6Code=0,
                   icmp6Len=4,
                   validChecksum=True,
                   baseLevel='',
                   extraSize=0,
                   mtu=None ):
   # Calculate raLen
   raLen = Icmp6RaBasicHeaderLen

   if raSrcLinkLayerOptionAddress:
      raLen += Icmp6SrcLinkLayerAddressOptionLen
   if raMtuOption:
      raLen += Icmp6MtuOptionLen
   if raPrefixInfoList:
      raLen += ( len( raPrefixInfoList ) * Icmp6PrefixInfoOptionLen )
   if raDnsServerInfoList:
      #assert( raDnsServerLifetime != None )
      raLen += IcmpDnsServerInfoOptionLen( raDnsServerInfoList )

   if raDnsSuffixInfoList:
      #assert( raDnsSuffixLifetime != None )
      raLen += IcmpDnsSuffixInfoOptionLen( raDnsSuffixInfoList )
   if raRouteInformationDict:
      raLen += 8 * Icmp6RouteInformationOptionLen( raRouteInformationDict )
   if raNetworkBootFileUrl:
      raLen += IcmpNetworkBootOptionLen( raLen, raNetworkBootFileUrl, mtu )
   if raPref64OptionInfo:
      raLen += Icmp6Pref64OptionLen

   if baseLevel == 'ICMP6_RA':
      (p, ethHdr, qHdr, ipHdr, raOpt, icmp6Hdr, currentOffset) = \
            ( Tac.newInstance( "Arnet::Pkt" ), None, None, None, None, None, 0 )
      p.newSharedHeadData = raLen
   else:
      (p, ethHdr, qHdr, ipHdr, raOpt, icmp6Hdr, currentOffset) = \
            newIcmp6Pkt( srcMacAddr,
                    dstMacAddr,
                    srcIpAddr,
                    dstIpAddr,
                    "icmp6TypeRouterAdvertisement",
                    icmp6Code,
                    vlanId=vlanId,
                    vlanPriority=vlanPriority,
                    cfiBit=cfiBit,
                    extraSize=raLen + extraSize,
                    ipHopLimit=ipHopLimit,
                    icmp6Len=icmp6Len,
                    baseLevel=baseLevel )
   # Add RA hdr
   raHdr = Tac.newInstance( "Arnet::Icmp6RouterAdvertisementHdrWrapper",
                                 p, currentOffset )
   currentOffset += Icmp6RaBasicHeaderLen
   raHdr.curHopLimit = raCurHopLimit

   raHdr.managedConfigFlag = raManagedConfigFlag
   raHdr.otherConfigFlag = raOtherConfigFlag
   raHdr.homeAgentFlag = False
   raHdr.routerPreference = raRouterPreference
   raHdr.neighborDiscoveryProxyFlag = False
   raHdr.reservedFlag = 0

   raHdr.routerLifetimeU16 = raRouterLifetime
   raHdr.reachableTimeU32 =  raRouterReachableTime
   raHdr.retransTimerU32 = raRetransTimer

   assert raHdr.routerPreference == raRouterPreference

   # Add src Link Layer address option
   srcLinkLayerAddrHdr = None
   if raSrcLinkLayerOptionAddress:
      ( srcLinkLayerAddrHdr, currentOffset ) = \
                     Icmp6FillSrcLinkLayerOption( p, currentOffset, \
                           raSrcLinkLayerOptionAddress)

   assert raHdr.routerPreference == raRouterPreference

   mtuHdr = None
   if raMtuOption:
      ( mtuHdr, currentOffset ) = Icmp6FillMtuOption( p, currentOffset, raMtuOption )

   assert raHdr.routerPreference == raRouterPreference

   if raPrefixInfoList:
      ( prefixInfoHdrList, currentOffset ) = Icmp6FillPrefixInfoOption( p,
                                                currentOffset, raPrefixInfoList )
   else:
      prefixInfoHdrList = None

   if raDnsServerInfoList:
      ( dnsServerInfoHdr, currentOffset ) = Icmp6FillDnsServerInfoOption( p,
                                            currentOffset, raDnsServerInfoList)
   else:
      dnsServerInfoHdr = None

   if raDnsSuffixInfoList:
      ( dnsSuffixInfoHdr, currentOffset ) = Icmp6FillDnsSuffixInfoOption( p,
                                            currentOffset, raDnsSuffixInfoList )
   else:
      dnsSuffixInfoHdr = None

   if raRouteInformationDict:
      ( routeInformationHdrList, currentOffset ) = Icmp6FillRouteInformationOption(
            p, currentOffset, raRouteInformationDict )
   else:
      routeInformationHdrList = None

   if raNetworkBootFileUrl:
      ( networkBootHdr, currentOffset ) = Icmp6FillNetworkBootOption( p,
                                          currentOffset, raNetworkBootFileUrl, mtu )
   else:
      networkBootHdr = None

   ( pref64Hdr, currentOffset ) = Icmp6FillPref64Option( p, currentOffset,
                                                         raPref64OptionInfo )

   assert raHdr.routerPreference == raRouterPreference

   # used to test validation error case where option has length 0
   if raMtuOption and mtuOptionLengthZero:
      mtuHdr.length = 0

   # when validChecksum=False:
   #   used to test validation error case where checksum is invalid
   icmp6Hdr.checksum = 0 if validChecksum else 1
   icmp6Checksum = icmp6Hdr.computedChecksum
   icmp6Hdr.checksum = icmp6Checksum

   assert raHdr.routerPreference == raRouterPreference

   return ( p, ethHdr, qHdr, ipHdr, raOpt, icmp6Hdr, raHdr,
            srcLinkLayerAddrHdr, mtuHdr, prefixInfoHdrList,
            dnsServerInfoHdr, dnsSuffixInfoHdr, networkBootHdr, currentOffset,
            routeInformationHdrList, pref64Hdr )

# Create a Router Solicitation Packet with ethernet, Ip6 and ICMPv6 headers
# using the given values.
# Return a tuple of the packet, the ethernet header wrapper, the dot1Q
# header wrapper, the ip6 header wrapper, icmpv6 header wrapper,
# Router Advt header wrapper and the current offset into the packet where the next
# header should go.
def newIcmp6RsPkt( srcMacAddr,
                   srcIpAddr,
                   dstMacAddr='33:33:00:00:00:02',
                   dstIpAddr='ff02::2',
                   rsSrcLinkLayerOptionAddress=None,
                   vlanId=None,
                   vlanPriority=None,
                   cfiBit=False,
                   ipHopLimit=255,
                   icmp6Code=0,
                   icmp6Len=4,
                   baseLevel='',
                   extraSize=0,
                   outerVlanId=None,
                   outerVlanTpid="ethTypeQinQ",
                   vlanTpid="ethTypeDot1Q" ):
   # Calculate rsLen
   raLen = Icmp6RsBasicHeaderLen
   if rsSrcLinkLayerOptionAddress:
      raLen += Icmp6SrcLinkLayerAddressOptionLen

   if baseLevel == 'ICMP6_RS':
      (p, ethHdr, qHdr, ipHdr, raOpt, icmp6Hdr, currentOffset) = \
            ( Tac.newInstance( "Arnet::Pkt" ), None, None, None, None, None, 0 )
      p.newSharedHeadData = raLen
   else:
      (p, ethHdr, qHdr, ipHdr, raOpt, icmp6Hdr, currentOffset) = \
            newIcmp6Pkt( srcMacAddr,
                    dstMacAddr,
                    srcIpAddr,
                    dstIpAddr,
                    "icmp6TypeRouterSolicitation",
                    icmp6Code,
                    vlanId=vlanId,
                    vlanPriority=vlanPriority,
                    cfiBit=cfiBit,
                    extraSize=raLen + extraSize,
                    ipHopLimit=ipHopLimit,
                    icmp6Len=icmp6Len,
                    baseLevel=baseLevel,
                    outerVlanId=outerVlanId,
                    outerVlanTpid=outerVlanTpid,
                    vlanTpid=vlanTpid )

   # Add RA hdr
   rsHdr = Tac.newInstance( "Arnet::Icmp6RouterSolicitationHdrWrapper",
                                 p, currentOffset )
   currentOffset += Icmp6RsBasicHeaderLen
   rsHdr.reservedU32 = 0

   # Add src Link Layer address option
   srcLinkLayerAddrHdr = None
   if rsSrcLinkLayerOptionAddress:
      ( srcLinkLayerAddrHdr, currentOffset ) = Icmp6FillSrcLinkLayerOption(
                                 p, currentOffset, rsSrcLinkLayerOptionAddress )

   icmp6Hdr.checksum = 0
   icmp6Checksum = icmp6Hdr.computedChecksum
   icmp6Hdr.checksum = icmp6Checksum

   return ( p, ethHdr, qHdr, ipHdr, raOpt, icmp6Hdr, rsHdr,
            srcLinkLayerAddrHdr, currentOffset )


# Create a Neighbor Solicitation Packet with ethernet, Ip6 and ICMPv6 headers
# using the given values.
# Return a tuple of the packet, the ethernet header wrapper, the dot1Q
# header wrapper, the ip6 header wrapper, icmpv6 header wrapper,
# If no dstIpAddr or dstMacAddr they are inferred from the targetIpAddr by computing
# the solicited-node multicast address.
def newIcmp6NsPkt( srcMacAddr,
                   srcIpAddr,
                   dstMacAddr=None,
                   dstIpAddr=None,
                   targetIpAddr=None,
                   nsSrcLinkLayerOptionAddress=None,
                   vlanId=None,
                   vlanPriority=None,
                   cfiBit=False,
                   extraSize=0,
                   ipHopLimit=255,
                   outerVlanId=None,
                   outerVlanTpid="ethTypeQinQ",
                   vlanTpid="ethTypeDot1Q" ):
   # Calculate icmp6 hdrLen
   nsLen = Icmp6NsBasicHeaderLen
   if nsSrcLinkLayerOptionAddress:
      nsLen += Icmp6SrcLinkLayerAddressOptionLen

   assert targetIpAddr

   if not dstIpAddr:
      dstIpAddr = ip6SolicitedNodeMcastAddrFromIp( targetIpAddr )

   if not dstMacAddr:
      dstMacAddr = macFromIp6Multicast( dstIpAddr )

   (p, ethHdr, qHdr, ipHdr, rtrOpt, icmp6Hdr, currentOffset) = \
       newIcmp6Pkt( srcMacAddr,
                    dstMacAddr,
                    srcIpAddr,
                    dstIpAddr,
                    "icmp6TypeNeighborSolicitation",
                    0,
                    ipHopLimit=ipHopLimit,
                    vlanId=vlanId,
                    vlanPriority=vlanPriority,
                    cfiBit=cfiBit,
                    extraSize=nsLen + extraSize,
                    outerVlanId=outerVlanId,
                    outerVlanTpid=outerVlanTpid,
                    vlanTpid=vlanTpid )
   # Add NS hdr
   nsHdr = Tac.newInstance( "Arnet::Icmp6NeighborSolicitationHdrWrapper",
                            p, currentOffset )
   nsHdr.targetIp6Addr = Arnet.Ip6Addr( targetIpAddr )
   nsHdr.reserved = 0
   currentOffset += Icmp6NsBasicHeaderLen

   # Add src Link Layer address option
   srcLinkLayerAddrHdr = None
   if nsSrcLinkLayerOptionAddress:
      ( srcLinkLayerAddrHdr, currentOffset ) = Icmp6FillSrcLinkLayerOption(
                                 p, currentOffset, nsSrcLinkLayerOptionAddress )

   icmp6Hdr.checksum = 0
   icmp6Checksum = icmp6Hdr.computedChecksum
   icmp6Hdr.checksum = icmp6Checksum

   return ( p, ethHdr, qHdr, ipHdr, rtrOpt, icmp6Hdr, nsHdr,
            srcLinkLayerAddrHdr, currentOffset )


# Create a Neighbor Advt Packet with ethernet, Ip6 and ICMPv6 headers
# using the given values.
# Return a tuple of the packet, the ethernet header wrapper, the dot1Q
# header wrapper, the ip6 header wrapper, icmpv6 header wrapper,
def newIcmp6NaPkt( srcMacAddr,
                   srcIpAddr,
                   dstMacAddr='33:33:00:00:00:01',
                   dstIpAddr='ff02::1',
                   naTgtIpAddr=None,
                   senderIsRouter=False,
                   solicited=False,
                   override=True,
                   naTgtLinkLayerOptionAddress=None,
                   vlanId=None,
                   vlanPriority=None,
                   cfiBit=False,
                   extraSize=0 ):
   # Calculate naLen
   naLen = Icmp6NaBasicHeaderLen

   if naTgtLinkLayerOptionAddress:
      naLen += Icmp6TgtLinkLayerAddressOptionLen

   (p, ethHdr, qHdr, ipHdr, rtrOpt, icmp6Hdr, currentOffset) = \
       newIcmp6Pkt( srcMacAddr,
                    dstMacAddr,
                    srcIpAddr,
                    dstIpAddr,
                    "icmp6TypeNeighborAdvertisement",
                    0,
                    vlanId=vlanId,
                    vlanPriority=vlanPriority,
                    cfiBit=cfiBit,
                    extraSize=naLen + extraSize )
   # Add Neighbor Advertisement hdr
   naHdr = Tac.newInstance( "Arnet::Icmp6NeighborAdvertisementHdrWrapper",
                            p, currentOffset )
   currentOffset += Icmp6NaBasicHeaderLen
   naHdr.senderIsRouter = senderIsRouter
   naHdr.solicited = solicited
   naHdr.override = override
   naHdr.reserved = 0

   # Add tgt ip address
   if naTgtIpAddr:
      naHdr.targetIp6Addr = Arnet.Ip6Addr(naTgtIpAddr)
   else:
      naHdr.targetIp6Addr = Arnet.Ip6Addr(srcIpAddr)

   # Add tgt Link Layer address option
   tgtLinkLayerAddrHdr = None
   if naTgtLinkLayerOptionAddress:
      ( tgtLinkLayerAddrHdr, currentOffset ) = Icmp6FillTgtLinkLayerOption(
         p, currentOffset, naTgtLinkLayerOptionAddress)

   icmp6Hdr.checksum = 0
   icmp6Checksum = icmp6Hdr.computedChecksum
   icmp6Hdr.checksum = icmp6Checksum

   return ( p, ethHdr, qHdr, ipHdr, rtrOpt, icmp6Hdr, naHdr, \
            tgtLinkLayerAddrHdr, currentOffset )

def newIcmp6EchoRequestPkt( srcMacAddr,
                            srcIpAddr,
                            dstMacAddr='33:33:00:00:00:02',
                            dstIpAddr='ff02::2',
                            identifier=0,
                            seqno=0,
                            vlanId=None,
                            vlanPriority=None,
                            cfiBit=False,
                            ipHopLimit=255,
                            icmp6Len=4,
                            data='',
                            extraSize=0 ):
   ping6Len = 4
   ( p, ethHdr, qHdr, ipHdr, ping6Opt, icmp6Hdr, currentOffset ) = \
         newIcmp6Pkt( srcMacAddr,
               dstMacAddr,
               srcIpAddr,
               dstIpAddr,
               "icmp6TypeEchoRequest",
               icmp6Code=0,
               vlanId=vlanId,
               vlanPriority=vlanPriority,
               cfiBit=cfiBit,
               extraSize=ping6Len + extraSize + len( data ),
               ipHopLimit=ipHopLimit,
               icmp6Len=icmp6Len )
   ping6Hdr = Tac.newInstance( "Arnet::Icmp6EchoRequestHdrWrapper",
                                 p, currentOffset )
   currentOffset += ping6Len
   ping6Hdr.identifierU16 = identifier
   ping6Hdr.seqnoU16 = seqno

   if isinstance( data, str ):
      data = data.encode()
   p.rawBytesAtIdxIs( currentOffset, data )

   icmp6Hdr.checksum = 0
   icmp6Checksum = icmp6Hdr.computedChecksum
   icmp6Hdr.checksum = icmp6Checksum

   return ( p, ethHdr, qHdr, ipHdr, ping6Opt, icmp6Hdr, ping6Hdr )

def convertMaskLen( maskLen, ipVersion=IPV4 ):
   if ipVersion == IPV4:
      return maskLen
   elif ipVersion == IPV6:
      return 0 if maskLen == 0 else 96 + maskLen
   else:
      assert False, "ipVersion neither v4 nor v6"

def convertIpv4MaskLen( maskLen ):
   return convertMaskLen( maskLen, ipVersion=IPV4 )

def convertIpv6MaskLen( maskLen ):
   return convertMaskLen( maskLen, ipVersion=IPV6 )

def convertIpAddr( ipAddr, offset=0, ipVersion=IPV4 ):
   if ipVersion == IPV4:
      return ipAddr
   elif ipVersion == IPV6:
      if ipAddr.find( ':' ) != -1:
         return ipAddr
      ip4Addr = ipAddr
      m = re.match( r"(([0-9]+)\.([0-9]+)\.([0-9]+)\.([0-9]+))/?(.*)", ip4Addr )
      ipv4Num = Arnet.IpAddress( m.group( 1 ) ).value
      if ipv4Num == 0: # for default route
         ipv6BaseAddr = '::'
      elif ( ipv4Num & 0xFF000000 ) == Arnet.IpAddress( '127.0.0.0' ).value:
         # 127 is a special subnet. This has the V4 loopback addr.
         # Use only the last 3 bytes in the conversion.
         # 127.0.0.1 --> ::1
         # 127.1.1.1 --> '::1:101'
         ipv6BaseAddr = '::'  # loopback
         ipAddr127str = '0'
         for s in m.groups()[ 2:5 ]:
            ipAddr127str += '.' + s
         ipv4Num = Arnet.IpAddress( ipAddr127str ).value
      elif ( ipv4Num & 0xFFFFFF00 ) == Arnet.IpAddress( '254.128.0.0' ).value:
         ipv6BaseAddr = 'fe80::'  # link local unicast
      elif ( ipv4Num & 0xFFFFFF00 ) == Arnet.IpAddress( '224.0.0.0' ).value:
         ipv6BaseAddr = 'ff02::'  # link local
      elif ( ipv4Num & 0xFFFF0000 ) == Arnet.IpAddress( '239.255.0.0' ).value:
         ipv6BaseAddr = 'ff03::'  # local scope
      elif ( ipv4Num & 0xFFFFC000 ) == Arnet.IpAddress( '239.192.0.0' ).value:
         ipv6BaseAddr = 'ff08::'  # organization scope
      elif ipv4Num >= Arnet.IpAddress( '224.0.1.0' ).value and \
           ipv4Num <= Arnet.IpAddress( '238.255.255.255' ).value:
         # In some platforms (like T4) ff0x is considered as reserved so using ff1x
         ipv6BaseAddr = 'ff18::'  # global scope
      else:
         ipv6BaseAddr = '2002::'

      ipObj = Arnet.IpAddress( ipv6BaseAddr, AF_INET6 )
      ipLong = Arnet.numFromAddr( ipObj )
      ip = ipLong | ( ipv4Num << offset )
      ip6Addr = str( Arnet.IpAddress( ip, AF_INET6 ) ) # pylint: disable-msg=W0621

      # if maskLen is specified with the ip4Addr
      maskLenStr = m.group( 6 )
      if maskLenStr:
         ip6Addr = ip6Addr + '/%d' % \
             ( convertMaskLen( int( maskLenStr ),
                  ipVersion=ipVersion ) - offset )
      return ip6Addr
   else:
      assert False, "ipVersion neither v4 nor v6"

def convertIpAddrList( ipAddrList, ipVersion=IPV4 ):
   result = []
   for ipAddr in ipAddrList:
      result.append( convertIpAddr( ipAddr, ipVersion=ipVersion ) )
   return result

def convertIpv4Addr( addr, offset=0 ):
   return convertIpAddr( addr, offset=offset, ipVersion=IPV4 )

def convertIpv6Addr( addr, offset=0 ):
   return convertIpAddr( addr, offset=offset, ipVersion=IPV6 )

# Helper function for Arnet::Ip6Addr
def ip6Addr( ip6AddrStr ):
   ip = Tac.Value( 'Arnet::Ip6Addr' )
   ip.stringValue = ip6AddrStr
   return ip

# Helper function for Arnet::Ip6AddrWithMask
def ip6AddrWithMask( ip6AddrStr, maskLen ):
   return Tac.Value( 'Arnet::Ip6AddrWithMask', ip6Addr( ip6AddrStr), maskLen )

# ip6KernelAddrCommon
#     Returns addrDict
#        addrDict:
#           Key: intfname
#           value: intfDict
#        intfDict:
#           Keys: name, flags, mtu, addr
#           intfDict[ 'addr' ] is a list of addrEntryDict
#        addrEntryDict
#           Dictionary of each IPv6 address configured on the interface.
#           Keys: addr, scope, dynamic, validLft, prefLft
def ip6KernelAddrCommon( ip6AddrCmdOutput ):
   addrDict = {}
   intfDict = {}
   addrEntryDict = {}
   addrLftLineExpected = False
   for line in ip6AddrCmdOutput:
      if not line:
         continue
      if addrLftLineExpected:
         # We should have a NON-Empty addrEntryDict
         assert addrEntryDict
         addrLftLinePattern = \
            ' +valid_lft (?P<validLft>((?P<validLftVal>[0-9]+)sec)|(forever))' \
            ' preferred_lft (?P<prefLft>((?P<prefLftVal>[0-9]+)sec)|(forever))'
         match = re.match( addrLftLinePattern, line )
         assert match
         validLftVal = match.group( 'validLftVal' )
         if not validLftVal:
            validLftVal = match.group( 'validLft' )
         assert validLftVal
         addrEntryDict[ 'validLft' ] = validLftVal
         prefLftVal = match.group( 'prefLftVal' )
         if not prefLftVal:
            prefLftVal = match.group( 'prefLft' )
         assert prefLftVal
         addrEntryDict[ 'prefLft' ] = prefLftVal
         # Add the addrEntryDict to list of addresses on the interface.
         intfDict[ 'addr' ].append( addrEntryDict )
         addrEntryDict = {}
         addrLftLineExpected = False
      else:
         # Check if this is Address Line Pattern:
         #  inet6 2000:1:1:1:5478:bfff:fe16:3c99/64 scope global dynamic
         # Here, the keyword 'dynamic' is optional
         #  Scope can be global or local
         addrLinePattern = \
            r' +inet6 (?P<addr>[0-9,a-f,:]+/[0-9]+) (scope (?P<scope>\S+))' \
            ' (?P<dynamic>dynamic)?'
         match = re.match( addrLinePattern, line )
         if match:
            assert intfDict and not addrEntryDict
            addrEntryDict[ 'addr' ] = match.group( 'addr' )
            addrEntryDict[ 'scope' ] = match.group( 'scope' )
            if match.group( 'dynamic' ):
               addrEntryDict[ 'dynamic' ] = True
            addrLftLineExpected = True
         else:
            # Must be an interface line
            # Interface line Pattern:
            #  5: macvlan-bond0@bond0: <BROADCAST,MULTICAST,UP,LOWER_UP> mtu 9000
            intfLinePattern = \
               r'\d+: (?P<name>[^:@\s]+)(@.*)?: <(?P<flags>[\S^>]+)> ' \
               r'(mtu (?P<mtu>\d+))?'
            match = re.match( intfLinePattern, line )
            assert match
            if intfDict:
               addrDict[ intfDict[ 'name' ] ] = intfDict
               intfDict = {}
            intfDict[ 'name' ] = match.group( 'name' )
            intfDict[ 'flags' ] = match.group( 'flags' )
            intfDict[ 'mtu' ] = match.group( 'mtu' )
            intfDict[ 'addr' ] = []
   if intfDict:
      addrDict[ intfDict[ 'name' ] ] = intfDict
   return addrDict

def ipAddrRoutable( ip, ipVersion=IPV4 ):
   if ipVersion == IPV4:
      ipAddr = Tac.Value( "Arnet::IpAddr", Arnet.IpAddress( ip ).value )
      return not ( ipAddr.isLoopback or ipAddr.isMulticast or \
                   ipAddr.isReserved( False ) )
   elif ipVersion == IPV6:
      # pylint: disable-msg=W0621
      ip6Addr = Tac.Value( "Arnet::Ip6Addr", stringValue=ip )
      return not ( ip6Addr.isMulticast or ip6Addr.isLinkLocal )
   else:
      return False

def compressIp6AddrOrPrefix( addr ):
   flds = addr.split( '/' )
   if len( flds ) == 1:
      return str( ip6Addr( flds[ 0 ] ) )
   else:
      return str( ip6Addr( flds[ 0 ] ) ) + '/' + flds[ 1 ]

def getTos( dscp, ecn ):
   return ( dscp << 2 ) | ecn

def randIpHdrFields( inner=False, srcIpAddr=None, dstIpAddr=None, dscp=None,
                      ecn=None, ipId=0, fragmentOffset=0, moreFrag=False,
                      dontFrag=False, ttl=None, protocol=None, flowLabel=0,
                      version=None ):
   version = ipVersionOf( version, srcIpAddr, dstIpAddr )
   if version not in [ 4, 6 ]:
      print( "Invalid IP version specified" )
      return None

   if not srcIpAddr and not dstIpAddr:
      ipAddrs = randIpAddr( num=2, version=version )
      srcIpAddr = ipAddrs[ 0 ]
      dstIpAddr = ipAddrs[ 1 ]
   elif not srcIpAddr:
      srcIpAddr = randIpAddr( version=version )[ 0 ]
   elif not dstIpAddr:
      dstIpAddr = randIpAddr( version=version )[ 0 ]
   if not dscp:
      dscp = randIpDscp()
   if not ecn:
      ecn = randIpEcn()
   tos = getTos( dscp, ecn )
   if not ttl:
      ttl = randIpTtl()
   if not protocol:
      protocol = random.choice( [ 'ipProtoUdp', 'ipProtoTcp' ] )

   fields = dict( srcIpAddr=srcIpAddr, dstIpAddr=dstIpAddr, tos=tos,
                  ttl=ttl, protocol=protocol, version=version )
   if version == 4:
      extraFields = dict( ipId=ipId, fragmentOffset=fragmentOffset,
                          dontFrag=dontFrag, moreFrag=moreFrag )
   elif version == 6:
      extraFields = dict( flowLabel=flowLabel )
   fields.update( extraFields )

   if inner:
      keys = list( fields )
      for key in keys:
         fields[ key + 'Inner' ] = fields.pop( key )

   return fields

def randomMcastIpAndMac( ver=IPV4 ):
   ipAddr = ''
   mac = ''
   b1 = random.getrandbits( 8 )
   b2 = random.getrandbits( 8 )
   b3 = random.getrandbits( 8 )
   if ver == IPV4:
      b0 = random.randint( 224, 239 )
      ipAddr = f'{b0}.{b1}.{b2}.{b3}'
      b1 &= 0x7f
      mac = f'01:00:5e:{b1:02x}:{b2:02x}:{b3:02x}'
   elif ver == IPV6:
      b0 = random.getrandbits( 8 )
      b4 = random.getrandbits( 8 )
      b5 = random.getrandbits( 16 )
      b6 = random.getrandbits( 16 )
      b7 = random.getrandbits( 16 )
      b8 = random.getrandbits( 16 )
      b9 = random.getrandbits( 16 )
      ipAddr = f'ff{b4:02x}:{b5:04x}:{b6:04x}:{b7:04x}:{b8:04x}:'\
               f'{b9:04x}:{b0:02x}{b1:02x}:{b2:02x}{b3:02x}'
      mac = f'33:33:{b0:02x}:{b1:02x}:{b2:02x}:{b3:02x}'
   return ( ipAddr, mac )

def randMplsHdrFields( num=1 ):
   label, ttl, exp = [], [], []
   for _ in range( 0, num ):
      label.append( random.randint( 16, 99999 ) )
      ttl.append( random.randint( 2, 255 ) )
      exp.append( random.randint( 0, 8 ) )
   return { 'labels':label, 'mplsTtl':ttl, 'exp':exp }

def randIpDscp():
   return random.randint( 0, 63 )

def randIpEcn():
   return random.randint( 0, 3 )

def randIpTtl():
   return random.randint( 2, 255 )

def randGreProtocol():
   return random.randint( 0, 65535 )

def randIp4Addr( num=1, prefixLength=32, exclude=None ):
   # Prefix length too small
   if ( 1 << prefixLength ) < num:
      return []

   # Exclude multicast and reserved IP addresses
   invalidHighOctets = [ 0, 10, 127, 169, 172, 192, 198, 203 ]
   invalidHighOctets.extend( range( 224, 256 ) )
   octet = range( 0, 255 )
   lowOctet = range( 1, 254 )
   highOctet = [ i for i in octet if i not in invalidHighOctets ]

   def genIpAddr():
      res = [ random.choice( highOctet ), random.choice( octet ), \
              random.choice( octet ), random.choice( lowOctet ) ]
      res = [ str( i ) for i in res ]
      return '.'.join( res )

   if not exclude:
      exclude = []
   else:
      exclude = [ IpAddrWithMask( ip, prefixLength ).allZerosAddr for ip in exclude ]
   ipAddrs = []
   while num > 0:
      ipAddrStr = genIpAddr()
      ipAddr = IpAddrWithMask( ipAddrStr, prefixLength )
      allOnesAddr = ipAddr.allOnesAddr
      allZerosAddr = ipAddr.allZerosAddr
      if allZerosAddr not in exclude and \
            ( prefixLength == 32 or ipAddrStr != allOnesAddr ) and \
            ( prefixLength == 32 or ipAddrStr != allZerosAddr ):
         exclude.append( allZerosAddr )
         ipAddrs.append( ipAddrStr )
         num -= 1

   return ipAddrs

def randIp6Addr( num=1, prefixLength=128, exclude=None ):
   # Prefix length too small
   if ( 1 << prefixLength ) < num:
      return []

   # Exclude reserved IP addresses
   invalidHighTwoBytes = [ 0, 256, 100, 8193, 8194, 64512, 65152, 65280 ]
   twoBytes = range( 0, 65536 )
   lowTwoBytes = range( 1, 65534 )
   highTwoBytes = [ i for i in twoBytes if i not in invalidHighTwoBytes ]

   def genIpAddr():
      res = [ random.choice( highTwoBytes ), random.choice( twoBytes ),
              random.choice( twoBytes ), random.choice( twoBytes ),
              random.choice( twoBytes ), random.choice( twoBytes ),
              random.choice( twoBytes ), random.choice( lowTwoBytes ) ]
      res = [ hex( i )[ 2: ] for i in res ]
      return Ip6Addr( ':'.join( res ) )

   if not exclude:
      exclude = []
   else:
      newExclude = []
      for ip in exclude:
         addr = Ip6Addr( ip )
         addrWithMask = Ip6AddrWithMask( addr, prefixLength )
         newExclude.append( addrWithMask.subnet )
      exclude = newExclude
   ipAddrs = []
   while num > 0:
      ipAddr = genIpAddr()
      ipAddrWithMask = Ip6AddrWithMask( ipAddr, prefixLength )
      if ipAddrWithMask.subnet not in exclude and ipAddr.isUnicast and \
         not ipAddr.isLinkLocalUnicast2:
         exclude.append( ipAddrWithMask.subnet )
         ipAddrs.append( ipAddr.stringValue )
         num -= 1

   return ipAddrs

def randIpAddr( num=1, prefixLength=None, exclude=None, version=4 ):
   if version == 4:
      if not prefixLength:
         prefixLength = 32
      return randIp4Addr( num, prefixLength, exclude )
   elif version == 6:
      if not prefixLength:
         prefixLength = 128
      return randIp6Addr( num, prefixLength, exclude )
   else:
      print( "Invalid IP version specified." )
      return None

# getRandomIp was originally in Arnet.ArpTestLib
def getRandomIp( v6 ):
   if not v6:
      ip = '.'.join( '%s' % random.randint( 0, 255 ) for i in range( 4 ) )
   else:
      ip = "1001:abcd:" + \
           ":".join( "%x" % random.randint( 0, 16**4 - 1 ) for i in range( 6 ) )
   return ip

# randomIpAddr was originally in Arnet.ArpSmashTestLib
def randomIpAddr():
   '''Return a random unicast IPv4 address.'''
   # Avoid the zero address, multicast and broadcast addresses, and network broadcast
   # addresses.
   byte0 = random.getrandbits( 7 ) | 1
   byte1 = random.getrandbits( 8 )
   byte2 = random.getrandbits( 8 )
   byte3 = random.getrandbits( 7 ) << 1
   return f"{byte0}.{byte1}.{byte2}.{byte3}"

# randomIp6Addr was originally in Arnet.ArpSmashTestLib
def randomIp6Addr():
   '''Return a random unicast IPv6 address.'''
   word0 = 0x2000
   word1 = random.getrandbits( 16 )
   word2 = random.getrandbits( 16 )
   word3 = random.getrandbits( 16 )
   word4 = random.getrandbits( 16 )
   word5 = random.getrandbits( 16 )
   word6 = random.getrandbits( 16 )
   word7 = random.getrandbits( 15 ) << 1
   return (
      f"{word0:0>4x}:{word1:0>4x}:{word2:0>4x}:{word3:0>4x}:"
      f"{word4:0>4x}:{word5:0>4x}:{word6:0>4x}:{word7:0>4x}"
   )

ip6ExtensionProtocols = {
   IPPROTO_HOPOPTS : 'ipProtoIpv6HopByHop',
   IPPROTO_FRAGMENT : 'ipProtoIpv6Frag',
   IPPROTO_AH : 'ipProtoAh',
   IPPROTO_DSTOPTS : 'ipProtoIpv6Opts',
   IPPROTO_ROUTING : 'ipProtoIpv6Route',
   IPPROTO_NONE : 'ipProtoNone'
   }

def isIpv6ExtensionProtocol( protocol ):
   if type( protocol ) == str: # pylint: disable=unidiomatic-typecheck
      return protocol in ip6ExtensionProtocols.values()
   return protocol in ip6ExtensionProtocols

def newIp6PktWithExtHdrs( ipVersion, extHdr, _srcIpAddr, _dstIpAddr,
                          srcPort, dstPort, data,
                          vlanId, dstMac, srcMac, ipProto='ipProtoTcp',
                          tos=0, flowLabel=0 ):
   """ Create an IPv6 packet with one or more extension headers.
       ipProto must be one of ipProtoTcp(default), ipProtoUdp, or ipProtoSctp
       extHdr is a list of one or more headers from the following list:
          'ipProtoIpv6HopByHop', 'ipProtoAh', 'ipProtoIpv6Opts',
          'ipProtoIpv6Route', 'ipProtoIpv6Frag'
       Extension headers are added once and in the order of the above list.
   """
   pkt = None
   currentOffset = None
   optionLen = 2
   dataLen = len( data )
   if ipProto == 'ipProtoTcp':
      ipProtoHdrSize = TcpHdrSize
   elif ipProto == 'ipProtoUdp':
      ipProtoHdrSize = UdpHdrSize
   elif ipProto == 'ipProtoSctp':
      ipProtoHdrSize = SctpHdrSize
   else:
      assert 0, "Invalid ipProto %s" % ipProto

   extHdrSize = len( extHdr ) * ( optionLen + 1 ) * 8
   size = ipProtoHdrSize + dataLen + extHdrSize

   ( pkt, ethHdr, _, ipHdr, _, currentOffset ) = \
       newIpPkt( srcMacAddr=srcMac,
                 dstMacAddr=dstMac,
                 vlanId=vlanId,
                 srcIpAddr=_srcIpAddr,
                 dstIpAddr=_dstIpAddr,
                 version=ipVersion,
                 protocol=ipProto,
                 data=data,
                 extraSize=size )

   ipHdrNextHdr = ipHdr.nextHeader
   # Need to add Extension Headers in correct order
   allExtHdrs = [ 'ipProtoIpv6HopByHop', 'ipProtoAh', 'ipProtoIpv6Opts',
                  'ipProtoIpv6Route', 'ipProtoIpv6Frag' ]
   ipNextHdrSet = False
   remainingHdrs = list( extHdr )
   remainingAllExtHdrs = list( allExtHdrs )
   for hdr in allExtHdrs:
      if hdr not in extHdr: # pylint: disable=no-else-continue
         continue
      else:
         remainingHdrs.remove( hdr )
         remainingAllExtHdrs.remove( hdr )
         if not ipNextHdrSet:
            ipHdr.protocolNum = hdr
            ipNextHdrSet = True

         ip6HdrOpt = Tac.newInstance( "Arnet::Ip6HdrOptionCommonWrapper",
                                      pkt, currentOffset )
         if not remainingHdrs:
            ip6HdrOpt.nextHdrNum = ipHdrNextHdr
         else:
            for nextHdr in remainingAllExtHdrs:
               if nextHdr in extHdr:
                  ip6HdrOpt.nextHdrNum = nextHdr
                  break

         if hdr == "ipProtoIpv6Frag":
            currentOffset += 8
         else:
            ip6HdrOpt.optionLength = optionLen
            currentOffset += ( optionLen + 1 ) * 8

         # Note that this instance is unused within python, but we
         # must bind it to a local variable or it gets garbage
         # collected
         _pad = Tac.newInstance( "Arnet::Ip6HdrPadNWrapper", pkt, currentOffset )

   if ipProto == 'ipProtoTcp':
      createTcpHdr( p=pkt,
                    srcIpAddr=_srcIpAddr,
                    dstIpAddr=_dstIpAddr,
                    srcPort=srcPort,
                    dstPort=dstPort,
                    sequenceNumber=0,
                    acknowledgementNumber=0,
                    dataOffset=None,
                    fin=False,
                    syn=True,
                    rst=False,
                    psh=False,
                    ack=False,
                    urg=False,
                    ece=False,
                    cwr=False,
                    ns=False,
                    window=0,
                    urgentPointer=0,
                    data=data,
                    currentOffset=currentOffset,
                    version=ipVersion )
   elif ipProto == 'ipProtoUdp':
      udpLen = UdpHdrSize + len( data )
      createUdpHdr( p=pkt,
                    srcIpAddr=_srcIpAddr,
                    dstIpAddr=_dstIpAddr,
                    srcPort=srcPort,
                    dstPort=dstPort,
                    data=data,
                    udpLen=udpLen,
                    currentOffset=currentOffset,
                    version=ipVersion )
   elif ipProto == 'ipProtoSctp':
      createSctpHdr( p=pkt,
                     srcIpAddr=_srcIpAddr,
                     dstIpAddr=_dstIpAddr,
                     srcPort=srcPort,
                     dstPort=dstPort,
                     data=data,
                     verificationTag=1000,
                     currentOffset=currentOffset,
                     version=ipVersion )
   return pkt, ethHdr

ipProtoNameFromValue = [
   '', 'icmp', 'igmp', 'ggp', 'ipv4', 'st', 'tcp', 'cbt',
   'egp', 'igp', 'bbn-rcc', 'nvp', 'pup', 'argus', 'emcon', 'xnet', 'chaos', 'udp',
   'mux', 'dcn', 'hmp', 'prm', 'xns-idp', 'trunk-1', 'trunk-2', 'leaf-1', 'leaf-2',
   'rdp', 'irtp', 'iso-tp4', 'netblt', 'mfe-nsp', 'merit-inp', 'dccp', '3pc',
   'idpr', 'xtp', 'ddp', 'idpr-cmtp', 'tp++', 'il', 'ipv6', 'sdrp', 'ipv6-route',
   'ipv6-frag', 'idrp', 'rsvp', 'gre', 'dsr', 'bna', 'esp', 'ah', 'i-nlsp',
   'swipe', 'narp', 'mobile', 'tlsp', 'skip', 'ipv6-icmp', 'ipv6-nonxt',
   'ipv6-opts', '61', 'cftp', '63', 'sat-expak', 'kryptolan', 'rvd', 'ippc', '68',
   'sat-mon', 'visa', 'ipcv', 'cpnx', 'cphb', 'wsn', 'pvp', 'br-sat-mon', 'sun-nd',
   'wb-mon', 'wb-expak', 'iso-ip', 'vmtp', 'secure-vmtp', 'vines', 'ttp',
   'nsfnet-igp', 'dgp', 'tcf', 'eigrp', 'ospf', 'sprite-rpc', 'larp', 'mtp',
   'ax.25', 'ipip', 'micp', 'scc-sp', 'etherip', 'encap', '99', 'gmtp', 'ifmp',
   'pnni', 'pim', 'aris', 'scps', 'qnx', 'a/n', 'ipcomp', 'snp', 'compaq-peer',
   'ipx-in-ip', 'vrrp', 'pgm', '114', 'l2tp', 'ddx', 'iatp', 'stp', 'srp', 'uti',
   'smp', 'sm', 'ptp', 'isis', 'fire', 'crtp', 'crudp', 'sscopmce', 'iplt', 'sps',
   'pipe', 'sctp', 'fc', 'rsvp-e2e-ignore', 'mobility-header', 'udplite',
   'mpls-in-ip', 'manet', 'hip', 'shim6', 'wesp', 'rohc' ]
ipProtoNameToNum = {}
for proNum, proName in enumerate( ipProtoNameFromValue ):
   ipProtoNameToNum[ proName ] = proNum

alternateProtocolNames = {
   'ipv6-mh' : 135,
   'icmpv6' : 58
   }

def getProtocolNameFromValue( protocol ):
   try:
      return ipProtoNameFromValue[ protocol ]
   except IndexError:
      return str( protocol )

def getProtocolNumFromName( protocolName ):
   """Get the protocol number from the name, but
   do the extra work here to handle cases where
   we're passed the number so every caller doesn't
   need to test.
   """
   if protocolName in ipProtoNameToNum:
      return ipProtoNameToNum[ protocolName ]
   elif protocolName in alternateProtocolNames:
      return alternateProtocolNames[ protocolName ]
   elif isinstance( protocolName, int ):
      return protocolName
   elif isinstance( protocolName, str ):
      try:
         return int( protocolName )
      except ValueError:
         # the only alternative protocol spelling we support is hop-by-hop
         # which uses protocol number 0
         pass
   assert protocolName == 'hop-by-hop'
   return 0
