# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import Tac
from CliModel import _TacAttributeType
from CliModel import Model
from CliModel import Int

class MacAddress( _TacAttributeType ):
   """A MAC address."""
   tacType = Tac.Type( "Arnet::EthAddr" )
   realType = ( tacType, str )

   def _createTacValue( self, value ):
      mac = self.tacType()
      mac.stringValue = value
      return mac


class Ip4Address( _TacAttributeType ):
   """An IPv4 address."""
   tacType = Tac.Type( "Arnet::IpAddr" )
   realType = ( tacType, str )

   def _createTacValue( self, value ):
      ip = self.tacType()
      ip.stringValue = value
      return ip


class Ip6Address( _TacAttributeType ):
   """An IPv6 address."""
   tacType = Tac.Type( "Arnet::Ip6Addr" )
   realType = ( tacType, str )

   def _createTacValue( self, value ):
      ip = self.tacType()
      ip.stringValue = value
      return ip

class IpGenericAddress( _TacAttributeType ):
   """ An IPv4 or IPv6 address. """
   tacType = Tac.Type( "Arnet::IpGenAddr" )
   realType = ( tacType, str )

   def _createTacValue( self, value ):
      m = re.match( '[0-9a-fA-F:.]+', value )
      if m:
         # While BUG687591 was in place, many cli models were created
         # using an IpGenericAddress that actually contain more info
         # at the end, e.g., a prefix, or some flags, or even some
         # more addresses.  Until those types are all changed, we
         # replicate the behavior that BUG687591 had: namely,
         # ignore the garbage after the IP address.
         value = m.group( 0 )
      return self.tacType( value )

class IpGenericPrefix( _TacAttributeType ):
   """ An IPv4 or IPv6 address prefix. """
   tacType = Tac.Type( "Arnet::IpGenPrefix" )
   realType = ( tacType, str )

   def _createTacValue( self, value ):
      return self.tacType( value )

class IpGenericAddrAndMask( _TacAttributeType ):
   """ An IPv4 or IPv6 address and mask length. """
   tacType = Tac.Type( "Arnet::IpGenAddrWithMask" )
   realType = ( tacType, str )
   
   def _createTacValue( self, value ):
      return self.tacType( value )

class IpGenericAddrAndFullMask( _TacAttributeType ):
   """ An IPv4 or IPv6 address with netmask. """
   tacType = Tac.Type( "Arnet::IpGenAddrWithFullMask" )
   realType = ( tacType, str )

   def _createTacValue( self, value ):
      return self.tacType( value )


# Do not use IpAddr, but use the inherited types or Ip4Address instead
class IpAddr( Model ):
   ip = Ip4Address( help="Ip address" )
   
   def formatStr( self ):
      return self.ip


class IpGenericAddr( Model ):
   ip = IpGenericAddress( help="IPv4 or IPv6 address" )

   def formatStr( self ):
      return self.ip

class IpGenericAddrAndPort( IpGenericAddr ):
   port = Int( help="Port" )

   def formatStr( self ):
      portVal = str( self.port )
      return '%s:%s' % ( self.ip, portVal ) # pylint: disable=consider-using-f-string

class IpGenPrefix( Model ):
   ip = IpGenericPrefix( help="IPv4 or IPv6 address prefix (addr/n syntax)" )

   def formatStr( self ):
      return self.ip

class IpGenAddrWithMask( Model ):
   ip = IpGenericAddrAndMask( 
        help="IPv4 or IPv6 address and mask length (addr/n syntax)" )

   def formatStr( self ):
      return self.ip

class IpGenAddrWithFullMask( Model ):
   ip = IpGenericAddrAndFullMask(
            help="IPv4 or IPv6 address with netmask (addr/addr syntax)" )

   def formatStr( self ):
      return self.ip


class IpAddrAndPort( IpAddr ):
   port = Int( help="Port" )
   
   def formatStr( self ):
      portVal = str( self.port )
      return '%s:%s' % ( self.ip, portVal ) # pylint: disable=consider-using-f-string


class IpAddrAndMask( IpAddr ):
   mask = Int( help="Ip address mask len" )
   
   def formatStr( self ):
      stringValue = ""
      if self.mask > 0 and self.mask <= 32:
         maskVal = str( self.mask ) if self.mask else ''
         # pylint: disable-next=consider-using-f-string
         stringValue = '%s/%s' % ( self.ip, maskVal )
      else:
         stringValue = \
         Tac.Value( "Arnet::IpAddrWithFullMask", self.ip, self.mask).stringValue
      return stringValue

# Do not use Ip6Addr, but use the inherited types or Ip6Address instead
class Ip6Addr( Model ):
   ip = Ip6Address( help="Ipv6 address" )

   def formatStr( self ):
      return self.ip


class Ip6AddrAndMask( Ip6Addr ):
   mask = Int( help="Ipv6 address mask len", optional=True )
   fullMask = Ip6Address( help="Ipv6 address full mask, used only when mask" \
                          "cannot be represented as a length", optional=True )

   def formatStr( self ):
      # pylint: disable-next=superfluous-parens
      assert ( self.mask is None ^ self.fullMask is None )
      if self.mask is not None:
         # pylint: disable-next=consider-using-f-string,bad-string-format-type
         return '%s/%d' % ( self.ip, self.mask )
      else:
         stringValue = Tac.Value( "Arnet::Ip6AddrWithFullMask", 
                                  self.ip, self.fullMask ).stringValue
         return stringValue
