#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import (
   IpGenAddr,
   )
import Tac
import unittest

class ArpLibTypesFactory:
   '''
   Factory for instantiating Tac.Types

   A module can import the arpLibTypes instantiation instantiation below and use
   it to instantiate the types.
   '''
   @property
   @Tac.memoize
   def QuickRefreshTime( self ):
      return Tac.Type( 'Arp::QuickRefreshTime' )

   @property
   @Tac.memoize
   def QuickRefreshStatus( self ):
      return Tac.Type( 'Arp::QuickRefreshStatus' )

   @property
   @Tac.memoize
   def InputConfig( self ):
      return Tac.Type( 'Arp::InputConfig' )

   @property
   @Tac.memoize
   def InputConfigKey( self ):
      return Tac.Type( 'Arp::InputConfigKey' )

   @property
   @Tac.memoize
   def InputConfigEntry( self ):
      return Tac.Type( 'Arp::InputConfigEntry' )

   @property
   @Tac.memoize
   def InputStatusEntry( self ):
      return Tac.Type( 'Arp::InputStatusEntry' )

   @property
   @Tac.memoize
   def ArpSourceId( self ):
      return Tac.Type( 'Arp::ArpSource::ArpSourceId' )

   @property
   @Tac.memoize
   def ArpSource( self ):
      return Tac.Type( 'Arp::ArpSource' )

   @property
   @Tac.memoize
   def ArpType( self ):
      return Tac.Type( 'Arp::ArpType' )

   @property
   @Tac.memoize
   def ArpTypeId( self ):
      return Tac.Type( 'Arp::ArpType::ArpTypeId' )

arpLibTypes = ArpLibTypesFactory()

noRefresh = arpLibTypes.QuickRefreshTime.none()
immediateRefresh = arpLibTypes.QuickRefreshTime.immediate()
refreshCompleted = arpLibTypes.QuickRefreshStatus.completed()
refreshInvalid = arpLibTypes.QuickRefreshStatus.invalid()
# arpSourceIdFoo is a string of the sourceId enum value
# arpSourceFoo is the int value
# If an actual ArpSource type is needed, then construct it
# via the tacArpSourceFromId/tacArpSourceFromValue methods
arpSourceIdCli = arpLibTypes.ArpSourceId.cli
arpSourceCli = arpLibTypes.ArpSource.idToValue( arpSourceIdCli )
arpSourceIdNeighborResolution = arpLibTypes.ArpSourceId.neighborResolution
arpSourceNeighborResolution = arpLibTypes.ArpSource.idToValue(
      arpSourceIdNeighborResolution )
arpSourceIdDfwService = arpLibTypes.ArpSourceId.dfwService
arpSourceDfwService = arpLibTypes.ArpSource.idToValue( arpSourceIdDfwService )
arpSourceIdDmf = arpLibTypes.ArpSourceId.dmf
arpSourceDmf = arpLibTypes.ArpSource.idToValue( arpSourceIdDmf )
arpSourceIdDps = arpLibTypes.ArpSourceId.dps
arpSourceDps = arpLibTypes.ArpSource.idToValue( arpSourceIdDps )
arpSourceIdKernel = arpLibTypes.ArpSourceId.kernel
arpSourceKernel = arpLibTypes.ArpSource.idToValue( arpSourceIdKernel )
arpSourceIdVxlan = arpLibTypes.ArpSourceId.vxlan
arpSourceVxlan = arpLibTypes.ArpSource.idToValue( arpSourceIdVxlan )
arpSourceIdEvpn = arpLibTypes.ArpSourceId.evpn
arpSourceEvpn = arpLibTypes.ArpSource.idToValue( arpSourceIdEvpn )
arpSourceIdEvpnVxlan = arpLibTypes.ArpSourceId.evpnVxlan
arpSourceEvpnVxlan = arpLibTypes.ArpSource.idToValue( arpSourceIdEvpnVxlan )
arpSourceIdArpSuppression = arpLibTypes.ArpSourceId.arpSuppression
arpSourceArpSuppression = arpLibTypes.ArpSource.idToValue(
      arpSourceIdArpSuppression )
arpSourceIdFhrp = arpLibTypes.ArpSourceId.fhrp
arpSourceFhrp = arpLibTypes.ArpSource.idToValue( arpSourceIdFhrp )
arpSourceIdFhrpArfa = arpLibTypes.ArpSourceId.fhrpArfa
arpSourceFhrpArfa = arpLibTypes.ArpSource.idToValue( arpSourceIdFhrpArfa )
arpSourceIdNat = arpLibTypes.ArpSourceId.nat
arpSourceNat = arpLibTypes.ArpSource.idToValue( arpSourceIdNat )
arpSourceIdEosSdk = arpLibTypes.ArpSourceId.eosSdk
arpSourceEosSdk = arpLibTypes.ArpSource.idToValue( arpSourceIdEosSdk )
arpSourceIdProxy = arpLibTypes.ArpSourceId.proxy
arpSourceProxy = arpLibTypes.ArpSource.idToValue( arpSourceIdProxy )
permanent = arpLibTypes.ArpType.idToValue( arpLibTypes.ArpTypeId.permanent )
dynamic = arpLibTypes.ArpType.idToValue( arpLibTypes.ArpTypeId.dynamic )

# Can be used as a mixin with classes
class ArpLibTypesMixin:
   ArpSource = arpLibTypes.ArpSource
   InputConfig = arpLibTypes.InputConfig
   InputConfigKey = arpLibTypes.InputConfigKey
   InputConfigEntry = arpLibTypes.InputConfigEntry
   InputStatusEntry = arpLibTypes.InputStatusEntry

   @staticmethod
   def tacArpSourceFromValue( sourceValue ):
      '''
      Contructs an ArpSource from an int value usually from one of the
      arpSourceFoo globals declared above
      '''
      return ArpLibTypesMixin.ArpSource( sourceValue )

   @staticmethod
   def tacArpSourceFromId( sourceId ):
      '''
      Constructs an ArpSource from a string id usually from one of the
      arpSourceIdFoo globals declared above
      '''
      sourceValue = ArpLibTypesMixin.ArpSource.idToValue( sourceId )
      return ArpLibTypesMixin.tacArpSourceFromValue( sourceValue )

   @staticmethod
   def tacInputConfig( name ):
      return ArpLibTypesMixin.InputConfig( name )

   @staticmethod
   def tacInputConfigKey( ip, intfId ):
      '''
      Construct an InputConfigKey.
      @ip can be either a string or an IpGenAddr instance
      @intfId can be either a string or an Arnet::IntfId
              note: interface name strings should be capitalized
              otherwise the IntfId constructor will throw
              ( Ethernet1, Test1, Vlan1, Port-Channel1 )
      '''
      def toCap( x ):
         return x if not x else x[ 0 ].upper() + x[ 1 : ]
      addr = ip if not isinstance( ip, str ) else IpGenAddr( ip )
      intf = intfId if not isinstance( intfId, str ) else toCap( intfId )
      return ArpLibTypesMixin.InputConfigKey( addr, intf )

   @staticmethod
   def tacInputConfigEntry( ip, intfId, mac, source=arpSourceCli, arpType=permanent,
         quickRefresh=noRefresh, genId=0 ):
      '''
      Construct an InputConfigEntry.
      @ip can be either a string or an IpGenAddr instance
      @intfId can be either a string or an Arnet::IntfId
      @mac should be a string
      @source should be an arpSourceFoo value (an int type) from those declared above
         or an ArpSource
      @arpType should be either permanent or dynamic (int types) from those declared
         above
      @quickRefresh should be an int type between 0 and 65535 usually by passing in
         noRefresh or immediate as declared above
      @genId should be an int, if arpType is dynamic then it must not be 0
      '''
      configKey = ArpLibTypesMixin.tacInputConfigKey( ip, intfId )
      return ArpLibTypesMixin.InputConfigEntry( configKey, mac, source, arpType,
            quickRefresh, genId )

   @staticmethod
   def tacInputStatusEntry( ip, intfId, mac, source=arpSourceCli, arpType=permanent,
         quickRefresh=noRefresh, genId=0 ):
      '''
      Construct an InputStatusEntry.
      @ip can be either a string or an IpGenAddr instance
      @intfId can be either a string or an Arnet::IntfId
      @mac should be a string
      @source should be an arpSourceFoo value (an int type) from those declared above
         or an ArpSource
      @arpType should be either permanent or dynamic (int types) from those declared
         above
      @quickRefresh should be an int type between 0 and 65535 usually by passing in
         noRefresh or immediate as declared above
      @genId should be an int, if arpType is dynamic then it must not be 0
      '''
      addr = ip if not isinstance( ip, str ) else IpGenAddr( ip )
      return ArpLibTypesMixin.InputStatusEntry( addr, intfId, mac, source, arpType,
            quickRefresh, genId )

   @staticmethod
   def tacInputStatusEntryFromConfigEntry( configEntry ):
      '''
      Construct an InputStatusEntry.
      @configEntry is an InputConfigEntry
      '''
      return ArpLibTypesMixin.InputStatusEntry.fromConfigEntry( configEntry )

# Can be used with standalone functions
def tacArpSourceFromValue( sourceValue ):
   return ArpLibTypesMixin.tacArpSourceFromValue( sourceValue )

def tacArpSourceFromId( sourceId ):
   return ArpLibTypesMixin.tacArpSourceFromId( sourceId )

def tacInputConfig( name ):
   return ArpLibTypesMixin.tacInputConfig( name )

def tacInputConfigKey( ip, intfId ):
   return ArpLibTypesMixin.tacInputConfigKey( ip, intfId )

def tacInputConfigEntry( ip, intfId, mac, source=arpSourceCli, arpType=permanent,
      quickRefresh=noRefresh, genId=0 ):
   return ArpLibTypesMixin.tacInputConfigEntry( ip, intfId, mac, source, arpType,
         quickRefresh, genId )

def tacInputStatusEntry( ip, intfId, mac, source=arpSourceCli, arpType=permanent,
      quickRefresh=noRefresh, genId=0 ):
   return ArpLibTypesMixin.tacInputStatusEntry( ip, intfId, mac, source, arpType,
         quickRefresh, genId )

def tacInputStatusEntryFromConfigEntry( configEntry ):
   return ArpLibTypesMixin.tacInputStatusEntryFromConfigEntry( configEntry )

# When this is run as a btest it causes an assertion in
# IntfIdDesc::stringToIntfId because its not initialized properly.
# But it runs fine manually. So can use it manually to verify any
# changes, but don't add this to the BUILD.qb
class InputTypesTest( unittest.TestCase ):
   def testInputConfigKey( self ):
      '''
      Verify that tacInputConfigKey capitializes the intfId correctly
      so that the C++ constructor doesn't throw an error
      '''
      key = tacInputConfigKey( '2:1:5:a:b:c:d:3', 'test1' )
      self.assertEqual( key.af, 'ipv6' )
      self.assertEqual( key.addr.stringValue, '2:1:5:a:b:c:d:3' )
      self.assertEqual( key.intfId, 'Test1' )
      key = tacInputConfigKey( '1.1.1.1', 'Ethernet1' )
      self.assertEqual( key.af, 'ipv4' )
      self.assertEqual( key.addr.stringValue, '1.1.1.1' )
      self.assertEqual( key.intfId, 'Ethernet1' )
      # vlan should be capitalized
      key = tacInputConfigKey( '4:dead:beaf:bad:f00d::7', 'vlan2000' )
      self.assertEqual( key.af, 'ipv6' )
      self.assertEqual( key.addr.stringValue, '4:dead:beaf:bad:f00d::7' )
      self.assertEqual( key.intfId, 'Vlan2000' )
      # port-channel should have Port and Channel capitalized
      key = tacInputConfigKey( '1.2.3.4', 'port-Channel1234' )
      self.assertEqual( key.af, 'ipv4' )
      self.assertEqual( key.addr.stringValue, '1.2.3.4' )
      self.assertEqual( key.intfId, 'Port-Channel1234' )
      # UnconnectedEthernet
      key = tacInputConfigKey( '2:3:4:5:6:7:8:9', 'unconnectedEthernet3' )
      self.assertEqual( key.af, 'ipv6' )
      self.assertEqual( key.addr.stringValue, '2:3:4:5:6:7:8:9' )
      self.assertEqual( key.intfId, 'UnconnectedEthernet3' )

   def testInputConfigEntry( self ):
      '''
      Verify that tacInputConfigEntry calls c++ constructor correctly
      '''
      entry = tacInputConfigEntry( '1.1.1.1', 'Test101', 'b.e.d' )
      self.assertEqual( entry.af, 'ipv4' )
      self.assertEqual( entry.addr.stringValue, '1.1.1.1' )
      self.assertEqual( entry.intfId, 'Test101' )
      self.assertEqual( entry.ethAddr, '00:0b:00:0e:00:0d' )
      self.assertEqual( entry.source, arpSourceCli )
      self.assertEqual( entry.type, permanent )
      self.assertEqual( entry.refreshTime, noRefresh )
      self.assertEqual( entry.genId, 0 )

      entry = tacInputConfigEntry( '1:2:3:4:a:b:c:d', 'vlan100', '0.deed.f00d',
            arpSourceEvpnVxlan, dynamic, immediateRefresh, 98765 )
      self.assertEqual( entry.af, 'ipv6' )
      self.assertEqual( entry.addr.stringValue, '1:2:3:4:a:b:c:d' )
      self.assertEqual( entry.intfId, 'Vlan100' )
      self.assertEqual( entry.ethAddr, '00:00:de:ed:f0:0d' )
      self.assertEqual( entry.source, arpSourceEvpnVxlan )
      self.assertEqual( entry.type, dynamic )
      self.assertEqual( entry.refreshTime, immediateRefresh )
      self.assertEqual( entry.genId, 98765 )

   def verify( self ):
      pass

if __name__ == '__main__':
   unittest.main( verbosity=2 )
