# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
This PStore Plugin is used to save IntfPolicyManager proposed status, including
interface state (activeness) and interface mode (speed, lane, fec and duplex) during
ASU. These statuses are helpful to recover HW port configuration on boot.

There are two keys used in this Plugin, "intfStateMapping" and "ethIntfModeMapping",
where "intfStateMapping" encaps the intfStateDir and "ethIntfModeMapping" encaps
ethIntfModeDir in resourceManagerStatus. They have the corresponding format:
- "intfStateMapping"
  { sliceId :
       { subSliceId :
            { intfId :
                 { "swState" : swState,
                   "hwState" : hwState, }, }, }, }

- "ethIntfModeMapping"
  { sliceId :
       { subSliceId :
            { intfId :
                 { "speed" : speed,
                   "duplex" : duplex,
                   "laneCount" : laneCount,
                   "fec" : fec,
                   "anegMode" : anegMode }, }, }, }
"""

from TypeFuture import TacLazyType
import AsuPStore
import SwagTypes

FabricIntfId = TacLazyType( 'Arnet::FabricIntfId' )

class AlePhyIntfPolicyManagerStatusPStoreEventHandler(
   AsuPStore.PStoreEventHandler ):

   def __init__( self, resourceManagerStatusSliceDir,
                 swagMemberId ):
      self.resourceManagerStatusSliceDir_ = resourceManagerStatusSliceDir
      self.swagMemberId_ = swagMemberId
      super().__init__()

   def getSupportedKeys( self ):
      return [ 'intfStateMapping',
               'ethIntfModeMapping',
               'intfStateChangeGenMapping' ]

   def getKeys( self ):
      return [ 'intfStateMapping',
               'ethIntfModeMapping',
               'intfStateChangeGenMapping' ]

   def getIntfStateMapping( self ):
      intfStateMapping = {}
      # e.g. sliceId is "FixedSystem" and subSlices is "[ "Strata-FixedSystem" ]".
      for sliceId, subSlices in \
         self.resourceManagerStatusSliceDir_.items():
         sliceIdKey = SwagTypes.getSliceId( sliceId, self.swagMemberId_ )
         intfStateMapping[ sliceIdKey ] = {}
         for subSliceId, resourceManagerStatus in subSlices.items():
            perSubSliceMapping = {}
            subSliceIdKey = SwagTypes.getSlicifiedAgentName(
                  subSliceId, self.swagMemberId_ )
            intfStateMapping[ sliceIdKey ][ subSliceIdKey ] = perSubSliceMapping
            for intfId, intfState in \
               resourceManagerStatus.intfStateDir.intfState.items():
               state = intfState.state
               intfIdKey = SwagTypes.getIntfId( intfId, self.swagMemberId_ )
               perSubSliceMapping[ intfIdKey ] = \
                  { "swState" : state.swState, "hwState" : state.hwState }
      return intfStateMapping

   def getEthIntfModeMapping( self ):
      ethIntfModeMapping = {}
      # e.g. sliceId is "FixedSystem" and subSlices is "[ "Strata-FixedSystem" ]".
      for sliceId, subSlices in \
         self.resourceManagerStatusSliceDir_.items():
         sliceIdKey = SwagTypes.getSliceId( sliceId, self.swagMemberId_ )
         ethIntfModeMapping[ sliceIdKey ] = {}
         for subSliceId, resourceManagerStatus in subSlices.items():
            perSubSliceMapping = {}
            subSliceIdKey = SwagTypes.getSlicifiedAgentName(
                  subSliceId, self.swagMemberId_ )
            ethIntfModeMapping[ sliceIdKey ][ subSliceIdKey ] = perSubSliceMapping
            for intfId, ethIntfMode in \
               resourceManagerStatus.ethIntfModeDir.ethIntfMode.items():
               intfIdKey = SwagTypes.getIntfId( intfId, self.swagMemberId_ )
               perSubSliceMapping[ intfIdKey ] = \
                  { "speed" : ethIntfMode.speed, "duplex" : ethIntfMode.duplex,
                    "laneCount" : ethIntfMode.laneCount, "fec" : ethIntfMode.fec }
               if FabricIntfId.isFabricIntfId( intfId ):
                  # Right now only fabric intfs use anegMode
                  perSubSliceMapping[ intfIdKey ].update(
                    { "anegMode" : ethIntfMode.anegMode } )
      return ethIntfModeMapping

   def getIntfStateChangeGenMapping( self ):
      intfStateChangeGenMapping = {}
      for sliceId, subSlices in \
         self.resourceManagerStatusSliceDir_.items():
         sliceIdKey = SwagTypes.getSliceId( sliceId, self.swagMemberId_ )
         intfStateChangeGenMapping[ sliceIdKey ] = {}
         for subSliceId, resourceManagerStatus in subSlices.items():
            subSliceIdKey = SwagTypes.getSlicifiedAgentName(
                  subSliceId, self.swagMemberId_ )
            intfStateChangeGenMapping[ sliceIdKey ][ subSliceIdKey ] = {}
            intfStateChangeGenMapping[ sliceIdKey ][ subSliceIdKey ][ "id" ] = \
               resourceManagerStatus.intfStateChangeGen.id
            intfStateChangeGenMapping[ sliceIdKey ][ subSliceIdKey ][ "valid" ] = \
               resourceManagerStatus.intfStateChangeGen.valid
      return intfStateChangeGenMapping

   def save( self, pStoreIO ):
      intfStateMapping = self.getIntfStateMapping()
      pStoreIO.set( 'intfStateMapping', intfStateMapping )
      ethIntfModeMapping = self.getEthIntfModeMapping()
      pStoreIO.set( 'ethIntfModeMapping', ethIntfModeMapping )
      intfStateChangeGenMapping = self.getIntfStateChangeGenMapping()
      pStoreIO.set( 'intfStateChangeGenMapping', intfStateChangeGenMapping )

def Plugin( ctx ):
   featureName = 'AlePhyIntfPolicyManagerStatus'
   fabricFeatureName = 'AlePhyFabricIntfPolicyManagerStatus'

   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler(
         featureName, AlePhyIntfPolicyManagerStatusPStoreEventHandler(
            None, None ) )
      ctx.registerAsuPStoreEventHandler(
         fabricFeatureName,
         AlePhyIntfPolicyManagerStatusPStoreEventHandler(
            None, None ) )
      return
   
   entityManager = ctx.entityManager()
   mg = entityManager.mountGroup()
   resourceManagerStatusSliceDir = mg.mount(
      "interface/resources/status/slice", "Tac::Dir", "ri" )
   fabricResourceManagerStatusSliceDir = mg.mount(
      "interface/resources/status/fabric/slice", "Tac::Dir", "ri" )
   swagStatus = mg.mount( 'swag/status', 'Swag::SwagStatus', 'r' )

   def registerEventHandlers():
      ctx.registerAsuPStoreEventHandler(
         featureName,
         AlePhyIntfPolicyManagerStatusPStoreEventHandler(
            resourceManagerStatusSliceDir,
            swagStatus.candidateMemberId ) )
      ctx.registerAsuPStoreEventHandler(
         fabricFeatureName,
         AlePhyIntfPolicyManagerStatusPStoreEventHandler(
            fabricResourceManagerStatusSliceDir,
            swagStatus.candidateMemberId ) )

   ctx.mountsComplete( mg, 'AlePhyIntfPolicyManagerStatusAsuPStore',
                       registerEventHandlers )
