#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import time

import AsuPStore
import Cell
from CliPlugin.AsuPStoreModel import ReloadHitlessWarningReason, \
   modifyValueOfWarningReasonsDict

def isPreviousReloadASU():
   '''Check if the previous reload was an ASU reload, verifying it by
   checking kexec params (i.e. /proc/cmdline) and see if 'asu_hitless'
   is in there
   '''
   kexecCmdLinePath = '/proc/cmdline'
   ssuPattern = 'arista.asu_hitless'
   if os.path.isfile( kexecCmdLinePath ):
      with open( kexecCmdLinePath ) as f:
         for line in f:
            m = re.search( ssuPattern, line )
            if m:
               return True
   return False

class AsuUptimeCheckPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   ''' Check if the uptime is sufficient since previous reload.
   '''
   def __init__( self, reloadCause ):
      AsuPStore.PStoreEventHandler.__init__( self )
      self.reloadCause_ = reloadCause

   def hitlessReloadSupported( self ):
      ''' Check the content of /proc/cmdline to see if the previous reload
      was an ASU reload. If it was, check the timestamp to make sure we have
      waited a reasonable amount of time before performing the next ASU reload.
      A warning message will be printed out when showing reload command within
      20 minutes after the previous ASU reload time.
      '''
      warningList, blockingList = [], []
      if isPreviousReloadASU():
         reloadCause = self.reloadCause_
         if reloadCause:
            reloadTime = reloadCause.timestamp
            currentTime = time.time()
            duration = int( currentTime - reloadTime )
            if duration < 1200:
               waitTime = int( ( 1200 - duration ) / 60 ) + 1
               reasonStr = ( 'Insufficient uptime since the previous reload: '
                             f'it is recommended to wait another {waitTime}'
                             ' minutes for the system to be ready for the'
                             ' next SSU.' )
               modifyValueOfWarningReasonsDict( "insufficientUptime", reasonStr )
               warningList.append( ReloadHitlessWarningReason(
                  reason="insufficientUptime" ) )
      return ( warningList, blockingList )

def Plugin( ctx ):
   featureName = 'Asu'
   if ctx.opcode() == 'CheckHitlessReloadSupported':
      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup()
      reloadCause = mg.mount( Cell.path( 'sys/reload/cause' ),
            'System::ReloadCause', 'ri' )
      mg.close( blocking=True )
      ctx.registerAsuPStoreEventHandler(
            featureName,
            AsuUptimeCheckPStoreEventHandler( reloadCause ) )
   else:
      ctx.registerAsuPStoreEventHandler(
            featureName,
            AsuUptimeCheckPStoreEventHandler( None ) )
