# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# N.B. If you edit this file you must regenerate and replace
# the associated AsuPatch patch RPM. For more information see
# /src/AsuPatch/packages/README.

import Tracing
import AsuPStore
import Tac
from Toggles.Dot1xToggleLib import toggleDot1xCoaPortShutdownEnabled
from Toggles.MvrpToggleLib import toggleDot1xMvrpVsaEnabled
__defaultTraceHandle__ = Tracing.Handle( "Dot1xAsu" )
t0 = Tracing.trace0
t1 = Tracing.trace1

class Dot1xPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   def __init__( self, stores ):
      self.stores = stores
      self.funcs = { 'status' : self.storeStatus,
                     'globalStatus': self.storeGlobalStatus,
                     'hostTable' : self.storeDot1xHostTable,
                     'ebraHostTable' : self.storeEbraHostTable,
                     'brConfig' : self.storeBrConfig,
                     'sessionIdStore' : self.storeSessionIdStore,
                     'allowedVlans' : self.storeAllowedVlans,
                     'vlanIdSet' : self.storeVlanIdSet,
                     'intfForwardingModel': self.storeIntfForwardingModel,
                     'blockedMacTable': self.storeBlockedMacTable,
                     }
      super().__init__()
   
   def save( self, pStoreIO ):
      # Save TAC collection as dictionary in string format
      for k in self.getKeys():
         t0( 'saving', k )
         pyCol = self.funcs[k]()
         pStoreIO.set( k, pyCol )
   
   def getKeys( self ):
      return [ 'status', 'hostTable', 'ebraHostTable', 'brConfig',
               'sessionIdStore', 'allowedVlans', 'vlanIdSet',
               'intfForwardingModel', 'blockedMacTable', 'globalStatus' ]

   def getSupportedKeys( self ):
      return self.getKeys()

   def hitlessReloadSupported( self ):
      t0( self.__class__.__name__ )
      return ( [], [] )

   def storeGlobalStatus( self ):
      t0( self.__class__.__name__, "store Dot1x status except dot1xIntfStatus" )
      status = self.stores[ 'status' ]
      pyCol = {}
      pyCol[ 'captivePortal' ] = status.captivePortal
      dynAclNameCol = pyCol.setdefault( 'dynamicAclName', {} )
      for key, value in status.dynamicAclName.items():
         dynAclNameCol[ key ] = value
      if toggleDot1xCoaPortShutdownEnabled():
         portShutdownForCoaCol = pyCol.setdefault( 'portShutdownForCoa', {} )
         intfRecoveryTimeStampCol = portShutdownForCoaCol.setdefault(
            'intfRecoveryTimeStamp', {} )
         # check if portShutdownForCoa has been initialized. currently if is
         # done in dot1xSm's initializer, checking if it is better to do in
         # doMountsComplete just after we init status.
         if status.portShutdownForCoa is None:
            return pyCol
         for key, value in status.portShutdownForCoa.intfRecoveryTimeStamp.items():
            if value == Tac.endOfTime:
               intfRecoveryTimeStampCol[ key ] = "inf"
            else:
               intfRecoveryTimeStampCol[ key ] = str( value )
      return pyCol

   def storeStatus( self ):
      t0( self.__class__.__name__, "store Dot1x interface Status." )
      status = self.stores[ 'status' ]
      pyCol = {}
      for intf, intfStatus in status.dot1xIntfStatus.items():
         # Saving all the relevant attributes in Dot1xIntfStatus entity
         # Refer to AID5417 on more details on why do we need to store only the
         # attributes mentioned below:
         info = pyCol.setdefault( intf, {} )
         info[ 'portAuthState' ] = intfStatus.portAuthState
         info[ 'dynamicVlanId' ] = intfStatus.dynamicVlanId
         info[ 'hostMode' ] = intfStatus.hostMode
         info[ 'mbvaEnabled' ] = intfStatus.mbvaEnabled
         info[ 'mbaConfigured' ] = intfStatus.mbaConfigured
         info[ 'reauthPeriod' ] = intfStatus.reauthPeriod
         info[ 'reauthBehavior' ] = intfStatus.reauthBehavior
         info[ 'lastManualReauthRequestTime' ] = \
               intfStatus.lastManualReauthRequestTime
         info[ 'aclMode' ] = intfStatus.aclMode
         info[ 'aclIpv4Ipv6Required' ] = intfStatus.aclIpv4Ipv6Required
         webAuthMacCol = info.setdefault( 'webAuthMac', {} )
         for mac in intfStatus.webAuthMac:
            webAuthMacCol[ mac ] = True

         trunkDynamicAllowedVlans = info.setdefault( 'trunkDynamicAllowedVlans',
                                                     {} )
         for vlan in intfStatus.trunkDynamicAllowedVlan:
            trunkDynamicAllowedVlans[ vlan ] = True
         
         authSupplicantCol = info.setdefault( 'authSupplicant', {} )
         for mac in intfStatus.authSupplicant:
            authSupplicantCol[ mac ] = True
         
         dot1xAuthSupplicantCol = info.setdefault( 'dot1xAuthSupplicant', {} )
         for mac in intfStatus.dot1xAuthSupplicant:
            dot1xAuthSupplicantCol[ mac ] = True
         
         info[ 'aclName' ] = intfStatus.aclName
         aclUserCol = info.setdefault( 'aclUser', {} )
         for mac in intfStatus.aclUser:
            aclUserCol[ mac ] = True

         info[ "aaaUnresponsiveApplyCachedResults" ] = \
               intfStatus.aaaUnresponsiveApplyCachedResults
         info[ "aaaUnresponsivePhoneApplyCachedResults" ] = \
               intfStatus.aaaUnresponsivePhoneApplyCachedResults
         info[ "aaaUnresponsivePhoneAllow" ] = \
               intfStatus.aaaUnresponsivePhoneAllow
         info[ "aaaUnresponsiveAcl" ] = intfStatus.aaaUnresponsiveAcl
         info[ "aaaUnresponsiveEapResponse" ] = \
               intfStatus.aaaUnresponsiveEapResponse
         allowCol = info.setdefault( "aaaUnresponsiveTrafficAllow", {} )
         allowCol[ "value" ] = intfStatus.aaaUnresponsiveTrafficAllow.value
         allowCol[ "isSet" ] = intfStatus.aaaUnresponsiveTrafficAllow.isSet
         allowCol[ "vlanId" ] = intfStatus.aaaUnresponsiveTrafficAllow.vlanId

         supplicantCol = info.setdefault( 'supplicant', {} )
         for mac, supplicant in intfStatus.supplicant.items():
            suppInfo = supplicantCol.setdefault( mac, {} )
            suppInfo[ 'authStage' ] = supplicant.authStage
            suppInfo[ 'identity' ] = supplicant.identity
            suppInfo[ 'vlan' ]  = supplicant.vlan
            suppInfo[ 'vTag' ] = supplicant.vTag
            suppInfo[ 'taggedSupplicant' ] = supplicant.taggedSupplicant
            suppInfo[ 'supplicantTimeoutCounter' ] = \
                  supplicant.supplicantTimeoutCounter
            suppInfo[ 'supplicantAuthFailCounter' ] = \
                  supplicant.supplicantAuthFailCounter
            suppInfo[ 'supplicantDot1xCapable' ] = supplicant.supplicantDot1xCapable
            # Tac.endOfTime (inf) value is not corrctly handled by json c parser.
            if supplicant.lastAuthSuccessTime != Tac.endOfTime:
               suppInfo[ 'lastAuthSuccessTime' ] = supplicant.lastAuthSuccessTime
            suppInfo[ 'lastActivityTime' ] = supplicant.lastActivityTime
            suppInfo[ 'authSuccessTime' ] = supplicant.authSuccessTime
            if supplicant.lastReauthTimerArmed != Tac.endOfTime:
               suppInfo[ 'lastReauthTimerArmed' ] = supplicant.lastReauthTimerArmed
            suppInfo[ 'sessionTimeout' ] = supplicant.sessionTimeout
            suppInfo[ 'reauthTimeoutSeen' ] = supplicant.reauthTimeoutSeen
            suppInfo[ 'authenticating' ] = supplicant.authenticating
            suppInfo[ 'mbaSupplicant' ] = supplicant.mbaSupplicant
            suppInfo[ 'inAuthFailVlan' ] = supplicant.inAuthFailVlan
            suppInfo[ 'inGuestVlan' ] = supplicant.inGuestVlan
            suppInfo[ 'forceAuthPhoneApplied' ] = supplicant.forceAuthPhoneApplied
            suppInfo[ 'deviceType' ] = supplicant.deviceType
            suppInfo[ 'inDefaultPhoneVlan' ] = supplicant.inDefaultPhoneVlan
            suppInfo[ 'inAaaUnresponsiveVlanNative' ] = \
               supplicant.inAaaUnresponsiveVlanNative
            suppInfo[ 'inAaaUnresponsiveVlanSpecific' ] = \
               supplicant.inAaaUnresponsiveVlanSpecific
            suppInfo[ 'aclName' ] = supplicant.aclName
            suppInfo[ 'nasFilterRule' ] = supplicant.nasFilterRule
            suppInfo[ 'webAuth' ] = supplicant.webAuth
            suppInfo[ 'captivePortal' ] = supplicant.captivePortal
            suppInfo[ 'captivePortalSource' ] = supplicant.captivePortalSource
            # EapKey related info
            suppInfo[ 'dot1xType' ] = supplicant.eapKey.dot1xType
            suppInfo[ 'masterSessionKey' ] = supplicant.eapKey.masterSessionKey
            suppInfo[ 'eapSessionId' ] = supplicant.eapKey.eapSessionId
            suppInfo[ 'peerMacAddr' ] = supplicant.eapKey.peerMacAddr
            suppInfo[ 'myMacAddr' ] = supplicant.eapKey.myMacAddr
            if supplicant.cacheUpdationTime != Tac.endOfTime:
               suppInfo[ 'cacheUpdationTime' ] = supplicant.cacheUpdationTime
            suppInfo[ 'sessionCached' ] = supplicant.sessionCached
            if toggleDot1xMvrpVsaEnabled():
               suppInfo[ 'mvrpVsa' ] = supplicant.mvrpVsa

         mbaAuthInfoCol = info.setdefault( 'mbaAuthInfo', {} )
         for mac, mbaAuthInfo in intfStatus.mbaAuthInfo.items():
            mbaAuthInfoFeild = mbaAuthInfoCol.setdefault( mac, {} )
            mbaAuthInfoFeild[ 'mbaActivityStartTime' ] =\
                  mbaAuthInfo.mbaActivityStartTime
            mbaAuthInfoFeild[ 'mbaAuthAllowed' ] = mbaAuthInfo.mbaAuthAllowed

      return pyCol

   def storeHostTable( self, hostTable ):
      pyCol = {}
      for mac, hostEntry in hostTable.hostEntry.items():
         info = pyCol.setdefault( mac, {} )
         info[ 'vlanId' ] = hostEntry.vlanId
         info[ 'intfId' ] = hostEntry.intfId
         info[ 'mbaHost' ] = hostEntry.mbaHost
         info[ 'secureMac' ] = hostEntry.secureMac
         info[ 'cos' ] = hostEntry.cos
         info[ 'macBasedVlan' ] = hostEntry.macBasedVlan

      return pyCol
   
   def storeDot1xHostTable( self ):
      t0( self.__class__.__name__, "store Dot1x HostTable." )
      dot1xHostTable = self.stores[ "dot1xHostTable" ]
      return self.storeHostTable( dot1xHostTable )

   def storeEbraHostTable( self ):
      t0( self.__class__.__name__, "store Ebra HostTable." )
      ebraHostTable = self.stores[ "ebraHostTable" ]
      return self.storeHostTable( ebraHostTable )

   def storeSessionIdStore( self ): 
      t0( self.__class__.__name__, "store Dot1x SessionId store." )
      sessionIdStore = self.stores[ 'sessionIdStore' ]
      pyCol = {}
      pyCol[ 'acctSessionUniqueId' ] = sessionIdStore.acctSessionUniqueId
      acctSessionIdCol = pyCol.setdefault( 'acctSessionId', {} )
      for key, value in sessionIdStore.acctSessionId.items():
         acctSessionIdCol[ key ] = value
      return pyCol
   
   def storeBrConfig( self ):
      t0( self.__class__.__name__, "store Dot1x BridgingInput config." )
      brConfig = self.stores[ 'brConfig' ]
      pyCol = {}
      pyCol[ 'switchIntfConfigPriority' ] = brConfig.switchIntfConfigPriority
      pyCol[ 'nonDefaultVal' ] = brConfig.nonDefaultVal
      switchIntfConfigCol = pyCol.setdefault( 'switchIntfConfig', {} )
      for intfId, switchIntfConfig in brConfig.switchIntfConfig.items():
         switchIntfConfigInfo = switchIntfConfigCol.setdefault( intfId, {} )
         switchIntfConfigInfo[ 'switchportMode' ] = switchIntfConfig.switchportMode
         switchIntfConfigInfo[ 'accessVlan' ] = switchIntfConfig.accessVlan
         switchIntfConfigInfo[ 'trunkNativeVlan' ] = \
               switchIntfConfig.trunkNativeVlan
         switchIntfConfigInfo[ 'trunkAllowedVlans' ] = \
               switchIntfConfig.trunkAllowedVlans
      return pyCol

   def storeAllowedVlans( self ):
      t0( self.__class__.__name__, "store Dot1x input dynamic allowed vlans." )
      allowedVlans = self.stores[ 'allowedVlans' ]
      pyCol = {}
      for intf, vlans in allowedVlans.vlans.items():
         pyCol[ intf ] = vlans
      return pyCol
   
   def storeVlanIdSet( self ):
      t0( self.__class__.__name__, "store Dot1x input dynamic vlan set." )
      vlanIdSet = self.stores[ 'vlanIdSet' ]
      pyCol = {}
      pyCol[ 'vlans' ] = vlanIdSet.vlans
      return pyCol
   
   def storeIntfForwardingModel( self ):
      t0( self.__class__.__name__, "store Dot1x interface forwarding model." )
      intfForwardingModel = self.stores[ 'intfForwardingModel' ]
      pyCol = {}
      for intf, forwardingModelInput in \
            intfForwardingModel.forwardingModelInput.items():
         pyCol[ intf ] = forwardingModelInput
      return pyCol

   def storeBlockedMacTable( self ):
      t0( self.__class__.__name__, "store Dot1x blocked MAC table." )
      blockedMacTable = self.stores[ 'blockedMacTable' ]
      pyCol = {}
      for mac, blockedMacEntry in blockedMacTable.blockedMacEntry.items():
         info = pyCol.setdefault( mac, {} )
         for intf, vlan in blockedMacEntry.intfVlan.items():
            info[ intf ] = vlan
      return pyCol

def Plugin( ctx ):
   featureName = 'Dot1x'

   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler( featureName,
             Dot1xPStoreEventHandler( None ) )
      return

   entityManager = ctx.entityManager()
   mg = entityManager.mountGroup()
   stores = {}

   status = mg.mount( 'dot1x/status', 'Dot1x::Status', 'r' )
   dot1xHostTable = mg.mount( 'bridging/input/hostTable/Dot1x', 'Dot1x::HostTable',
                              'r' )
   ebraHostTable = mg.mount( 'bridging/input/hostTable/Ebra', 'Dot1x::HostTable',
                              'r' )
   brConfig = mg.mount( 'bridging/input/config/dot1x',
                        'Bridging::Input::Config', 'r' )
   sessionIdStore = mg.mount( 'dot1x/sessionIdStore',
                              'Dot1x::AaaSessionIdStore', 'r' )
   allowedVlans = mg.mount( 'bridging/input/dynvlan/allowedvlan/dot1x',
                            'Bridging::Input::AllowedVlans', 'r' )
   vlanIdSet = mg.mount( 'bridging/input/dynvlan/vlan/dot1x',
                         'Bridging::Input::VlanIdSet', 'r' )
   intfForwardingModel = mg.mount( 'interface/input/forwardingmodel/dot1x',
                                   'Interface::IntfForwardingModelInputDir', 'r' )
   blockedMacTable = mg.mount( "dot1x/blockedMacTable",
                               "Dot1x::BlockedMacTable", "r" )
   mg.close( blocking=True )

   # Stuff we actually store for ASU2
   stores[ 'status' ] = status
   stores[ 'dot1xHostTable' ] = dot1xHostTable
   stores[ 'ebraHostTable' ] = ebraHostTable
   stores[ 'brConfig' ] = brConfig
   stores[ 'sessionIdStore' ] = sessionIdStore
   stores[ 'allowedVlans' ] = allowedVlans
   stores[ 'vlanIdSet' ] = vlanIdSet
   stores[ 'intfForwardingModel' ] = intfForwardingModel
   stores[ 'blockedMacTable' ] = blockedMacTable
   
   ctx.registerAsuPStoreEventHandler( featureName,
                                      Dot1xPStoreEventHandler( stores ) )
