# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Tracing
import AsuPStore
from CliPlugin.AsuPStoreModel import ReloadHitlessWarningReason

__defaultTraceHandle__ = Tracing.Handle( "Ospf3Asu" )
t0 = Tracing.trace0

class Ospf3AsuPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   '''We should warn on ASU2 hitless reload if there is at least one OSPF3 instance
   enabled on switch but not configured for graceful restart.  Apis for store and
   restore are supported in order to save the grace time on a per vrf per AF basis.
   '''
   def __init__( self, ospf3Config, ospf3Status ):
      self.ospf3Config = ospf3Config
      self.ospf3Status = ospf3Status
      AsuPStore.PStoreEventHandler.__init__( self )

   def save( self, pStoreIO ):
      t0( 'Saving ospf3 asu state' )
      reloadOspf3Shutdown = {}
      for vrfName in self.ospf3Status:
         for inst in self.ospf3Status[ vrfName ].instanceStatus:
            reloadOspf3Shutdown[ vrfName ] = {}
            instanceStatus = self.ospf3Status[ vrfName ].instanceStatus[ inst ]
            asuGraceTime = instanceStatus.asuGraceTime
            t0( 'Saving ospf3 asu grace time for vrf %s inst %s: (%s).' %
                ( vrfName, inst, asuGraceTime ) )
            reloadOspf3Shutdown[ vrfName ][ str(inst) ] = ( asuGraceTime )
      pStoreIO.set( 'reloadOspf3Shutdown', reloadOspf3Shutdown )

   def getKeys( self ):
      return [ 'reloadOspf3Shutdown' ]

   def getSupportedKeys( self ):
      return [ 'reloadOspf3Shutdown' ]

   def hitlessReloadSupported( self ):   
      warningList, blockingList = None, None
      for vrf in self.ospf3Config.vrfConfig:
         vrfConfig = self.ospf3Config.vrfConfig[ vrf ]
         # If graceful-restart is globally configured under "router ospfv3"
         # we don't need to check for every AF
         if 'allAf' in vrfConfig.instanceConfig and \
            vrfConfig.instanceConfig[ 'allAf' ].enable and \
            vrfConfig.instanceConfig[ 'allAf' ].gracefulRestart:
            break
         # If there is no AF config and graceful-restart not configured
         # under router ospfv3 issue the warning
         if all( af not in vrfConfig.instanceConfig
                 for af in ( 'afIPv6', 'afIPv4' ) ):
            t0( 'graceful-restart not appropriately configured for vrf %s' % vrf )
            warningList = [ ReloadHitlessWarningReason( reason='ospf3NoGrace' ) ]
            break
         for i in vrfConfig.instanceConfig:
            # Check graceful-restart config for the addr families configured
            # as the global and per AF config for "graceful-restart"
            # doesn't coexist.
            if ( i != 'allAf' ) and vrfConfig.instanceConfig[ i ].enable and \
               not vrfConfig.instanceConfig[ i ].gracefulRestart:
               t0( 'graceful-restart not appropriately configured for vrf %s'
                   'and address family %s' % ( vrf, i ) )
               warningList = [ ReloadHitlessWarningReason( reason='ospf3NoGrace' ) ]
               break
         if warningList:
            break
      return ( warningList, blockingList )

def Plugin( ctx ):
   featureName = 'Ospf3'
   t0( ctx.opcode() )

   if ctx.opcode() == 'CheckHitlessReloadSupported':
      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup()
      ospf3Config = mg.mount( 'routing6/ospf3/config',
                              'Routing6::Ospf3::Config', 'r' )
      mg.close( blocking=True )
      ctx.registerAsuPStoreEventHandler( featureName,
                                         Ospf3AsuPStoreEventHandler( ospf3Config,
                                                                     None ) )
   elif ctx.opcode() == 'Store':
      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup()
      ospf3Status = mg.mount( 'routing6/ospf3/status', 'Tac::Dir', 'ri' )
      mg.close( blocking=True )
      ctx.registerAsuPStoreEventHandler( featureName,
                                         Ospf3AsuPStoreEventHandler( None,
                                                                     ospf3Status ) )
   else:
      ctx.registerAsuPStoreEventHandler( featureName,
                                         Ospf3AsuPStoreEventHandler( None,
                                                                     None ) )

