# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Tracing
import AsuPStore
from CliPlugin.AsuPStoreModel import ReloadHitlessWarningReason

__defaultTraceHandle__ = Tracing.Handle( "OspfAsu" )
t0 = Tracing.trace0

class OspfAsuPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   '''We should warn on ASU2 hitless reload if there is at least one OSPF instance
   enabled on switch but not configured for graceful restart.  Apis for store and
   restore are supported in order to save the real and montonic times on a per
   vrf basis.
   '''
   def __init__( self, ospfConfig, ospfStatus):
      self.ospfConfig = ospfConfig
      self.ospfStatus = ospfStatus
      AsuPStore.PStoreEventHandler.__init__( self )

   def save( self, pStoreIO ):
      reloadOspfShutdown = {}
      for vrfName in self.ospfStatus:
         for i in self.ospfStatus[vrfName].instanceStatus.members():
            asuMonoTime = self.ospfStatus[vrfName].instanceStatus[i].asuMonoTime
            asuGraceTime = self.ospfStatus[vrfName].instanceStatus[i].asuGraceTime
            t0( 'Saving ospf asu shutdown times %s (%s, %s).' %
                ( i,
                  self.ospfStatus[vrfName].instanceStatus[i].asuMonoTime,
                  self.ospfStatus[vrfName].instanceStatus[i].asuGraceTime ) )
            reloadOspfShutdown[i] = (asuMonoTime, asuGraceTime)
      pStoreIO.set( 'reloadOspfShutdown', reloadOspfShutdown )

   def getKeys( self ):
      return [ 'reloadOspfShutdown' ]

   def getSupportedKeys( self ):
      return [ 'reloadOspfShutdown' ]


   def hitlessReloadSupported( self ):   
      warningList, blockingList = None, None
      for i in self.ospfConfig.instanceConfig:
         if (self.ospfConfig.instanceConfig[ i ].enable and not \
             self.ospfConfig.instanceConfig[ i ].gracefulRestart) :
            warningList = [ ReloadHitlessWarningReason( reason='ospfNoGrace' ) ]
            break
      return ( warningList, blockingList )

def Plugin( ctx ):
   featureName = 'Ospf'

   if ctx.opcode() == 'CheckHitlessReloadSupported':
      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup()
      ospfConfig = mg.mount( 'routing/ospf/config', 'Routing::Ospf::Config', 'r' )
      mg.close( blocking=True )
      ctx.registerAsuPStoreEventHandler( featureName,
                                         OspfAsuPStoreEventHandler( ospfConfig,
                                                                    None ) )
   elif ctx.opcode() == 'Store':
      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup()
      ospfStatus = mg.mount( 'routing/ospf/status', 'Tac::Dir', 'ri' )
      mg.close( blocking=True )
      ctx.registerAsuPStoreEventHandler( featureName,
                                         OspfAsuPStoreEventHandler( None,
                                                                    ospfStatus ) )
   else:
      ctx.registerAsuPStoreEventHandler( featureName,
                                         OspfAsuPStoreEventHandler( None,
                                                                    None ) )
