#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import sys
import imp
import getopt
import Tac
import syslog
import os
import datetime
import re
import six
try:
   from AsuPatchRetiredDb import retiredDb
except ImportError:
   retiredDb = []

#
# Please refer to aid/4596 before making changes here.
# We lookup the filesystem to find the exact files in AsuPatchPkgs/.
# Also refer to AsuPatchRetiredDb::retiredDb list to determine if a patch is
# applicable.
#
releaseDb = {
   # We need AsuPatchArp to write /mnt/flash/arpAsuDb when upgrading
   # to an ASU3 supported release from a non-ASU3 supported release.
   #
   # N.B. As of 29-Oct-2018 this is the earliest release that supports
   # auto-patch. Anything prior to this release will require an uberpatch
   '4.18.8' : {
      'preInstall' : {},
      'package' : {
         'all' : [ 'AleMroute_S4_18_T4_22.i686.rpm',
                   'AclAgent_S4_18_8_T4_21_3.i686.rpm' ]
      },
      'postInstall' : {
         'all' : [ 'AsuPatchArp_S4_18_1_T4_22.py',
                   'AleMroute_S4_18_T4_22.py',
                   'AclAgent_S4_18_8_T4_21_3.py',
                   'KernelCmdlineFilter_S4_18_T4_31_0.py' ],
         'Strata' : [ 'StrataLacpPkt_S4_18_T4_21_3.py' ]
      }
   },
   '4.19.8' : {
      'preInstall' : {},
      'package' : {
         'all' : [ 'AleMroute_S4_18_T4_22.i686.rpm',
                   'AclAgent_S4_18_8_T4_21_3.i686.rpm' ]
      },
      'postInstall' : {
         'all' : [ 'AsuPatchArp_S4_18_1_T4_22.py',
                   'AsuAllowReEcmp_S4_19_8_T4_21_3.py',
                   'AleMroute_S4_18_T4_22.py',
                   'AclAgent_S4_18_8_T4_21_3.py',
                   'KernelCmdlineFilter_S4_18_T4_31_0.py' ],
         'Strata' : [ 'StrataLacpPkt_S4_18_T4_21_3.py' ]
      }
   },
   # ASU FPGA auto upgrade begins with bloomington-rel ( 4.20.5 ),
   # so AsuCli's AsuSkipFpgaUpgrade patch starts here
   '4.20.5' : {
      'preInstall' : {},
      'package' : {
         'all' : [ 'AleMroute_S4_18_T4_22.i686.rpm',
                   'AclAgent_S4_18_8_T4_21_3.i686.rpm' ]
      },
      'postInstall' : {
         'all' : [ 'AsuPatchArp_S4_18_1_T4_22.py',
                   'AsuAllowReEcmp_S4_19_8_T4_21_3.py',
                   'AleMroute_S4_18_T4_22.py',
                   'AclAgent_S4_18_8_T4_21_3.py',
                   'AsuCli_S4_20_5_T4_26_2.py',
                   'KernelCmdlineFilter_S4_18_T4_31_0.py' ],
         'Strata' : [ 'StrataLacpPkt_S4_18_T4_21_3.py' ]
      }
   },
   '4.20.7' : {
      # StrataL3 AsuPatch PStorePlugin included here as Vxlan support
      # added in 4.20.5.
      # StrataLogicalPort PStorePlugin included here for hitless
      # upgrade to releases with DLP. Only needed for TH2 which is
      # first supported in 4.20.3
      'preInstall' : {},
      'package' : {
         'all' : [ 'AleMroute_S4_18_T4_22.i686.rpm',
                   'AclAgent_S4_18_8_T4_21_3.i686.rpm' ],
         'Strata' : [ 'StrataL3_S4_20_T4_22.i686.rpm',
                      'Strata_S4_20_T4_22.i686.rpm' ]
      },
      'postInstall' : {
         'all' : [ 'AsuPatchArp_S4_18_1_T4_22.py',
                   'FpgaCliFix_S4_20_T4_21_3.py',
                   'AsuAllowReEcmp_S4_19_8_T4_21_3.py',
                   'AleMroute_S4_18_T4_22.py',
                   'AclAgent_S4_18_8_T4_21_3.py',
                   'AsuCli_S4_20_5_T4_26_2.py',
                   'KernelCmdlineFilter_S4_18_T4_31_0.py' ],
         'Strata' : [ 'StrataL3_S4_20_T4_22.py',
                      'Strata_S4_20_T4_22.py',
                      'StrataLacpPkt_S4_18_T4_21_3.py' ]
      }
   },
   '4.21.3' : {
      'preInstall' : {
      },
      'package' : {
         'all' : [ 'AclAgent_S4_18_8_T4_21_3.i686.rpm' ],
         'Strata' : [ 'Strata_S4_20_T4_22.i686.rpm' ]
      },
      'postInstall' : {
         'all' : [ 'AclAgent_S4_18_8_T4_21_3.py',
                   'AsuCli_S4_20_5_T4_26_2.py',
                   'KernelCmdlineFilter_S4_18_T4_31_0.py' ],
         'Strata' : [ 'Strata_S4_20_T4_22.py',
                      'StrataDmaShut_S4_4_21_3_T_4_22.py' ]
      }
   },
   '4.21.5' : {
      # Dot1x ASUPatch PStore plugin added here to support upgrade from
      # delhi.B1-rel onwards to eugene-rel. This patch is not applicable
      # for releases prior to delhi.B1-rel
      'preInstall' : {
      },
      'package' : {
         'all' : [ 'Dot1x_S4_21_5T4_22.i686.rpm',
                   'AclAgent_S4_18_8_T4_21_3.i686.rpm' ],
         'Strata' : [ 'Strata_S4_20_T4_22.i686.rpm' ]
      },
      'postInstall' : {
         'all' : [ 'Dot1x_S4_21_5T4_22.py',
                   'AclAgent_S4_18_8_T4_21_3.py',
                   'AsuCli_S4_20_5_T4_26_2.py',
                   'KernelCmdlineFilter_S4_18_T4_31_0.py' ],
         'Strata' : [ 'Strata_S4_20_T4_22.py',
                      'StrataDmaShut_S4_4_21_3_T_4_22.py' ]
      }
   },
   '4.22.0' : {
      'preInstall' : {
      },
      'package' : {
         'Sand' : [ 'SandFap_S4_22_T4_23.i686.rpm',
                    'Sand_S4_22_T4_24.i686.rpm' ],
         'Strata' : [ 'Strata_S4_20_T4_22.i686.rpm' ]
      },
      'postInstall' : {
         'all' : [ 'AsuCli_S4_20_5_T4_26_2.py',
                   'KernelCmdlineFilter_S4_18_T4_31_0.py' ],
         'Sand' : [ 'SandFap_S4_22_T4_23.py',
                    'Sand_S4_22_T4_24.py' ],
         'Strata' : [ 'Strata_S4_20_T4_22.py',
                      'StrataDmaShut_S4_4_21_3_T_4_22.py' ]
      }
   },
   '4.22.2' : {
      'preInstall' : {
         'all' : [
            'ArBgp_S4_22_2_T4_31.py',
            'SaveShutPathLogs_S4_22_2_T4_31.py'
         ],
      },
      'package' : {
         'Sand' : [ 'SandFap_S4_22_T4_23.i686.rpm',
                    'Sand_S4_22_T4_24.i686.rpm' ],
         'Strata' : [ 'Strata_S4_20_T4_22.i686.rpm' ]
      },
      'postInstall' : {
         'all' : [ 'AsuCli_S4_20_5_T4_26_2.py',
                   'KernelCmdlineFilter_S4_18_T4_31_0.py' ],
         'Sand' : [ 'SandFap_S4_22_T4_23.py',
                    'Sand_S4_22_T4_24.py' ],
         'Strata' : [ 'Strata_S4_20_T4_22.py',
                      'StrataDmaShut_S4_4_21_3_T_4_22.py' ]
      }
   },
   '4.24.1' : {
      'preInstall' : {
         'all' : [
            'ArBgp_S4_22_2_T4_31.py',
            'SaveShutPathLogs_S4_22_2_T4_31.py'
         ],
      },
      'package' : {
         'all' : [ 'ArBgp_S4_24_1_T4_29.i686.rpm' ]
      },
      'postInstall' : {
         'all' : [ 'AsuCli_S4_20_5_T4_26_2.py',
                   'KernelCmdlineFilter_S4_18_T4_31_0.py',
                   'ArBgp_S4_24_1_T4_29.py' ]
      }
   },
   '4.24.2' : {
      'preInstall' : {
         'all' : [
            'ArBgp_S4_22_2_T4_31.py',
            'SaveShutPathLogs_S4_22_2_T4_31.py'
         ],
      },
      'package' : {
         'all' : [ 'Nat_S4_24_2_T4_25_2.i686.rpm',
                   'ArBgp_S4_24_1_T4_29.i686.rpm' ],
      },
      'postInstall' : {
         'all' : [ 'Nat_S4_24_2_T4_25_2.py',
                   'AsuCli_S4_20_5_T4_26_2.py',
                   'KernelCmdlineFilter_S4_18_T4_31_0.py',
                   'ArBgp_S4_24_1_T4_29.py' ],
      }
   },
   '4.26.0' : {
      'preInstall' : {
         'all' : [
            'ArBgp_S4_22_2_T4_31.py',
            'SaveShutPathLogs_S4_22_2_T4_31.py'
         ],
      },
      'package' : {
         'all' : [ 'ArBgp_S4_24_1_T4_29.i686.rpm' ],
         'Strata' : [
            'AlePhy_S4_26T4_28_2.i686.rpm' ],
      },
      'postInstall' : {
         'all' : [ 'KernelCmdlineFilter_S4_18_T4_31_0.py',
                   'ArBgp_S4_24_1_T4_29.py' ],
         'Strata' : [ 'AlePhy_S4_26T4_28_2.py' ],
      }
   },
   '4.27.0' : {
      'preInstall' : {
         'all' : [
            'ArBgp_S4_22_2_T4_31.py',
            'SaveShutPathLogs_S4_22_2_T4_31.py',
            'WatchdogTimer_S4_27_0_T4_31.py'
         ],
      },
      'package' : {
         'all' : [ 'ArBgp_S4_24_1_T4_29.i686.rpm' ],
         'Strata' : [
            'AlePhy_S4_26T4_28_2.i686.rpm' ],
      },
      'postInstall' : {
         'all' : [ 'KernelCmdlineFilter_S4_18_T4_31_0.py',
                   'ArBgp_S4_24_1_T4_29.py' ],
         'Strata' : [ 'AlePhy_S4_26T4_28_2.py' ],
      }
   },
   '4.29.0' : {
      'preInstall' : {
         'all' : [
            'ArBgp_S4_22_2_T4_31.py',
            'SaveShutPathLogs_S4_22_2_T4_31.py',
            'WatchdogTimer_S4_27_0_T4_31.py'
         ],
      },
      'package' : {
         'all' : [ 'ArBgp_S4_29_T4_32_1.x86_64.rpm' ],
      },
      'postInstall' : {
         'all' : [ 'KernelCmdlineFilter_S4_18_T4_31_0.py',
                   'ArBgp_S4_29_T4_32_1.py' ],
      }
   }
}

# This is used internally by ptests and these files are NOT copied from packages
ptestDb = {
      'test' : {
         'checkSwi' : [ 'AsuReload.py' ],
         'preInstall' : {
             'all' : [ 'AsuReload.py',
                       'AsuPatchUnreachable.py' ]
         },
         'package' : {
            'all' : []
         },
         'postInstall' : {
             'all' : [ 'AsuReload.py',
                       'AsuPatchUnreachable.py' ]
         }
      }
}

releaseDb.update( ptestDb )
simulation = False

# Stages involved in the AsuPatch process (this is a copy from AsuPatchBase.py)
class Stage( object ):
   PREINSTALL = 1
   POSTINSTALL = 2
   CLEANUP = 3
   CHECKSWI = 4

# Updating AsuPatchData would require incrementing version num (any change here
# should be propogated to AsuPatchBase.py)
class AsuPatchData( object ):
   def __init__( self ):
      self.version = 2
      self.logFunc = lambda *args, **kwargs: True
      self.checkSwi = lambda arg: True

_asuPatchData = AsuPatchData()

# The AsuEvent.log will hold the Asu event history. Format:
# YYYY-MM-DD HH:MM:SS <agentName>: <message>
# We also call the logFunc to dump the log.
def logAsuPatch( message ):
   if _asuPatchData.version >= 1:
      _asuPatchData.logFunc( 'AsuPatchDb: ' + message )
   dirPath = '/mnt/flash/persist'
   if not os.path.exists( dirPath ):
      print( 'AsuPatchDb: ' + message )
      return
   filePath = os.path.join( dirPath, 'AsuEvent.log' )
   with open( filePath, 'a' ) as f:
      f.write( datetime.datetime.now().strftime( '%Y-%m-%d %H:%M:%S ' ) +
              'AsuPatchDb: ' + message + '\n' )

class PatchMgr( object ):
   def __init__( self, relDb, version, model ):
      self.relDb = relDb
      self.base = 1
      for rel in relDb.keys():
         if rel != 'test':
            self.base = max( self.base, len( self.versionTuple( rel ) ) )
      self.relKey = None
      self.initRelKey( version )
      self.model = model

   # We expect versionStr of the format '4.12.3F-*' or '15.2.20M-*' or
   # 4.22.1FX-*, etc. So we split it into parts and compare them part-wise
   # ('4.12.3F' is split into part1=4, part2=12, part3=3).
   def versionTuple( self, versionStr ):
      verSplit = versionStr.split( '-' )[ 0 ]
      vNumsAndChars = [ _f for _f in re.split( r'(\d+)|\.', verSplit ) if _f ]
      # collect the numeric parts of the version till we encounter a char
      parts = []
      for part in vNumsAndChars:
         if part.isdigit():
            parts.append( int( part ) )
         else:
            break
      while len( parts ) < self.base:
         parts.append( 0 )
      assert parts
      return tuple( parts )

   def initRelKey( self, version ):
      assert isinstance( version, str )
      if version == 'test':
         self.relKey = version
         return
      self.relDb.pop( 'test', None )
      versionCandidate = self.versionTuple( version )
      self.base = max( self.base, len( versionCandidate ) )
      versionList = sorted( self.relDb.keys(), key=self.versionTuple )

      # Do binary search to find the largest version less than equal to
      # versionCandidate
      start = 0
      end = len( versionList ) - 1
      mid = -1
      while start <= end:
         mid = start + ( end - start ) // 2
         midVersion = self.versionTuple( versionList[ mid ] )
         if midVersion == versionCandidate:
            break
         elif midVersion > versionCandidate:
            end = mid - 1
            mid = end
         else:
            start = mid + 1
      if mid != -1:
         self.relKey = versionList[ mid ]

   def skipRetiredPatches( self, patches ):
      out = []
      for patch in patches:
         if patch in retiredDb:
            logAsuPatch( 'Skip retired patch: %s' % patch )
         else:
            out.append( patch )
      return out

   def getRpms( self ):
      out = []
      if self.relKey:
         if 'all' in self.relDb[ self.relKey ][ 'package' ]:
            out.extend( self.relDb[ self.relKey ][ 'package' ][ 'all' ] )
         if self.model in self.relDb[ self.relKey ][ 'package' ]:
            out.extend( self.relDb[ self.relKey ][ 'package' ][ self.model ] )
      return self.skipRetiredPatches( out )

   def execDefaultCheckSwi( self ):
      assert _asuPatchData.version >= 2
      return _asuPatchData.checkSwi()

   # Starting from version 2, we allow the checkSwi to be called as the first
   # step in doPatch
   def getCheckSwis( self ):
      out = []
      if _asuPatchData.version >= 2:
         if self.relKey and 'checkSwi' in self.relDb[ self.relKey ]:
            out.extend( self.relDb[ self.relKey ][ 'checkSwi' ] )
      return self.skipRetiredPatches( out )

   def getPreInstalls( self ):
      out = []
      if self.relKey:
         if 'all' in self.relDb[ self.relKey ][ 'preInstall' ]:
            out.extend( self.relDb[ self.relKey ][ 'preInstall' ][ 'all' ] )
         if self.model in self.relDb[ self.relKey ][ 'preInstall' ]:
            out.extend( self.relDb[ self.relKey ][ 'preInstall' ][ self.model ] )
      return self.skipRetiredPatches( out )

   def getPostInstalls( self ):
      out = []
      if self.relKey:
         if 'all' in self.relDb[ self.relKey ][ 'postInstall' ]:
            out.extend( self.relDb[ self.relKey ][ 'postInstall' ][ 'all' ] )
         if self.model in self.relDb[ self.relKey ][ 'postInstall' ]:
            out.extend( self.relDb[ self.relKey ][ 'postInstall' ][ self.model ] )
      return self.skipRetiredPatches( out )

   # If we detected an error we will call execute function for all modules
   # prior to the current module (where we found an issue). The execute
   # function is called after updating the stage to doCleanup.
   def cleanup( self, patchPkgsPathName, nameList, name, stage, *args,
                **kwargs ):
      if stage == Stage.POSTINSTALL:
         stage = Stage.CLEANUP
         for cleanupName in nameList:
            if cleanupName == name:
               break
            else:
               self.installExecute( patchPkgsPathName, nameList, cleanupName,
                                    stage, *args, **kwargs )

   # This function first tries to import the module and then call execute().
   def installExecute( self, patchPkgsPathName, nameList, name, stage, *args,
                       **kwargs ):
      ret = 0
      try:
         # You have to crop the extension off the filename if you're going to pass
         # that as 'name' to imp.load_source(). Python views names with a '.' in it
         # as <package>.<submodule>. For example AleMrouteAsuPStore.py in
         # /usr/lib/python-2.7/site-packages/AsuPStorePlugins is named as
         # 'AsuPStorePlugins.AleMrouteAsuPstore'.
         #
         # Doing this gets rid of the "Parent module '...' not found while doing
         # absolute import" RuntimeWarning we were seeing in testing.
         filename, _ = os.path.splitext( os.path.basename( name ) )
         installExec = imp.load_source( '%s' % filename, patchPkgsPathName + name )
      except IOError:
         if simulation:
            print( 'Failed to open', name )
         else:
            syslog.syslog( syslog.LOG_INFO, 'Failed to open %s' % (
               patchPkgsPathName + name ) )
            logAsuPatch( 'Failed to open %s' % ( patchPkgsPathName + name ) )
         self.cleanup( patchPkgsPathName, nameList, name, stage, *args,
                       **kwargs )
         return 1
      except SyntaxError as err:
         if simulation:
            print( 'File corrupted', name )
         else:
            syslog.syslog( syslog.LOG_INFO, 'File corrupted %s' % (
               patchPkgsPathName + name ) )
            logAsuPatch( 'File corrupted %s' % ( patchPkgsPathName + name ) )
         self.cleanup( patchPkgsPathName, nameList, name, stage, *args,
                       **kwargs )
         raise err
      else:
         ret = installExec.execute( stage, *args, **kwargs )
         if ret is not None and ret != 0:
            self.cleanup( patchPkgsPathName, nameList, name, stage, *args,
                          **kwargs )
         else:
            ret = 0
      return ret

# This is the entry point from AsuReloadCli. The caller has to specify the
# current version number of Eos and the platform model name.
# Not specifying the model would default to only 'all' rpm packages
# If successful, we return 0
# If any execute() returned non-zero we return the error after cleanup()
def doPatch( version, model, *args, **kwargs ):
   def installRPM( rpm ):
      if simulation:
         print( name )
         return None
      logAsuPatch( 'Installing patch rpm: %s' % rpm )
      cmd = 'rpm -Uvh %s --replacefiles --replacepkgs' % rpm
      return Tac.run( cmd.split( ' ' ), stdout=Tac.DISCARD, stderr=Tac.CAPTURE,
                      ignoreReturnCode=True, asRoot=True )

   patchPkgsPath = kwargs.get( 'patchPath', '/tmp/AsuPatchPkgs/' )

   assert version is not None and model is not None
   relDb = None
   if kwargs is not None:
      for argName, argVal in six.iteritems( kwargs ):
         if argName == 'relDb':
            relDb = argVal
         elif argName == 'asuPatchData':
            global _asuPatchData
            _asuPatchData = argVal
      if 'asuPatchData' not in kwargs:
         _asuPatchData.version = 0
   else:
      _asuPatchData.version = 0
   patchMgr = PatchMgr( relDb if relDb else releaseDb, version, model )
   stage = Stage.CHECKSWI
   nameList = patchMgr.getCheckSwis()
   if _asuPatchData.version >= 2 and not nameList:
      # We call the supplied checkSwi() when there is no script overriding it
      ret = patchMgr.execDefaultCheckSwi()
      if ret:
         logAsuPatch( 'Default checkSwi failed: ret=%d' % ret )
         return ret
   for name in nameList:
      # call the execute function in these scripts with stage as CHECKSWI
      ret = patchMgr.installExecute( patchPkgsPath, nameList, name, stage,
                                     *args, **kwargs )
      if ret:
         return ret
   stage = Stage.PREINSTALL
   nameList = patchMgr.getPreInstalls()
   for name in nameList:
      # call the execute function in these scripts with stage as PREINSTALL
      ret = patchMgr.installExecute( patchPkgsPath, nameList, name, stage,
                                     *args, **kwargs )
      if ret:
         return ret
   nameList = patchMgr.getRpms()
   for name in nameList:
      # install the rpms
      err = installRPM( patchPkgsPath + name )
      if err:
         logAsuPatch( 'Installing patch rpm: %s %s' % ( name, err ) )
         return 1
   stage = Stage.POSTINSTALL
   nameList = patchMgr.getPostInstalls()
   for name in nameList:
      # call the execute function in these scripts with stage as POSTINSTALL
      ret = patchMgr.installExecute( patchPkgsPath, nameList, name, stage,
                                     *args, **kwargs )
      if ret:
         return ret
   return 0

def usage():
   print( 'AsuPatchDb.py -v <versionNum> -m <modelName> [-s]' )

# This is currently used for manual patching
def main( argv ):
   version = model = None
   try:
      opts, extraArgs = getopt.getopt( argv[ 1 : ], 'hv:m:s', [ 'help',
                                                    'version=', 'model=' ] )
      if extraArgs:
         return usage()
   except getopt.GetoptError as err:
      print( err )
      return usage()
   for opt, arg in opts:
      if opt in ( '-h', '--help' ):
         return usage()
      elif opt in ( '-v', '--version' ):
         version = arg
      elif opt in ( '-m', '--model' ):
         model = arg
      elif opt in '-s':
         global simulation
         simulation = True
   return doPatch( version, model )

if __name__ == '__main__':
   main( sys.argv )

