# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import Tac

X509_SPIFFE = "x509-spiffe"
METADATA = "metadata"
X509_COMMON_NAME = "x509-common-name"

cliCommandBase = "authentication username priority"

usernameExtractionSource = Tac.Type( "Gnmi::usernameExtractionSource" )
usernameSourceType = Tac.Type( "Gnmi::sourceType" )
commonName = usernameExtractionSource()
commonName.source = usernameSourceType.commonName
spiffe = usernameExtractionSource()
spiffe.source = usernameSourceType.spiffe
metadata = usernameExtractionSource()
metadata.source = usernameSourceType.metadata
defaultAuthnUserPriority = [ spiffe, metadata, commonName ]
defaultAuthnUserPriorityWithCUA = [ spiffe, commonName ]
authnUsernamePriorityFromCLI = {
   X509_COMMON_NAME: commonName,
   METADATA: metadata,
   X509_SPIFFE: spiffe,
}
authnUsernamePriorityToCLI = {
   "spiffe": X509_SPIFFE,
   "metadata": METADATA,
   "commonName": X509_COMMON_NAME,
}
matcherX509Oid = CliMatcher.PatternMatcher( pattern='x509-[0-9.]+',
      helpdesc='Extract username from x509 OID', helpname='x509-OID' )
matcherX509CommonName = CliMatcher.KeywordMatcher( X509_COMMON_NAME,
      helpdesc='Extract username from x509 common name' )
matcherX509Spiffe = CliMatcher.KeywordMatcher( X509_SPIFFE,
      helpdesc='Extract username from x509 SPIFFE ID' )
matcherGrpcMetadata = CliMatcher.KeywordMatcher( METADATA,
      helpdesc='Extract username from gRPC context metadata' )

# --------------------------------------------------------------------------------
# [ no | default ] authentication username priority [{ x509-<oid>,
#                   x509-common-name, x509-spiffe, metadata}]
# --------------------------------------------------------------------------------
def isPriorityTokensEqual( authnUsernamePriority, newCliTokens ):
   if len( authnUsernamePriority ) != len( newCliTokens ):
      return False
   for i, val in enumerate( newCliTokens ):
      currElm = authnUsernamePriority[ i ]
      currVal = ( currElm.source if currElm.source !=
            usernameSourceType.x509Oid else currElm.x509Oid )
      currToken = ( authnUsernamePriorityToCLI[ currVal ]
            if currVal in authnUsernamePriorityToCLI
            else f"x509-{currVal}" )
      if ( ( isinstance( val, str ) and currToken != val ) or
           ( isinstance( val, usernameExtractionSource ) and
             currElm != val ) ):
         return False
   return True

class authnUsernamePriorityCmdBase( CliCommand.CliCommandClass ):
   # modify args so we can have the tokens in insertion order
   @staticmethod
   def adapter( mode, args, argList ):
      args.clear()
      # ignore tokens from cliCommandBase
      args[ "tokenPriority" ] = [ b for _, b in
            argList[ len( cliCommandBase.split() ) : ] ]

   @staticmethod
   def handler( mode, args ):
      if mode.modeKey == 'p4-runtime-transport':
         authnUsernamePriority = mode.config_.authnUsernamePriority
      else:
         authnUsernamePriority = mode.config_.endpoints.get(
               mode.name ).authnUsernamePriority
      tokens = args[ "tokenPriority" ]
      if isPriorityTokensEqual( authnUsernamePriority, tokens ):
         return

      authnUsernamePriority.clear()

      for elm in tokens:
         if elm in authnUsernamePriorityFromCLI:
            authnUsernamePriority.push(
                  authnUsernamePriorityFromCLI[ elm ] )
            continue
         oid = usernameExtractionSource()
         oid.x509Oid = elm.split( "-" )[ 1 ]
         oid.source = usernameSourceType.x509Oid
         authnUsernamePriority.push( oid )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.modeKey == 'p4-runtime-transport':
         authnUsernamePriority = mode.config_.authnUsernamePriority
      else:
         authnUsernamePriority = mode.config_.endpoints.get(
               mode.name ).authnUsernamePriority
      if isPriorityTokensEqual( authnUsernamePriority, defaultAuthnUserPriority ):
         return
      authnUsernamePriority.clear()
      for elm in defaultAuthnUserPriority:
         authnUsernamePriority.push( elm )

class authnUsernamePriorityCmd( authnUsernamePriorityCmdBase ):
   syntax = ( f"{cliCommandBase}"
              "{ x509-oid | "
              f"{X509_COMMON_NAME} | "
              f"{X509_SPIFFE} | "
              f"{METADATA} }}" )
   noOrDefaultSyntax = f"{cliCommandBase} ..."
   data = {
      'authentication': 'Configure authentication options',
      'username': 'Configure authentication username options',
      'priority': 'Configure AAA username extraction priority',
      METADATA: CliCommand.Node( matcher=matcherGrpcMetadata,
         maxMatches=1 ),
      'x509-oid': matcherX509Oid,
      X509_COMMON_NAME: CliCommand.Node(
         matcher=matcherX509CommonName, maxMatches=1 ),
      X509_SPIFFE: CliCommand.Node( matcher=matcherX509Spiffe,
         maxMatches=1 ),
   }

class authnUsernamePriorityNoOidCmd( authnUsernamePriorityCmdBase ):
   syntax = ( f"{cliCommandBase} {{ "
              f"{X509_COMMON_NAME} | "
              f"{X509_SPIFFE} | "
              f"{METADATA} }}" )
   noOrDefaultSyntax = f"{cliCommandBase} ..."
   data = {
      'authentication': 'Configure authentication options',
      'username': 'Configure authentication username options',
      'priority': 'Configure AAA username extraction priority',
      METADATA: CliCommand.Node( matcher=matcherGrpcMetadata,
         maxMatches=1 ),
      X509_COMMON_NAME: CliCommand.Node(
         matcher=matcherX509CommonName, maxMatches=1 ),
      X509_SPIFFE: CliCommand.Node( matcher=matcherX509Spiffe,
         maxMatches=1 ),
   }
