# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import re
import sys

import BasicCliUtil
import CliCommon
import CliAaa
import CliCommand
import CliMatcher
import CliModel
import CliParser
import CliSession
import CliToken.Configure
import ConfigMount
import ShowCommand
import Tac
from CliDynamicSymbol import resolveSymbol

# This is set by the CliCli plugin and is a ConfigMount (unlike session.cliConfig)
cliConfig_ = None

showActiveUrlFactory = None

def exitProcess( session ):
   import TerminalUtil # pylint: disable=import-outside-toplevel
   import WaitForWarmup # pylint: disable=import-outside-toplevel
   entityManager = session.entityManager_
   if entityManager.isLocalEm():
      # Prevent ReferenceErrors during timer notifications during destruction.
      # Normally done by atexit handlers, but standlone Cli exits with os._exit(...)
      Tac.Notifiee.closeAllNotifiees()
   else:
      # We need to make sure that we don't exit until our attrlog has
      # flushed to Sysdb.
      WaitForWarmup.wait( entityManager, [ 'Sysdb' ], sleep=True )
   # enable Ctrl-Z before exiting Cli
   TerminalUtil.enableCtrlZ( True )
   sys.stdout.flush()
   sys.exit( 0 )

#-------------------------------------------------------------------------------
# Configuration modes.
#-------------------------------------------------------------------------------
def showRunningConfigWithMode( url, mode ):
   ''' Helper function for showActive, for filtering out config
   commands based on the filter regular expression. '''

   # Iterate through parent modes if we are a child to determine our indent level
   # and insert the mode filter expression for each parent mode to a queue at the
   # head i.e. in order of parent modes saved in config.
   parentFilterExp = []
   pmode = mode.parent_
   while pmode and not isinstance( pmode, GlobalConfigMode ):
      filterExp = pmode.filterExp()
      parentFilterExp.insert( 0, filterExp )
      pmode = pmode.parent_

   try:
      with url.open() as f:
         BasicCliUtil.showRunningConfigWithFilter( f, mode.filterExp(),
                                                   parentFilterExp )
   except OSError as e:
      mode.addError( "Cannot display running-config: %s" % e )

showActiveNode = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher(
      keyword='active',
      helpdesc='Show the current running-config for this sub mode' ) )

showActiveAllNode = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher(
      keyword='all',
      helpdesc='Show configuration with defaults' ) )

showActiveAllDetailNode = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher(
      keyword='detail',
      helpdesc='Show detailed configuration with defaults' ) )

class ShowActiveCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show active'
   data = {
      'active': showActiveNode,
   }

   @staticmethod
   def handler( mode, args ):
      ''' Run within a config sub mode, to show the current running config
      specific to that sub mode. All sub modes which support this command
      should have implemented the showActive() method. '''
      mode.showActive()

class ShowActiveAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show active all [ detail ]'
   data = {
      'active': showActiveNode,
      'all': showActiveAllNode,
      'detail': showActiveAllDetailNode
   }

   @staticmethod
   def handler( mode, args ):
      mode.showActiveAll( 'detail' in args )

##########################################################
#     Config mode comments
#          - comment (multi-line)
#          - !! (append one line)
#          - show comment
##########################################################

commentKwMatcher = CliMatcher.KeywordMatcher( 'comment',
                   helpdesc='Up to 240 characters, comment for this mode',
                   common=True )

class AddCommentCmd( CliCommand.CliCommandClass ):
   syntax = 'comment'
   noOrDefaultSyntax = syntax
   data = {
      'comment': commentKwMatcher
   }

   @staticmethod
   def handler( mode, args ):
      commentStr = BasicCliUtil.getMultiLineInput( mode, cmd='new comment' )
      mode.setComment( commentStr )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.removeComment()

class AppendCommentCmd( CliCommand.CliCommandClass ):
   syntax = 'APPEND [ COMMENT ]'
   noOrDefaultSyntax = syntax
   data = {
      'APPEND': CliMatcher.KeywordMatcher( CliCommon.commentAppendStr,
         helpdesc='Append to comment',
         common=True ),
      'COMMENT': CliMatcher.StringMatcher( helpname='LINE',
         helpdesc='Append a comment to existing comments for current mode.' )
   }

   @staticmethod
   def handler( mode, args ):
      commentStr = args.get( 'COMMENT', '' ).strip()
      mode.appendComment( commentStr )

   noOrDefaultHandler = AddCommentCmd.noOrDefaultHandler

#--------------------------------------------------
# show comment command
#--------------------------------------------------
class ShowCommentCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show comment'
   data = {
      'comment': commentKwMatcher
   }

   @staticmethod
   def handler( mode, args ):
      for m in mode.getIndividualModes():
         commentKey = m.commentKey()
         if commentKey in cliConfig_.comment:
            print( 'Comment for %s:' % commentKey )
            comment = cliConfig_.comment[ commentKey ]
            for line in comment.splitlines():
               print( '    %s' % line )
         else:
            print( 'No comment exists for %s' % commentKey )

def removeCommentWithKey( commentKey ):
   # Remove comment of a specified mode key. Plugins can use it to clean up
   # comments when a mode is deleted or set to default.
   if cliConfig_:
      del cliConfig_.comment[ commentKey ]

class ConfigModeBase( CliParser.Mode, metaclass=BasicCliUtil.NonOverridablePrompt ):
   name = 'Configure'
   showActiveCmdRegistered_ = False
   showActiveAllCmdRegistered_ = False
   showCommentRegistered_ = False
   commentRegistered_ = False
   commentSupported = True
   commonHelpSupported = True
   allowCache = True
   privileged = True

   @classmethod
   def registerCommentCommands( cls ):
      if not cls.commentRegistered_:
         cls.addCommandClass( AddCommentCmd )
         cls.addCommandClass( AppendCommentCmd )

         if not cls.showCommentRegistered_:
            cls.addShowCommandClass( ShowCommentCmd )

      cls.showCommentRegistered_ = True
      cls.commentRegistered_ = True

   @classmethod
   def regCmds( cls ):
      # Below are all related to the 'show active' and 'comment' command.
      # Register the commands.
      if not issubclass( cls, GlobalConfigMode ):
         if not cls.showActiveCmdRegistered_:
            cls.addShowCommandClass( ShowActiveCmd )
            cls.showActiveCmdRegistered_ = True

         if not cls.showActiveAllCmdRegistered_:
            cls.addShowCommandClass( ShowActiveAllCmd )
            cls.showActiveAllCmdRegistered_ = True

      if cls.commentSupported:
         cls.registerCommentCommands()

      if ExitCmd not in cls.commandClasses():
         cls.addCommandClass( ExitCmd )

      if DoExecCmd not in cls.commandClasses():
         cls.addCommandClass( DoExecCmd )

   def __init__( self, parent, session, multiInstance=False, multiModes=() ):
      self.session_ = session
      self.multiInstance_ = multiInstance
      self.individualModes_ = multiModes
      self.pendingComments_ = []
      assert hasattr( self, 'modeKey' )
      assert hasattr( self, 'longModeKey' )
      clazz = self.__class__

      # pylint: disable-msg=no-member
      if self.modeKey == "":
         clazz.prompt = "(config)#"
      else:
         clazz.prompt = "(config-%s)#" % self.modeKey
      # pylint: enable-msg=no-member
      clazz.regCmds()
      CliParser.Mode.__init__( self, parent, session )

   @classmethod
   def clear( cls, mode, args ):
      # This function is used to clear all commands under a config mode.
      # It can be used by config modes to implement the "no/default" command
      # handler. Note that the mode is the parent mode where the no/default
      # handler is registered.
      #
      # The mode has to be singleton.
      newMode = mode.childMode( cls )
      newArgs = { CliCommand.NAME_DEFAULT: "default" }
      for cc in cls.modeParseTree_.commandClasses_:
         if handler := getattr( cc, 'defaultHandler', None ):
            if isinstance( handler, str ):
               handler = resolveSymbol( handler )
            handler( newMode, newArgs )

   def filterExp( self ):
      """ Return the filter expression to get configuration which only
      belongs to this mode. Usually this expression is similar to the
      enterCmd string. The sub mode can overide this if necessary. """
      if hasattr( self, 'enterCmd' ):
         return "^%s$" % re.escape( self.enterCmd() ) # pylint: disable-msg=no-member
      else:
         raise NotImplementedError

   def _checkShowActiveUrl( self ):
      if not showActiveUrlFactory:
         self.addError( "running-config URL is not available. Make sure "
                        "relevant plugins are loaded." )
         raise CliParser.AlreadyHandledError()

   def showActive( self ):
      """ Base implementation of show active. Sub mode can overide to
      provide alternative implementation. """
      self._checkShowActiveUrl()
      assert callable( showActiveUrlFactory )
      # pylint: disable-msg=not-callable
      runningConfigUrl = showActiveUrlFactory( self, False, False )
      showRunningConfigWithMode( runningConfigUrl, self )
      # pylint: enable-msg=not-callable

   def showActiveAll( self, showDetail ):
      self._checkShowActiveUrl()
      assert callable( showActiveUrlFactory )
      # pylint: disable-msg=not-callable
      runningConfigUrl = showActiveUrlFactory( self, True, showDetail )
      showRunningConfigWithMode( runningConfigUrl, self )
      # pylint: enable-msg=not-callable

   def shortPrompt( self ):
      if self.session_.inConfigSession():
         configPrompt = "config-s"
      else:
         configPrompt = "config"

      # pylint: disable-msg=no-member
      return "({}{})#".format( configPrompt,
                               "-" + self.modeKey if self.modeKey else "" )
      # pylint: enable-msg=no-member

   def longPrompt( self ):
      if self.session_.inConfigSession():
         cs = CliSession.currentSession( self.session_.entityManager_ )
         configPrompt = "config-s-%s" % cs[ : 10 ]
      else:
         configPrompt = "config"

      # pylint: disable-msg=no-member
      return "({}{})#".format(
         configPrompt,
         "-" + self.longModeKey if self.longModeKey else self.modeKey
      )
      # pylint: enable-msg=no-member

   def getIndividualModes( self ):
      if self.multiInstance_ and self.individualModes_:
         retModes = []
         for m in self.individualModes_:
            retModes += m.getIndividualModes()
         return retModes
      else:
         return [ self ]

   def setComment( self, commentStr ):
      # set comment to a mode
      for m in self.getIndividualModes():
         commentKey = m.commentKey() # pylint: disable=no-member
         cliConfig_.comment[ commentKey ] = commentStr

   def removeComment( self ):
      # Remove comment of the current mode. Plugins can use it to clean up
      # comments when a mode is deleted or set to default.
      if cliConfig_:
         for m in self.getIndividualModes():
            commentKey = m.commentKey() # pylint: disable=no-member
            removeCommentWithKey( commentKey )
            # clear pending comments if we removed the comment
            m.pendingComments_ = []

   def appendComment( self, commentStr ):
      for m in self.getIndividualModes():
         if self.session.isInteractive():
            # pylint: disable=no-member
            comment = ( cliConfig_.comment.get( m.commentKey(), '' ) +
                        commentStr + '\n' )
            m.setComment( comment )
         else:
            # BUG521397: for non-interactive session, we defer appending the
            # comment until we exit the mode.
            m.pendingComments_.append( commentStr )

   def commitPendingComment( self ):
      # Append pending comments if they are not identical to existing comments
      if not self.pendingComments_:
         return
      commentStr = '\n'.join( self.pendingComments_ ) + '\n'
      for m in self.getIndividualModes():
         commentKey = m.commentKey() # pylint: disable=no-member
         oldCommentStr = cliConfig_.comment.setdefault( commentKey, '' )
         # ignore identical comment
         if oldCommentStr != commentStr:
            cliConfig_.comment[ commentKey ] = oldCommentStr + commentStr
      self.pendingComments_ = []

   def onExit( self ):
      modes = self.getIndividualModes()
      if not ( len( modes ) == 1 and modes[ 0 ] == self ):
         # just recursively call it
         for m in modes:
            m.onExit()
         return

      # single mode, do stuff
      self.commitPendingComment()

   @classmethod
   def isConfigMode( cls ):
      return True

class GlobalConfigMode( ConfigModeBase ):
   name = 'Configure'
   commentSupported = False

   def __init__( self, parent, session ):
      self.modeKey = ""
      self.longModeKey = ""

      ConfigModeBase.__init__( self, parent, session )

   def enter( self ):
      pass

   @classmethod
   def isGlobalConfigMode( cls ):
      return True

class GlobalConfigModeWithHistory( GlobalConfigMode ):
   # need this as we are inherit from our parent
   modeParseTreeShared = True

   def enter( self ):
      BasicCliUtil.logConfigSource( "console", BasicCliUtil.SYS_CONFIG_E )
      # Entering config mode hence adding entry to historyEventTable
      self.session_.addToHistoryEventTable( "commandLine", "commandSource",
                                            "running" )

   def onExit( self ):
      # exit from config mode hence adding entry to historyEventTable
      self.session_.addToHistoryEventTable( "commandLine", "commandSource",
                                            "running" )
      # pylint: disable-msg=no-member
      BasicCliUtil.logConfigSource( "console", BasicCliUtil.SYS_CONFIG_I )
      # pylint: enable-msg=no-member
      GlobalConfigMode.onExit( self )

class ExecMode( CliParser.Mode ):
   # Common mode for UnprivMode and EnableMode
   #
   # ExecMode.addCommandClass() will add commands to both UnprivMode and EnableMode
   name = 'Exec'
   inhibitImplicitModeChange = True
   inhibitAmbiguousCommandErrorToParent = True

   duplicateCommandClassChecked = False

   @classmethod
   def getExtraModeParseTree( cls ):
      return ExecMode.modeParseTree_

   def _onEnable( self, privLevel ):
      raise NotImplementedError()

   def _onDisable( self, privLevel ):
      raise NotImplementedError()

   def __init__( self, *args, **kwargs ):
      CliParser.Mode.__init__( self, *args, **kwargs )
      # sanity check if Unpriv and Enable modes have dupicate command classes
      if not ExecMode.duplicateCommandClassChecked:
         ExecMode.duplicateCommandClassChecked = True
         unprivCls = set( UnprivMode.commandClasses() )
         privCls = set( EnableMode.commandClasses() )
         dupCls = unprivCls & privCls
         if dupCls:
            cl = list( dupCls )[ 0 ]
            assert False, "CommandClass %s is regsitered to both UnprivMode and " \
               "EnableMode: use ExecMode.addCommandClass() instead" % cl

   @BasicCliUtil.EapiIncompatible()
   def exit( self, args ):
      exitProcess( self.session_ )  # pylint: disable-msg=no-member

   def enable( self, args ):
      # pylint: disable-msg=no-member
      privLevel = args.get( 'PRIV_LEVEL', CliCommon.MAX_PRIV_LVL )
      currPrivLevel = self.session_.privLevel_
      assert currPrivLevel >= CliCommon.MIN_PRIV_LVL, (
            "Current privLevel is invalid: %r" % currPrivLevel )
      if currPrivLevel < privLevel and self.session_.authenticationEnabled():
         if not CliAaa.authenticateEnable( self, privLevel ):
            raise CliModel.PermissionDenied()  # Access denied.
      self.session_.changePrivLevel( privLevel )
      self._onEnable( privLevel )
      # pylint: enable-msg=no-member

   def disable( self, args ):
      # pylint: disable-msg=no-member
      privLevel = args.get( 'PRIV_LEVEL', CliCommon.DEFAULT_PRIV_LVL )
      currPrivLevel = self.session_.privLevel_
      assert currPrivLevel >= CliCommon.MIN_PRIV_LVL, (
            "Current privLevel is invalid: %r" % currPrivLevel )
      if privLevel > currPrivLevel:
         errmsg = "New privilege level must be less than current privilege level"
         self.session_.addError( errmsg )
         raise CliModel.BadRequest( errmsg )
      self.session_.changePrivLevel( privLevel )
      self._onDisable( privLevel )
      # pylint: enable-msg=no-member

#-------------------------------------------------------------------------------
# Enable mode.
#-------------------------------------------------------------------------------
class EnableMode( ExecMode ):
   prompt = '#'
   configSessionHook = None
   privileged = True

   def configure( self, args ):
      # pylint: disable=not-callable
      if callable( self.configSessionHook ) and self.configSessionHook( args ):
         return
      # pylint: enable=not-callable
      # We know we'll be entering config mode from exec mode, so
      # in case we are typed in a config mode already, we need to
      # re-enable ConfigMounts and call the onExit() handlers for
      # config modes we've entered.
      with ConfigMount.ConfigMountDisabler( disable=False ):
         self.session_.commitMode()
      # pylint: disable-next=singleton-comparison
      if self.session_.cliSessionStatus != None:
         if self.session_.cliSessionStatus.commitTimerInProgress:
            c = self.session_.cliSessionStatus.sessionStateDir.commitTimerSessionName
            self.session_.addWarning(
               ( f"Config session {c} is pending commit timer. "
               "Any changes done here will be reverted if the session"
               " is not committed." )
            )
      childMode = self.childMode( GlobalConfigModeWithHistory )
      self.session_.gotoChildMode( childMode )
      childMode.enter()

   def _onEnable( self, privLevel ):
      if privLevel <= CliCommon.DEFAULT_PRIV_LVL:
         self.session_.gotoParentMode()

   _onDisable = _onEnable

#-------------------------------------------------------------------------------
# Unprivileged mode.
#-------------------------------------------------------------------------------
class UnprivMode( ExecMode ):
   prompt = ">"

   def onInitialMode( self ):
      # Automatically transition into EnableMode if appropriate.  This helps
      # match industry standard behavior until UnprivMode and EnableMode are
      # unified into a single "ExecMode".
      if self.session_.privLevel_ > CliCommon.DEFAULT_PRIV_LVL:
         self.session_.gotoChildMode( self.childMode( EnableMode ) )

   def _onEnable( self, privLevel ):
      if privLevel > CliCommon.DEFAULT_PRIV_LVL:
         self.session_.gotoChildMode( self.childMode( EnableMode ) )

   def _onDisable( self, privLevel ):
      # I don't have any less privileged mode to which I can transition, so
      # I stay in this mode.
      pass

class EndCmd( CliCommand.CliCommandClass ):
   syntax = 'end'
   data = { 'end': 'Leave config mode' }
   authz = False

   @staticmethod
   def handler( mode, args ):
      mode.session_.gotoParentMode()

GlobalConfigMode.addCommandClass( EndCmd )

class EnableConfigureCmd( CliCommand.CliCommandClass ):
   syntax = 'configure [ terminal ]'
   data = {
            'configure': CliToken.Configure.configureParseNode,
            'terminal': 'Config mode'
          }
   handler = EnableMode.configure

EnableMode.addCommandClass( EnableConfigureCmd )

# BUG121 We need to make the helpdesc of the 'exit' command be different for
# each configuration mode (for example, 'Leave interface configuration
# mode').
def _exitFn( mode ):
   return { 'exit': 'Leave %s mode' % mode.name }

def addBasicCommandsToMode( mode ):
   privLevelMatcher = CliMatcher.IntegerMatcher( CliCommon.MIN_PRIV_LVL,
         CliCommon.MAX_PRIV_LVL, helpdesc='Privilege level' )

   class EnableCmd( CliCommand.CliCommandClass ):
      syntax = 'enable [ PRIV_LEVEL ]'
      data = {
               'enable': 'Enable commands for a specified privilege level',
               'PRIV_LEVEL': privLevelMatcher
             }
      handler = mode.enable

   class DisableCmd( CliCommand.CliCommandClass ):
      syntax = 'disable [ PRIV_LEVEL ]'
      data = {
               'disable': 'Disable commands for a specified privilege level',
               'PRIV_LEVEL': privLevelMatcher
             }
      handler = mode.disable

   class ExitModeCmd( CliCommand.CliCommandClass ):
      syntax = 'exit | logout | quit'
      data = {
               'exit': CliMatcher.DynamicKeywordMatcher( _exitFn, common=True ),
               'logout': 'Exit from EXEC mode',
               'quit': CliCommand.hiddenKeyword( 'quit' )
             }
      handler = mode.exit
      authz = False

   #----------------------------------------------------------------
   # echo STRING
   # CliTestClient uses this echo in it's sync code
   #----------------------------------------------------------------
   class EchoCmd( CliCommand.CliCommandClass ):
      syntax = 'echo STRING'
      data = {
         'echo': 'Echo a string',
         'STRING': CliMatcher.StringMatcher( helpdesc='Character string to echo',
            helpname='string' )
      }
      hidden = True

      @staticmethod
      def handler( mode, args ):
         print( args[ 'STRING' ] )

   mode.addCommandClass( EnableCmd )
   mode.addCommandClass( DisableCmd )
   mode.addCommandClass( ExitModeCmd )
   mode.addCommandClass( EchoCmd )

addBasicCommandsToMode( UnprivMode )
addBasicCommandsToMode( EnableMode )

# Keep this hidden as it's just for backward compatibility and not necessary
class DoExecCmd( CliCommand.CliCommandClass ):
   syntax = 'do CMD'
   data = {
      'do': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'do',
            helpdesc='Run EXEC commands in config mode' ),
         hidden=True ),
      'CMD': CliMatcher.StringMatcher( helpname='COMMAND',
         helpdesc='Exec command' )
   }

   @staticmethod
   def authzFunc( mode, privLevel, tokens ):
      # Authorize "do" as a separate command
      # the command itself with be authorized when we run the command
      return CliAaa.authorizeCommand( mode, privLevel, [ 'do' ] )

   @staticmethod
   def acctFunc( mode, privLevel, tokens ):
      # No accounting for the 'do' version
      pass

   @staticmethod
   def handler( mode, args ):
      mode.session_.runCmd( args[ 'CMD' ], expandAliases=True )

class ExitCmd( CliCommand.CliCommandClass ):
   syntax = 'exit'
   data = {
      'exit': CliMatcher.DynamicKeywordMatcher( _exitFn, common=True )
   }
   authz = False

   @staticmethod
   def handler( mode, args ):
      mode.session_.gotoParentMode()

def addShowCommandClass( cls ):
   """Registers a show command that will be present in Enabled and Unprivileged
   modes, unless the keyword argument 'privileged' is specified, in which case
   it will not be present in Unprivileged mode.

   Commands registered with this function automatically support output filtering,
   such as "show ... | include blah", or "show ... | redirect myfile"."""
   if not cls.privileged:
      # This causes the show command to be added to ExecMode, which eventually
      # will be merged into UnprivMode and EnableMode. The nodes are shared.
      ExecMode.addShowCommandClass( cls )
   else:
      EnableMode.addShowCommandClass( cls )
