# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import contextlib
import Tac
import FpgaUtil
from BcmAsicQspiUtils import CmicGen, QspiFlash
import time

class BroadcomSmbusAsicQspi( FpgaUtil.Fpga ):
   cmicGen = CmicGen.CMICX

   scdPciAddr = None
   accelId = None
   busId = None
   smbusSpacing = 0x80
   deviceId = 0x44

   # SCDs have a reset set and reset clear register, which are usually at
   # the following addresses. The bits used to control one or more ASICs
   # depends on the platform.
   resetSetAddr = 0x4000
   resetClearAddr = 0x4010
   resetMask = None # includes SYS_RESET and PCI_RESET
   resetSys = None  # SYS_RESET only

   def __init__( self ):
      super().__init__()
      self.qspi = QspiFlash( pciAddr=self.scdPciAddr,
                             accelId=self.accelId,
                             busId=self.busId,
                             addrSize=4,
                             smbusSpacing=self.smbusSpacing,
                             responderAddr=self.deviceId,
                             cmicGen=self.cmicGen )
      self.origAsicReset = None

   def fpgaClassName( self ):
      return "broadcomAsicQspi"

   def fwFileName( self ):
      raise NotImplementedError

   def imageFilePath( self ):
      return f"/usr/share/fpga/{self.fwFileName()}.abin"

   def flashSize( self ):
      # Firmware itself is very small but Broadcom recommends a large QSPI
      return 0x2000000

   def getAsicReset( self ):
      """
         Fetches the current reset state of a Broadcom ASIC via SCD
      """
      try:
         output = Tac.run( [ "scd", self.scdPciAddr, "read",
                           str( self.resetSetAddr ) ], stdout=Tac.CAPTURE )
      except Tac.SystemCommandError:
         raise FpgaUtil.UnprogrammedFpga( # pylint: disable=raise-missing-from
              "Could not read scd reset register value" )

      val = FpgaUtil.Fpga.parseRegVal( output, self.resetSetAddr )
      isReset = val & self.resetMask == self.resetMask
      if self.origAsicReset is None:
         self.origAsicReset = isReset
      return isReset

   def applyAsicReset( self, set_ ):
      """
         Sets or clears the reset bits on a Broadcom ASIC
      """
      reg = self.resetSetAddr if set_ else self.resetClearAddr

      try:
         # If we are clearing the reset and resetSys bit is defined
         # bring the Asic out of reset by unsetting SYS_RESET first, wait
         # 150ms, then PCI_RESET (which will be taken out with the resetMask)
         if not set_ and self.resetSys:
            Tac.run( [ "scd", self.scdPciAddr, "write", str( reg ),
                       str( self.resetSys ) ], stdout=Tac.CAPTURE )
            time.sleep( .150 )

         Tac.run( [ "scd", self.scdPciAddr, "write", str( reg ),
                    str( self.resetMask ) ], stdout=Tac.CAPTURE )
         output = Tac.run( [ "scd", self.scdPciAddr, "read",
                             str( self.resetSetAddr ) ], stdout=Tac.CAPTURE )
      except Tac.SystemCommandError:
         # pylint: disable-next=raise-missing-from
         raise FpgaUtil.UnprogrammedFpga( "Failed to write ASIC reset bits in SCD" )

      val = FpgaUtil.Fpga.parseRegVal( output, self.resetSetAddr )
      if ( set_ and val & self.resetMask != self.resetMask ) or \
            ( not set_ and val & self.resetMask != 0 ):
         raise FpgaUtil.UnprogrammedFpga( "Failed to set ASIC reset bits in SCD" )

   def revertAsicReset( self ):
      """
         Reverts the Broadcom ASIC to a state prior to performing upgrade
         operations
      """
      currentState = self.getAsicReset()
      if self.origAsicReset != currentState:
         self.applyAsicReset( self.origAsicReset )

   @contextlib.contextmanager
   def asicResetAs( self, reset ):
      """
         Ensures the ASIC is in the desired reset state for the with block, reverting
         to the previous state at the end
      """
      try:
         wasReset = self.getAsicReset()
         self.applyAsicReset( reset )
         yield wasReset
      finally:
         self.revertAsicReset()

   def hardwareVersion( self ):
      def getFwVersion():
         try:
            return self.qspi.getFwVersion()
         except Exception: # pylint: disable=broad-except
            return None

      try:
         # Disable the Root PCI port to the ASIC so that it is not seen on PCI
         # this is needed to avoid any race during getting the QSPI's version
         # Do this only if ASIC is in reset
         wasReset = self.getAsicReset()
         if hasattr( self, "disableRootPciPort" ):
            if wasReset:
               self.disableRootPciPort()
         with self.asicResetAs( False ):
            retValue = Tac.waitFor( getFwVersion, timeout=30 )
      except Tac.Timeout:
         try:
            # Try one more time so we can grab and report the exception.
            retValue = self.qspi.getFwVersion()
         except Exception as e:
            # pylint: disable-next=consider-using-f-string
            raise FpgaUtil.UnprogrammedFpga( "Error reading QSPI image version: {}"
                                             .format( e ) )
      finally:
         # Re-enable the Root PCI port to the ASIC so that it gets discoverd
         # Only if we did disable above
         if hasattr( self, "enableRootPciPort" ):
            if wasReset:
               self.enableRootPciPort()

      return retValue

   def erase( self ):
      try:
         with self.asicResetAs( False ):
            self.qspi.eraseFlash( self.flashSize() )
      except Exception as e:
         raise FpgaUtil.FpgaException( f"Failed to erase QSPI flash: {e}" )

   def program( self ):
      try:
         with self.asicResetAs( False ):
            self.qspi.programFlash( self.imageFilePath() )
      except Exception as e:
         raise FpgaUtil.UpdateFailed( f"Error updating QSPI flash: {e}" )
