# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This script is a modified version of Dos/DosLib/SpiFlash.py from diags' codebase.
# It provides methods to talk to the QSPI controller present in a number of Broadcom
# ASICs (e.g. Trident4, Jericho2) via iProc registers over SMBus. This allows, for
# example, the PCIe configuration QSPI flash to be reprogammed without needing a
# working a PCIe link.

from enum import Enum
import re
import struct

import FpgaUtil
import SmbusUtil
import Tac

class CmicGen( Enum ):
   CMICX = 'CMICx'
   CMICR = 'CMICr'

# MSPI Controller commands
BRCM_CMD_CONTINUE   = 0x80
BRCM_CMD_NOT_8BITS  = 0x40    # Number of bits comes from QSPI_MSPI_SPCR0_MSB.BITS
BRCM_CMD_DELAY      = 0x20
BRCM_CMD_SS1        = 0x02    # Peripheral select

# QSPI commands (taken from Micron N25Q_256MB data sheet)
QSPI_CMD_READ_ID        = 0x9f
QSPI_CMD_READ           = 0x03
QSPI_CMD_PROGRAM        = 0x02
QSPI_CMD_SECTOR_ERASE   = 0xd8
QSPI_CMD_WRITE_EN       = 0x06
QSPI_CMD_WRITE_DIS      = 0x04
QSPI_CMD_GET_STATUS     = 0x05

# QSPI status
QSPI_STATUS_WIP         = 0x01 # Write in progress

# Number of bytes written at a time to the QSPI
QSPI_WRITE_SIZE = 8
PAGE_SIZE = 0x100

# iProc register addresses
class _CruRegOffsets:
   _baseAddrs = {
      CmicGen.CMICX: 0x1802E000,
      CmicGen.CMICR: 0x02930000,
   }

   CRU_CONTROL = 0x0000

class _PaxbRegOffsets:
   _baseAddrs = {
      CmicGen.CMICX: 0x18012000,
      CmicGen.CMICR: 0x0292C000,
   }

   PAXB_0_UC_LOADER_STATUS = 0x0F84

class _QspiRegOffsets:
   _baseAddrs = {
      CmicGen.CMICX: 0x18021000,
      CmicGen.CMICR: 0x02924000,
   }

   # BSPI Registers
   QSPI_BSPI_REGISTERS_REVISION_ID = 0x0000
   QSPI_BSPI_REGISTERS_MAST_N_BOOT_CTRL = 0x0008
   QSPI_BSPI_REGISTERS_BUSY_STATUS = 0x000C
   QSPI_BSPI_REGISTERS_B0_CTRL = 0x0018
   QSPI_BSPI_REGISTERS_B1_CTRL = 0x0020
   QSPI_BSPI_REGISTERS_FLEX_MODE_ENABLE = 0x0028
   QSPI_BSPI_REGISTERS_BITS_PER_CYCLE = 0x002C
   QSPI_BSPI_REGISTERS_BITS_PER_PHASE = 0x0030
   QSPI_BSPI_REGISTERS_CMD_AND_MODE_BYTE = 0x0034

   # MSPI Registers
   QSPI_MSPI_SPCR0_LSB = 0x0200
   QSPI_MSPI_SPCR0_MSB = 0x0204
   QSPI_MSPI_NEWQP = 0x0210
   QSPI_MSPI_ENDQP = 0x0214
   QSPI_MSPI_SPCR2 = 0x0218
   QSPI_MSPI_MSPI_STATUS = 0x0220
   QSPI_MSPI_TXRAM00 = 0x0240
   QSPI_MSPI_RXRAM00 = 0x02c0
   QSPI_MSPI_CDRAM00 = 0x0340
   QSPI_MSPI_WRITE_LOCK = 0x0380

# As defined in pciephy_fw_hdr_s (see
# //src/SandSdkV7/sdk/sdk-dnx-6.5.30/src/appl/diag/pcie.c)
PCIE_FW_VER_STR_LEN = 12
# Size of header data aligned to QSPI_WRITE_SIZE (struct is 32 bytes)
PCIE_FW_HEADER_WRITE_SIZE = 4 * QSPI_WRITE_SIZE
class _FwHeaderOffsets:
   _baseAddrs = {
      CmicGen.CMICX: 0x2000,
      CmicGen.CMICR: 0x2c00,
   }

   PCIE_FW_MAGIC = 0x0
   PCIE_FW_LDR_VER = 0x04
   PCIE_FW_VER_STR = 0x10

class Registers:
   def __init__( self, cmicGen=CmicGen.CMICX ):
      self.cmicGen = cmicGen

   def __getattr__( self, attr ):
      for offsets in (
            _CruRegOffsets, _PaxbRegOffsets, _QspiRegOffsets,
            _FwHeaderOffsets ):
         try:
            return getattr( offsets, attr ) + offsets._baseAddrs[ self.cmicGen ]
         except AttributeError:
            pass
      raise AttributeError( attr )

class QspiInitMode( Enum ):
   MSPI8 = 'mspi8'
   MSPI16 = 'mspi16'
   BSPI = 'bspi'

def _needsQspiControllerInitMode( mode=QspiInitMode.MSPI8 ):
   def inner( f ):
      def wrapper( self, *args, **kwargs ):
         if self.initMode == mode:
            pass
         elif mode == QspiInitMode.MSPI8:
            self.qspiControllerInit( mode )
            self.initMode = QspiInitMode.MSPI8
         elif mode == QspiInitMode.MSPI16:
            self.qspiControllerInit( mode )
            self.initMode = QspiInitMode.MSPI16
         elif mode == QspiInitMode.BSPI:
            self.qspiControllerInitBspi()
            self.initMode = QspiInitMode.BSPI
         return f( self, *args, **kwargs )
      return wrapper
   return inner


class QspiFlash:
   def __init__( self, pciAddr, accelId, busId, addrSize, smbusSpacing,
                 verifyReadAck=True, verifyWriteAck=False, responderAddr=None,
                 factoryType='raw', cmicGen=CmicGen.CMICX ):
      assert cmicGen in ( CmicGen.CMICX, CmicGen.CMICR ), (
         f'Invalid CMIC gen {cmicGen}' )

      self.pciAddr = pciAddr
      self.accelId = accelId
      self.busId = busId
      self.addrSize = addrSize
      self.smbusSpacing = smbusSpacing
      self.offset = None
      self.verifyReadAck = verifyReadAck
      self.verifyWriteAck = verifyWriteAck
      self.responderAddr = responderAddr
      self.factoryType = factoryType
      self.cmicGen = cmicGen
      self.bspiMmapBase = None

      self.initMode = None
      self.r = Registers( self.cmicGen )

   @staticmethod
   def parseVersion( versionBytes, cmicGen=CmicGen.CMICX ):
      if cmicGen == CmicGen.CMICX:
         m = re.match(
            br'([0-9a-fA-F]{1,4})_([0-9a-fA-F]{1,4})',
            versionBytes.strip( b'\x00' ) )
         assert m, f"Failed to parse version string: {versionBytes}"
         majorVersion = int( m.group( 1 ), 16 )
         minorVersion = int( m.group( 2 ), 16 )
      elif cmicGen == CmicGen.CMICR:
         # In older iProc, the firmware version we used was the version string in the
         # header. From the SDK, it now seems that the version string is blank. Only
         # the numeric loader version remains, although it's now
         # `(u16)major.(u12)minor.(u4)patch`. Since the FPGA infra only deals with
         # `major.minor`, let's pretend that's what it is.
         #
         # A BCD-formatted date field also seems to have appeared in the `crc` field,
         # e.g. 0x20230905 (although it's stored little-endian, as everything else).
         if versionBytes == b'\xff\xff\xff\xff':
            # Special case, flash is blank
            majorVersion, minorVersion = 0, 0
         else:
            minorVersion, majorVersion = struct.unpack( '<HH', versionBytes )
      else:
         assert False, f'Invalid CMIC gen {cmicGen}'

      return FpgaUtil.FpgaVersion( majorVersion, minorVersion )

   @Tac.memoize
   def device( self, verifyAck=True ):
      factory = SmbusUtil.Factory( factoryType=self.factoryType )
      return factory.device( self.accelId, self.busId,
                             self.responderAddr,
                             self.addrSize,
                             readDelayMs='delay0',
                             writeDelayMs='delay0',
                             busTimeout='busTimeout1000ms',
                             writeNoStopReadCurrent=False,
                             smbusAddrSpacing=self.smbusSpacing,
                             smbusAgentId=None,
                             pciAddress=self.pciAddr,
                             verbose=False,
                             verifyAck=verifyAck )

   def calculateBinOffset( self, binBytes ):
      """
         This function should only be called for *.abin files, to determine
         the length of the Arista header.
      """
      off = 0
      for byte in binBytes:
         if int( byte ) == 0:
            break
         off += 1
      else:
         raise AssertionError( "Failed to find null character" )

      self.offset = off + 1

   def readIprocRegister( self, regAddr ):
      # Reads from the ASIC registers _should_ generate a correct ACK
      result = self.device( verifyAck=self.verifyReadAck ).read32( regAddr )

      val = 0
      for i in range( self.addrSize ):
         val = ( val << 8 ) + ( ( result >> ( 8 * i ) ) & 0xff )
      result = val
      return result

   def readIprocRegisterBulk( self, regAddr, count ):
      return self.device(
         verifyAck=self.verifyReadAck ).readBulk32( regAddr, count )

   def writeIprocRegister( self, regAddr, regVal ):
      d = 0
      data = regVal
      for _ in range( self.addrSize ):
         d = ( d << 8 ) + ( data & 0xff )
         data = data >> 8

      # According to Broadcom (DBG16S-AN118), write transactions will generate a
      # NACK for the 4th byte, so verifyAck should be disabled
      try:
         self.device( verifyAck=self.verifyWriteAck ).write32( regAddr, d )
      except SmbusUtil.SmbusFailure:
         if self.factoryType == 'raw':
            raise

   def writeIprocRegisterBulk( self, regAddr, regValArr ):
      dArr = []
      for data in regValArr:
         d = 0
         for _ in range( self.addrSize ):
            d = ( d << 8 ) + ( data & 0xff )
            data = data >> 8
         dArr.append( d )

      # According to Broadcom (DBG16S-AN118), write transactions will generate a
      # NACK for the 4th byte, so verifyAck should be disabled
      try:
         self.device( verifyAck=self.verifyWriteAck ).writeBulk32( regAddr, dArr )
      except SmbusUtil.SmbusFailure:
         if self.factoryType == 'raw':
            raise

   def getFwVersion( self, readMode=QspiInitMode.MSPI8 ):
      if self.cmicGen == CmicGen.CMICX:
         startAddr = self.r.PCIE_FW_VER_STR
         endAddr = self.r.PCIE_FW_VER_STR + PCIE_FW_VER_STR_LEN
      elif self.cmicGen == CmicGen.CMICR:
         startAddr = self.r.PCIE_FW_LDR_VER
         endAddr = self.r.PCIE_FW_LDR_VER + 4

      if readMode == QspiInitMode.BSPI:
         hexData = self.qspiDeviceReadBspi( startAddr, endAddr )
      elif readMode == QspiInitMode.MSPI16:
         hexData = self.qspiDeviceRead16( startAddr, endAddr )
      else:
         hexData = self.qspiDeviceRead( startAddr, endAddr )

      return self.parseVersion(
         hexData[ : endAddr - startAddr ], cmicGen=self.cmicGen )

   def qspiControllerInit( self, mode=QspiInitMode.MSPI8 ):
      """
      Enable the MSPI controller in iProc and bring it to a known state
      """

      # Select the MSPI controller instead of BSPI
      self.writeIprocRegister( self.r.QSPI_BSPI_REGISTERS_MAST_N_BOOT_CTRL, 0x1 )

      # Clear the secondary control register to ensure no transactions take place
      self.writeIprocRegister( self.r.QSPI_MSPI_SPCR2, 0x0 )
      self.writeIprocRegister( self.r.QSPI_MSPI_MSPI_STATUS, 0x0 )

      # Configure the QSPI operating frequency
      self.writeIprocRegister( self.r.QSPI_MSPI_SPCR0_LSB, 0x8 )

      bits = 16 if mode == QspiInitMode.MSPI16 else 8

      # Configure the QSPI operating mode ( 8 bit mode, SPI mode 3 )
      regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR0_MSB )
      regVal = regVal & ~( 1 << 6 )               # STARTTRANSDELAY = 0
      regVal = regVal | ( 1 << 7 )                # MSTR = 1
      regVal = regVal | ( 1 << 1 )                # CPOL = 1
      regVal = regVal | ( 1 << 0 )                # CPHA = 1
      regVal = regVal & ~( 0xf << 2 ) | ( bits << 2 )  # BITS = 8
      self.writeIprocRegister( self.r.QSPI_MSPI_SPCR0_MSB, regVal )

      # Clear out the transaction configuration FIFO
      for fifoIdx in range( 16 ):
         entryAddr = self.r.QSPI_MSPI_CDRAM00 + ( fifoIdx * 4 )
         self.writeIprocRegister( entryAddr, 0x0 )

      # Clear out the transmit and receive FIFOs
      for fifoIdx in range( 32 ):
         entryAddr = self.r.QSPI_MSPI_TXRAM00 + ( fifoIdx * 4 )
         self.writeIprocRegister( entryAddr, 0x0 )

         entryAddr = self.r.QSPI_MSPI_RXRAM00 + ( fifoIdx * 4 )
         self.writeIprocRegister( entryAddr, 0x0 )

   def qspiControllerInitBspi( self ):
      """
      Enable the BSPI controller in iProc and bring it to a known state
      """
      if self.cmicGen == CmicGen.CMICX:
         self.bspiMmapBase = 0x1c000000
      elif self.cmicGen == CmicGen.CMICR:
         self.bspiMmapBase = 0x04000000

      # BSPI: clock configuration according to the flash maximum capability
      regVal = self.readIprocRegister( self.r.CRU_CONTROL )
      regVal &= ~0x00000006
      regVal |= 0x00000004
      self.writeIprocRegister( self.r.CRU_CONTROL, regVal )

      # Disable flex mode first
      self.writeIprocRegister( self.r.QSPI_BSPI_REGISTERS_FLEX_MODE_ENABLE, 0x0 )

      # Data / Address lanes
      regVal = self.readIprocRegister( self.r.QSPI_BSPI_REGISTERS_BITS_PER_CYCLE )
      # Clear {DATA|MODE|ADDR|CMD}_BPC_SELECT (to select 1 bit per cycle for each)
      regVal &= ~0x03030303
      self.writeIprocRegister( self.r.QSPI_BSPI_REGISTERS_BITS_PER_CYCLE, 0 )

      # Dummy cycles
      regVal = self.readIprocRegister( self.r.QSPI_BSPI_REGISTERS_BITS_PER_PHASE )
      regVal &= ~0xff
      regVal |= 8
      self.writeIprocRegister( self.r.QSPI_BSPI_REGISTERS_BITS_PER_PHASE, regVal )

      # Set read Command byte for BSPI
      regVal = 0xb
      self.writeIprocRegister( self.r.QSPI_BSPI_REGISTERS_CMD_AND_MODE_BYTE, regVal )

      # Enable flex mode to take effect
      regVal = 1
      self.writeIprocRegister( self.r.QSPI_BSPI_REGISTERS_FLEX_MODE_ENABLE, regVal )

      # Disable MSPI
      # Disable write lock
      self.writeIprocRegister( self.r.QSPI_MSPI_WRITE_LOCK, 0 )

      # Flush prefetch buffers
      self.writeIprocRegister( self.r.QSPI_BSPI_REGISTERS_B0_CTRL, 0 )
      self.writeIprocRegister( self.r.QSPI_BSPI_REGISTERS_B1_CTRL, 0 )
      self.writeIprocRegister( self.r.QSPI_BSPI_REGISTERS_B0_CTRL, 1 )
      self.writeIprocRegister( self.r.QSPI_BSPI_REGISTERS_B1_CTRL, 1 )

      # Select the BSPI controller instead of MSPI
      self.writeIprocRegister( self.r.QSPI_BSPI_REGISTERS_MAST_N_BOOT_CTRL, 0x0 )

      # Read the Revision Register
      regVal = self.readIprocRegister( self.r.QSPI_BSPI_REGISTERS_REVISION_ID )

      # perform a dummy read from flash to overcome qspi flash access issue
      regVal = self.readIprocRegister( self.bspiMmapBase + 0x2000 )

   @_needsQspiControllerInitMode( QspiInitMode.MSPI8 )
   def qspiDeviceReadId( self ):
      """
      Read the QSPI Manufacturer ID ("JEDEC ID") and device ID
      """
      # Setup the configuration FIFO.  There are a total of 4 operations
      # required to read the 1-byte manufacturer and 2-byte device IDs.
      # (The first operation is a write of the READ_ID command itself).
      # Drive slave select 0 low for all transactions, and
      # only release it after the final transaction completes
      regVal = 0
      regVal = regVal & ~( 1 << 0 )   # SS0N is asserted
      regVal = regVal | ( 1 << 1 )    # SS1N is deasserted
      regVal = regVal | ( 1 << 7 )    # Assert SS0N after transaction completes

      entryAddr = self.r.QSPI_MSPI_CDRAM00
      for _ in range( 4 ):
         self.writeIprocRegister( entryAddr, regVal )
         entryAddr = entryAddr + 4
      regVal = regVal & ~( 1 << 7 )    # Deassert SSxN after final transaction
      self.writeIprocRegister( entryAddr, regVal )

      # Configure the transaction FIFO pointers for 3 operations
      self.writeIprocRegister( self.r.QSPI_MSPI_NEWQP, 0 )
      self.writeIprocRegister( self.r.QSPI_MSPI_ENDQP, 3 )

      # Configure the transmit commands
      # The first command is always a read command, so we can set it one time
      self.writeIprocRegister( self.r.QSPI_MSPI_TXRAM00 + 0, QSPI_CMD_READ_ID )

      # Start the operation
      regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )
      regVal = regVal | ( 1 << 7 )    # CONT_AFTER_CMD = 1
      regVal = regVal | ( 1 << 6 )    # SPE = 1
      self.writeIprocRegister( self.r.QSPI_MSPI_SPCR2, regVal )

      # Poll for completion (SPE will be cleared )
      regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )
      while regVal & ( 1 << 6 ):
         regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )

      data = bytearray()
      entryAddr = self.r.QSPI_MSPI_RXRAM00 + ( 3 * 4 )
      for _ in range( 3 ):
         data.append( self.readIprocRegister( entryAddr ) )
         entryAddr = entryAddr + 8

      return struct.unpack( '>BH', bytes( data ) )

   @_needsQspiControllerInitMode( QspiInitMode.MSPI8 )
   def qspiDeviceRead( self, startAddr=0, endAddr=0x400 ):
      """
      Reads out a range of data from the QSPI device and returns the data
      When called with no arguments, the first 1KB will be read and returned
      """

      bytesRead = bytearray()

      # Setup the configuration FIFO.  There are a total of 12 operations
      # required to read 8 bytesRead (four writes for command and address,
      # eight reads ). Drive slave select 0 low for all transactions, and
      # only release it after the final transaction completes
      regVal = 0
      regVal = regVal & ~( 1 << 0 )   # SS0N is asserted
      regVal = regVal | ( 1 << 1 )    # SS1N is deasserted
      regVal = regVal | ( 1 << 7 )    # Assert SS0N after transaction completes

      entryAddr = self.r.QSPI_MSPI_CDRAM00
      for _ in range( 11 ):
         self.writeIprocRegister( entryAddr, regVal )
         entryAddr = entryAddr + 4
      regVal = regVal & ~( 1 << 7 )    # Deassert SSxN after final transaction
      self.writeIprocRegister( entryAddr, regVal )

      # Configure the transaction FIFO pointers for 12 operations
      self.writeIprocRegister( self.r.QSPI_MSPI_NEWQP, 0 )
      self.writeIprocRegister( self.r.QSPI_MSPI_ENDQP, 11 )

      # Configure the transmit commands
      # The first command is always a read command, so we can set it one time
      self.writeIprocRegister( self.r.QSPI_MSPI_TXRAM00 + 0, QSPI_CMD_READ )
      for qspiAddr in range( startAddr, endAddr, 8 ):
         # Setup the address to read from
         self.writeIprocRegister(
            self.r.QSPI_MSPI_TXRAM00 + 8, ( qspiAddr >> 16 ) & 0xff )
         self.writeIprocRegister(
            self.r.QSPI_MSPI_TXRAM00 + 16, ( qspiAddr >> 8 ) & 0xff )
         self.writeIprocRegister(
            self.r.QSPI_MSPI_TXRAM00 + 24, qspiAddr & 0xff )

         # Start the read operation
         regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )
         regVal = regVal | ( 1 << 7 )    # CONT_AFTER_CMD = 1
         regVal = regVal | ( 1 << 6 )    # SPE = 1
         self.writeIprocRegister( self.r.QSPI_MSPI_SPCR2, regVal )

         # Poll for completion (SPE will be cleared )
         regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )
         while regVal & ( 1 << 6 ):
            regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )

         entryAddr = self.r.QSPI_MSPI_RXRAM00 + ( 9 * 4 )
         for _ in range( 8 ):
            bytesRead.append( self.readIprocRegister( entryAddr ) )
            entryAddr = entryAddr + 8

      return bytes( bytesRead )

   @_needsQspiControllerInitMode( QspiInitMode.MSPI16 )
   def qspiDeviceRead16( self, startAddr=0, endAddr=0x400 ):
      """
      Reads out a range of data from the QSPI device and returns the data
      When called with no arguments, the first 1KB will be read and returned
      """

      bytesRead = bytearray()

      # Setup the configuration FIFO.  There are a total of 12 operations
      # required to read 8 bytesRead (four writes for command and address,
      # eight reads ). Drive slave select 0 low for all transactions, and
      # only release it after the final transaction completes
      regVal = 0
      regVal = regVal & ~( 1 << 0 )   # SS0N is asserted
      regVal = regVal | ( 1 << 1 )    # SS1N is deasserted
      regVal = regVal | ( 1 << 6 )    # BITSE for 16-bit reads
      regVal = regVal | ( 1 << 7 )    # Assert SS0N after transaction completes

      regValArr = []
      for _ in range( 11 ):
         regValArr.append( regVal )
      regVal = regVal & ~( 1 << 7 )    # Deassert SSxN after final transaction
      regValArr.append( regVal )
      self.writeIprocRegisterBulk( self.r.QSPI_MSPI_CDRAM00, regValArr )

      # Configure the transaction FIFO pointers for 12 operations
      self.writeIprocRegisterBulk( self.r.QSPI_MSPI_NEWQP, [ 0, 11 ] )

      # Configure the transmit commands
      # The first command is always a read command, so we can set it one time
      self.writeIprocRegister( self.r.QSPI_MSPI_TXRAM00 + 0, QSPI_CMD_READ )
      for qspiAddr in range( startAddr, endAddr, 8 ):
         addrArr = [
            ( qspiAddr >> 16 ) & 0xff,
            ( qspiAddr >> 8 ) & 0xff,
            qspiAddr & 0xff,
         ]

         self.writeIprocRegisterBulk( self.r.QSPI_MSPI_TXRAM00 + 4, addrArr )

         self._startOperation()
         self._waitForClockOut()

         # Poll for completion (SPE will be cleared )
         regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )
         while regVal & ( 1 << 6 ):
            regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )

         entryAddr = self.r.QSPI_MSPI_RXRAM00 + ( 4 * 4 )
         retValArr = self.readIprocRegisterBulk( entryAddr, 32 )
         bytesRead.extend( retValArr[ : : 4 ] )

      return bytes( bytesRead )

   @_needsQspiControllerInitMode( QspiInitMode.BSPI )
   def qspiDeviceReadBspi( self, startAddr=0, endAddr=0x400 ):
      """
      Reads out a range of data from the QSPI device and returns the data
      When called with no arguments, the first 1KB will be read and returned
      """

      bytesRead = self.readIprocRegisterBulk(
         self.bspiMmapBase + startAddr, endAddr - startAddr )
      return bytes( bytesRead )

   @_needsQspiControllerInitMode( QspiInitMode.MSPI8 )
   def qspiDeviceEraseSector( self, sectorNum = -1 ):
      """
      Erases a 64KB sector in the device
      """

      assert sectorNum >= 0, f'Illegal sector number ({sectorNum}) for erase'
      sectorAddr = sectorNum * 0x10000

      # Setup commands to enable writes
      self.writeIprocRegister( self.r.QSPI_MSPI_CDRAM00, BRCM_CMD_SS1 )
      self.writeIprocRegister( self.r.QSPI_MSPI_TXRAM00, QSPI_CMD_WRITE_EN )
      self.writeIprocRegister( self.r.QSPI_MSPI_NEWQP, 0 )
      self.writeIprocRegister( self.r.QSPI_MSPI_ENDQP, 0 )

      # Start the operation
      regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )
      regVal = regVal | ( 1 << 7 )    # CONT_AFTER_CMD = 1
      regVal = regVal | ( 1 << 6 )    # SPE = 1
      self.writeIprocRegister( self.r.QSPI_MSPI_SPCR2, regVal )

      # Ensure everything has been clocked out (SPE will be cleared)
      regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )
      while regVal & ( 1 << 6 ):
         regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )

      # Setup page erase command for given page
      self.writeIprocRegister(
         self.r.QSPI_MSPI_CDRAM00 + 0x0, BRCM_CMD_CONTINUE | BRCM_CMD_SS1 )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_CDRAM00 + 0x4, BRCM_CMD_CONTINUE | BRCM_CMD_SS1 )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_CDRAM00 + 0x8, BRCM_CMD_CONTINUE | BRCM_CMD_SS1 )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_CDRAM00 + 0xc, BRCM_CMD_SS1 )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_TXRAM00 + 0x0, QSPI_CMD_SECTOR_ERASE )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_TXRAM00 + 0x8, ( sectorAddr >> 16 ) & 0xff )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_TXRAM00 + 0x10, ( sectorAddr >> 8 ) & 0xff )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_TXRAM00 + 0x18, sectorAddr & 0xff )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_NEWQP, 0 )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_ENDQP, 3 )

      # Start the operation
      regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )
      regVal = regVal | ( 1 << 7 )    # CONT_AFTER_CMD = 1
      regVal = regVal | ( 1 << 6 )    # SPE = 1
      self.writeIprocRegister( self.r.QSPI_MSPI_SPCR2, regVal )

      # Ensure everything has been clocked out (SPE will be cleared)
      regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )
      while regVal & ( 1 << 6 ):
         regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )

      # Poll for completion in QSPI device
      self.writeIprocRegister(
         self.r.QSPI_MSPI_CDRAM00 + 0x0, BRCM_CMD_CONTINUE | BRCM_CMD_SS1 )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_CDRAM00 + 0x4, BRCM_CMD_SS1 )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_TXRAM00 + 0x0, QSPI_CMD_GET_STATUS )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_NEWQP, 0 )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_ENDQP, 1 )

      while True:
         regVal = self.readIprocRegister( self.r.QSPI_MSPI_SPCR2 )
         regVal = regVal | ( 1 << 7 )    # CONT_AFTER_CMD = 1
         regVal = regVal | ( 1 << 6 )    # SPE = 1
         self.writeIprocRegister( self.r.QSPI_MSPI_SPCR2, regVal )

         regVal = self.readIprocRegister( self.r.QSPI_MSPI_RXRAM00 + 0xc )
         if regVal & QSPI_STATUS_WIP == 0:
            break

   @_needsQspiControllerInitMode( QspiInitMode.MSPI16 )
   def qspiDeviceWritePage( self, byteArray, pageStartAddr ):
      """
      Write an array of bytes to the QSPI device at the given offset
      """

      assert pageStartAddr % PAGE_SIZE + len( byteArray ) <= PAGE_SIZE,\
            "Data should fit in the same page for each write page func call"

      self.writeIprocRegister( self.r.QSPI_MSPI_CDRAM00, BRCM_CMD_SS1 )
      self.writeIprocRegister( self.r.QSPI_MSPI_TXRAM00, QSPI_CMD_WRITE_EN )
      self.writeIprocRegisterBulk( self.r.QSPI_MSPI_NEWQP, [ 0, 0 ] )
      
      self._startOperation()
      self._waitForClockOut()

      regVal = 0
      regVal = regVal & ~( 1 << 0 )   # SS0N is asserted
      regVal = regVal | ( 1 << 1 )    # SS1N is deasserted
      regVal = regVal | ( 1 << 6 )    # BITSE: 16 bits
      regVal = regVal | ( 1 << 7 )    # Assert SS0N after transaction completes

      regValArr = []
      for _ in range( 2 ):
         regValArr.append( regVal )

      self.writeIprocRegisterBulk( self.r.QSPI_MSPI_CDRAM00, regValArr )

      self.writeIprocRegisterBulk( self.r.QSPI_MSPI_NEWQP, [ 0, 1 ] )

      dataArr = [
         QSPI_CMD_PROGRAM,
         ( pageStartAddr >> 16 ) & 0xff,
         ( pageStartAddr >> 8 ) & 0xff,
         pageStartAddr & 0xff,
      ]

      self.writeIprocRegisterBulk( self.r.QSPI_MSPI_TXRAM00, dataArr )

      self._startOperation()
      self._waitForClockOut()

      self.writeIprocRegister( self.r.QSPI_MSPI_MSPI_STATUS, 0 )

      totalBytes = len( byteArray )

      regValArr = []
      for _ in range( 16 ):
         regValArr.append( regVal )
      self.writeIprocRegisterBulk( self.r.QSPI_MSPI_CDRAM00, regValArr )

      self.writeIprocRegisterBulk( self.r.QSPI_MSPI_NEWQP, [ 0, 15 ] )

      for loopIdx in range( 8 ):
         remainingBytes = totalBytes - ( loopIdx * 32 )
         if remainingBytes <= 32:
            regVal = regVal & ~( 1 << 7 )    # Deassert SSxN after final transaction
            regNo = int( ( remainingBytes / 2 ) - 1 )
            self.writeIprocRegister( self.r.QSPI_MSPI_CDRAM00 + regNo * 4, regVal )

         if totalBytes >= ( loopIdx * 32 ) + 32:
            dataArr = byteArray[ loopIdx * 32 : ( loopIdx * 32 ) + 32 ]
         else:
            dataArr = byteArray[ loopIdx * 32 : totalBytes ]

         self.writeIprocRegisterBulk( self.r.QSPI_MSPI_TXRAM00, dataArr )

         self._startOperation()
         self._waitForClockOut()

         if totalBytes < ( loopIdx * 32 ) + 32:
            break

      self.writeIprocRegisterBulk(
         self.r.QSPI_MSPI_CDRAM00 + ( 14 * 4 ),
         [ BRCM_CMD_CONTINUE | BRCM_CMD_SS1, BRCM_CMD_SS1 ] )
      self.writeIprocRegister(
         self.r.QSPI_MSPI_TXRAM00 + ( 14 * 8 ), QSPI_CMD_GET_STATUS )

      self.writeIprocRegisterBulk( self.r.QSPI_MSPI_NEWQP, [ 14, 14 ] )
      while True:
         self._startOperation()
         self._waitForClockOut()

         regVal = self.readIprocRegister( self.r.QSPI_MSPI_RXRAM00 + 0xc )
         if regVal & QSPI_STATUS_WIP == 0:
            break

      self.writeIprocRegister( self.r.QSPI_MSPI_SPCR2, 0x0 )

   @_needsQspiControllerInitMode( QspiInitMode.MSPI16 )
   def qspiDeviceWrite( self, byteArray, offset ):

      self.writeIprocRegister( self.r.QSPI_MSPI_WRITE_LOCK, 1 )

      pageStart = offset \
         if offset % PAGE_SIZE == 0 \
         else ( offset & 0xFFFFFF00 ) + PAGE_SIZE

      # Write the part until the start of the first page
      prePage = pageStart - offset
      if prePage > len( byteArray ):
         prePage = len( byteArray )
      if prePage > 0:
         self.qspiDeviceWritePage( byteArray[ : prePage ], offset )

      # Write all the full pages
      pageCnt = int( ( len( byteArray ) - prePage ) / PAGE_SIZE )
      for idx in range( pageCnt ):
         self.qspiDeviceWritePage(
               byteArray[ prePage + idx * PAGE_SIZE :
               prePage + ( idx + 1 ) * PAGE_SIZE ],
               pageStart + idx * PAGE_SIZE )

      # Write the remaining part after the last full page
      remaining = len( byteArray ) - ( prePage + pageCnt * PAGE_SIZE )
      if remaining > 0:
         self.qspiDeviceWritePage(
            byteArray[ prePage + pageCnt * PAGE_SIZE : ],
            pageStart + pageCnt * PAGE_SIZE )

      self.writeIprocRegister( self.r.QSPI_MSPI_WRITE_LOCK, 0 )

   def _startOperation( self ):
      regVal = 0
      regVal = regVal | ( 1 << 7 )    # CONT_AFTER_CMD = 1
      regVal = regVal | ( 1 << 6 )    # SPE = 1
      self.writeIprocRegister( self.r.QSPI_MSPI_SPCR2, regVal )

   def _waitForClockOut( self ):
      while self.readIprocRegister( self.r.QSPI_MSPI_MSPI_STATUS ) & 1 == 0:
         pass

   def eraseFlash( self, bytesToErase ):
      requiredSectors = int( ( bytesToErase  / 0x10000 ) + 1 )
      for curSector in range( 0, requiredSectors ):
         self.qspiDeviceEraseSector( curSector )

   def validateVersion( self, bytesData ):
      """
         This function validates if a valid version string is present at
         the expected offset.
      """

      if self.cmicGen == CmicGen.CMICX:
         # Validate the version at PCIE_FW_VER_ADDR
         versionBytes = bytesData[ self.r.PCIE_FW_VER_STR :
                                   ( self.r.PCIE_FW_VER_STR + PCIE_FW_VER_STR_LEN ) ]
      elif self.cmicGen == CmicGen.CMICR:
         versionBytes = bytesData[ self.r.PCIE_FW_LDR_VER :
                                   self.r.PCIE_FW_LDR_VER + 4 ]

      self.parseVersion( versionBytes, self.cmicGen )

   def verifyFlash( self, fileName, readMode=QspiInitMode.MSPI8 ):
      # We only support abin and bin files for programming the flash
      supported_extensions = [ "abin", "bin" ]
      assert fileName.endswith( tuple( supported_extensions ) ), (
                                 "Image file not supported" )

      with open( fileName, 'rb' ) as qspiFlashBin:
         bytesRead = bytearray( qspiFlashBin.read( ) )

      if fileName.endswith( ".abin" ):
         # Trying to determine the offset for Arista header
         self.calculateBinOffset( bytesRead[ : 512 ] )
      else:
         self.offset = 0

      # Truncating the header bytes if present
      bytesData = bytesRead[ self.offset : ]

      # Validate if the version present in the binary
      self.validateVersion( bytes( bytesData ) )

      # Reading back the data for verification
      if readMode == QspiInitMode.BSPI:
         qspiData = self.qspiDeviceReadBspi( 0, len( bytesData ) )
      elif readMode == QspiInitMode.MSPI16:
         qspiData = self.qspiDeviceRead16( 0, len( bytesData ) )
      else:
         qspiData = self.qspiDeviceRead( 0, len( bytesData ) )

      assert bytesData == qspiData, "Qspi flash bytes verification failed"
      return True

   def programFlash( self, fileName, readMode=QspiInitMode.MSPI8 ):
      # We only support abin and bin files for programming the flash
      supported_extensions =  [ "abin", "bin" ]
      assert fileName.endswith( tuple( supported_extensions ) ), (
                                 "Image file not supported" )

      with open( fileName, 'rb' ) as qspiFlashBin:
         bytesRead = bytearray( qspiFlashBin.read( ) )

      if fileName.endswith( ".abin" ):
         # Trying to determine the offset for Arista header
         self.calculateBinOffset( bytesRead[ :512 ] )
      else:
         self.offset = 0

      # Truncating the header bytes if present
      bytesData = bytesRead[ self.offset: ]

      # Validate if the version present in the binary
      self.validateVersion( bytes( bytesData ) )

      # Ensure the array of bytes is QSPI_WRITE_SIZE granular
      while len( bytesData ) % QSPI_WRITE_SIZE != 0:
         bytesData.append( 0x00 )

      # Erase required amount of bytes
      self.eraseFlash( len( bytesData ) )

      # Do programming in 3 steps:
      #  1. Write up to header bytes
      #  2. Write after header bytes
      #  3. Write header bytes
      # This way, if flashing is interrupted, it's far more likely that corruption
      # can be detected (since version bytes are checked specifically on boot).
      headerEnd = self.r.PCIE_FW_MAGIC + PCIE_FW_HEADER_WRITE_SIZE
      # Programming the QSPI device (up to just before the header)
      self.qspiDeviceWrite( bytesData[ : self.r.PCIE_FW_MAGIC ], 0 )

      # Program the remainder after the header
      self.qspiDeviceWrite( bytesData[ headerEnd : ], headerEnd )

      # Finally, program the header bytes
      self.qspiDeviceWrite(
         bytesData[ self.r.PCIE_FW_MAGIC : headerEnd ], self.r.PCIE_FW_MAGIC )

      # Reading back the data for verification
      if readMode == QspiInitMode.BSPI:
         qspiData = self.qspiDeviceReadBspi( 0, len( bytesData ) )
      elif readMode == QspiInitMode.MSPI16:
         qspiData = self.qspiDeviceRead16( 0, len( bytesData ) )
      else:
         qspiData = self.qspiDeviceRead( 0, len( bytesData ) )

      assert bytesData == qspiData, "Qspi flash bytes verification failed"
