#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import time
from socket import AF_INET, AF_INET6, inet_pton
import Tac
from TypeFuture import TacLazyType
IntervalType = Tac.Type( 'Bfd::BfdInterval' )
MultiplierType = Tac.Type( 'Bfd::BfdMultiplier' )
operSessionType = Tac.Type( "Bfd::OperSessionType" )
perlinkMode = Tac.Type( "Bfd::PerlinkMode" )
configSessionType = Tac.Type( 'Bfd::SessionType' )
TunnelIdType = TacLazyType( "Tunnel::TunnelTable::TunnelId" )
TunnelType = TacLazyType( 'Tunnel::TunnelTable::TunnelType' )

class VxlanPeerConfig:
   def __init__( self, ip, flags, vni, innerSrcIp, innerDstMac, innerSrcMac,
                 outerDstIp, outerSrcIp, outerDstPort, outerSrcPort,
                 minTx=IntervalType.defval, minRx=IntervalType.defval,
                 multiplier=MultiplierType.defval ):
      ip = Tac.Value( 'Arnet::IpGenAddr', ip )
      self.peer = Tac.Value( 'Bfd::Peer', ip, 'default' )
      self.peer.type = 'vxlanTunnel'
      self.vxlanTunnel = Tac.Value( 'Bfd::VxlanTunnel' )
      self.vxlanTunnel.flags = flags
      vniTacValue = Tac.Value( 'Arnet::VxlanVniReserved' )
      vniTacValue.vni = vni
      self.vxlanTunnel.vni = vniTacValue
      self.vxlanTunnel.innerSrcIp = Tac.Value( 'Arnet::IpGenAddr', innerSrcIp )
      self.vxlanTunnel.innerDstMac = innerDstMac
      self.vxlanTunnel.innerSrcMac = innerSrcMac
      self.vxlanTunnel.outerDstIp = Tac.Value( 'Arnet::IpGenAddr', outerDstIp )
      self.vxlanTunnel.outerSrcIp = Tac.Value( 'Arnet::IpGenAddr', outerSrcIp )
      self.vxlanTunnel.outerDstPort = outerDstPort
      self.vxlanTunnel.outerSrcPort = outerSrcPort
      self.intervalParams = Tac.Value( 'Bfd::BfdIntervalConfig', minTx, minRx,
                                       multiplier )
      self.intf = ''

def getVxlanValidFlags():
   flags = Tac.Value( 'Arnet::VxlanFlags', 0 )
   flags.vniValid = 1
   return flags

def comparePeerKey( x ):
   return ( not x[ 0 ].startswith( "Port-Channel" ), x[ 0 ] )

def compareIpKey( ipKey ):
   ip = ipKey[ 0 ]
   addrFamily = AF_INET if '.' in str( ip ) else AF_INET6
   ip = inet_pton( addrFamily, str( ip ) )
   return ip

def compareHwSessionKey( x ):
   ipX = x.ip if hasattr( x, 'ip' ) else x[ 'ip' ]
   addrFamily = AF_INET if '.' in str( ipX ) else AF_INET6
   return ( addrFamily != AF_INET, compareIpKey( [ ipX ] ) )

def dispTime( timeSecs, shortDisp=False ):
   if timeSecs == 0:
      return "NA"
   if shortDisp:
      # pylint: disable-next=consider-using-f-string
      disp = "%s" % ( time.strftime( "%m/%d/%y %H:%M",
                                     time.localtime( timeSecs ) ) )
   else:
      ms = repr( timeSecs ).split( '.' )[ 1 ][ : 3 ]
      # pylint: disable-next=consider-using-f-string
      disp = "%s" % ( time.strftime( f"%m/%d/%y %H:%M:%S.{ms}",
                                     time.localtime( timeSecs ) ) )
   return disp

diagEnumToReason = { 'diagNone': 'No Diagnostic',
                     'diagCtrlTimeout': 'Detect Time Exp',
                     'diagEchoFail': 'Echo Function Failed',
                     'diagNeighDown': 'Nbr Signaled Down',
                     'diagForwardingReset': 'Forwarding Plane Reset',
                     'diagPathDown': 'Path Down',
                     'diagConcatPathDown': 'Concatenated Path Down',
                     'diagAdminDown': 'Administratively Down',
                     'diagRevConcatPathDown': 'Reverse Concatenated Path Down',
                   }

authType = Tac.Type( "Bfd::BfdAuthType" )
authEnumTypeToText = { authType.authNone: "None",
                       authType.authPassword: "Password",
                       authType.authKeyedMd5: "Keyed Md5",
                       authType.authMeticulousMd5: "Meticulous Md5",
                       authType.authKeyedSha1: "Keyed Sha1",
                       authType.authMeticulousSha1: "Meticulous Sha1",
                       authType.authInherit: "None",
                       authType.authDebugSeqNum: "None", }

authEnumNumToText = { 0: "None",
                      1: "Password",
                      2: "Keyed Md5",
                      3: "Meticulous Md5",
                      4: "Keyed Sha1",
                      5: "Meticulous Sha1",
                      254: "None",
                      255: "None", }

authTypeDict = { authType.authNone: 0,
                 authType.authPassword: 1,
                 authType.authKeyedMd5: 2,
                 authType.authMeticulousMd5: 3,
                 authType.authKeyedSha1: 4,
                 authType.authMeticulousSha1: 5,
                 authType.authInherit: 254,
                 authType.authDebugSeqNum: 255 }

operSessionEnumToType = { operSessionType.sessionTypeNormal: 'normal',
                          operSessionType.sessionTypeLagLegacy: 'per-link',
                          operSessionType.sessionTypeLagRfc7130: 'RFC7130',
                          operSessionType.sessionTypeMicroLegacy: 'per-link',
                          operSessionType.sessionTypeMicroRfc7130: 'RFC7130',
                          operSessionType.sessionTypeVxlanTunnel: 'VXLAN',
                          operSessionType.sessionTypeMultihop: 'multihop',
                          operSessionType.sessionTypeSbfdInitiator: 'initiator',
                          operSessionType.sessionTypeSbfdReflector: 'reflector',
                        }

def operSessionTypeForPerlink( perLink ):
   # pylint: disable-next=consider-using-in
   if perLink == perlinkMode.rfc7130 or perLink == perlinkMode.rfc7130Interop:
      return operSessionEnumToType[ operSessionType.sessionTypeLagRfc7130 ]
   if perLink == perlinkMode.legacy:
      return operSessionEnumToType[ operSessionType.sessionTypeLagLegacy ]
   return operSessionEnumToType[ operSessionType.sessionTypeNormal ]

diagEnumToValue = { 'diagNone': 0,
                     'diagCtrlTimeout': 1,
                     'diagEchoFail': 2,
                     'diagNeighDown': 3,
                     'diagForwardingReset': 4,
                     'diagPathDown': 5,
                     'diagConcatPathDown': 6,
                     'diagAdminDown': 7,
                     'diagRevConcatPathDown': 8,
                   }

def sbfdLocalIntfDefault():
   return Tac.Value( 'Arnet::IntfId', '' )

def initiatorConfigDefault():
   intervalDefault = Tac.Type( 'Bfd::BfdInterval' ).defval
   multDefault = Tac.Type( 'Bfd::BfdMultiplier' ).defval
   return Tac.Value( 'Bfd::InitiatorTimerConfig', intervalDefault, multDefault )

def reflectorMinRxDefault():
   return Tac.Type( 'Bfd::BfdInterval' ).defval

def reflectorLocalDiscIpDefault():
   return Tac.Value( 'Arnet::IpAddr', 0 )

def peerSessionAf( peer, bfdConfigGlobal ):
   tunnelId = TunnelIdType( peer.tunnelId )
   tunnelType = tunnelId.tunnelType()
   if peer.type == configSessionType.sbfdInitiator and \
         tunnelType == TunnelType.srTeSegmentListTunnel:
      return 'ipv6' if bfdConfigGlobal.sbfdLocalIntfIp6 else 'ipv4'
   else:
      return peer.ip.af
