#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Border Gateway Protocol."""

import dpkt
import struct, socket
import array
import Tac

# Border Gateway Protocol 4 - RFC 4271
# Communities Attribute - RFC 1997
# Capabilities - RFC 3392
# Route Refresh - RFC 2918
# Route Reflection - RFC 4456
# Confederations - RFC 3065
# Cease Subcodes - RFC 4486
# NOPEER Community - RFC 3765
# Multiprotocol Extensions - 2858

# Message Types
OPEN                            = 1
UPDATE                          = 2
NOTIFICATION                    = 3
KEEPALIVE                       = 4
ROUTE_REFRESH                   = 5

# Attribute Types
ORIGIN                          = 1
AS_PATH                         = 2
NEXT_HOP                        = 3
MULTI_EXIT_DISC                 = 4
LOCAL_PREF                      = 5
ATOMIC_AGGREGATE                = 6
AGGREGATOR                      = 7
COMMUNITIES                     = 8
ORIGINATOR_ID                   = 9
CLUSTER_LIST                    = 10
MP_REACH_NLRI                   = 14
MP_UNREACH_NLRI                 = 15
EXTENDED_COMMUNITIES            = 16
EXTENDED_COMMUNITIESIP6         = 25
LARGE_COMMUNITIES               = 32

# Origin Types
ORIGIN_IGP                      = 0
ORIGIN_EGP                      = 1
INCOMPLETE                      = 2

# AS Path Types
AS_SET                          = 1
AS_SEQUENCE                     = 2
AS_CONFED_SEQUENCE              = 3
AS_CONFED_SET                   = 4

# Reserved Communities Types
NO_EXPORT                       = 0xffffff01
NO_ADVERTISE                    = 0xffffff02
NO_EXPORT_SUBCONFED             = 0xffffff03
NO_PEER                         = 0xffffff04

# Common AFI types
AFI_IPV4                        = 1
AFI_IPV6                        = 2

# Multiprotocol SAFI types
SAFI_UNICAST                    = 1
SAFI_MULTICAST                  = 2
SAFI_UNICAST_MULTICAST = 3
SAFI_LABELED_UNICAST = 4
SAFI_MPLS_VPN = 128
SAFI_FLOWSPEC = 133

# OPEN Message Optional Parameters
AUTHENTICATION                  = 1
CAPABILITY                      = 2

# Capability Types
CAP_MULTIPROTOCOL               = 1
CAP_ROUTE_REFRESH               = 2

# NOTIFICATION Error Codes
MESSAGE_HEADER_ERROR            = 1
OPEN_MESSAGE_ERROR              = 2
UPDATE_MESSAGE_ERROR            = 3
HOLD_TIMER_EXPIRED              = 4
FSM_ERROR                       = 5
CEASE                           = 6

# Message Header Error Subcodes
CONNECTION_NOT_SYNCHRONIZED     = 1
BAD_MESSAGE_LENGTH              = 2
BAD_MESSAGE_TYPE                = 3

# OPEN Message Error Subcodes
UNSUPPORTED_VERSION_NUMBER      = 1
BAD_PEER_AS                     = 2
BAD_BGP_IDENTIFIER              = 3
UNSUPPORTED_OPTIONAL_PARAMETER  = 4
AUTHENTICATION_FAILURE          = 5
UNACCEPTABLE_HOLD_TIME          = 6
UNSUPPORTED_CAPABILITY          = 7

# UPDATE Message Error Subcodes
MALFORMED_ATTRIBUTE_LIST        = 1
UNRECOGNIZED_ATTRIBUTE          = 2
MISSING_ATTRIBUTE               = 3
ATTRIBUTE_FLAGS_ERROR           = 4
ATTRIBUTE_LENGTH_ERROR          = 5
INVALID_ORIGIN_ATTRIBUTE        = 6
AS_ROUTING_LOOP                 = 7
INVALID_NEXT_HOP_ATTRIBUTE      = 8
OPTIONAL_ATTRIBUTE_ERROR        = 9
INVALID_NETWORK_FIELD           = 10
MALFORMED_AS_PATH               = 11

# Cease Error Subcodes
MAX_NUMBER_OF_PREFIXES_REACHED  = 1
ADMINISTRATIVE_SHUTDOWN         = 2
PEER_DECONFIGURED               = 3
ADMINISTRATIVE_RESET            = 4
CONNECTION_REJECTED             = 5
OTHER_CONFIGURATION_CHANGE      = 6
CONNECTION_COLLISION_RESOLUTION = 7
OUT_OF_RESOURCES                = 8

# Flowspec Component Types
FLOWSPEC_COMP_TYPE_DEST_PREFIX = 1
FLOWSPEC_COMP_TYPE_SRC_PREFIX = 2
FLOWSPEC_COMP_TYPE_IP_PROTO = 3
FLOWSPEC_COMP_TYPE_IP_PORT = 4
FLOWSPEC_COMP_TYPE_DEST_PORT = 5
FLOWSPEC_COMP_TYPE_SRC_PORT = 6
FLOWSPEC_COMP_TYPE_ICMP_TYPE = 7
FLOWSPEC_COMP_TYPE_ICMP_CODE = 8
FLOWSPEC_COMP_TYPE_TCP_FLAGS = 9
FLOWSPEC_COMP_TYPE_PKT_LEN = 10
FLOWSPEC_COMP_TYPE_DSCP_CODE_POINT = 11
FLOWSPEC_COMP_TYPE_FRAGMENT = 12
FLOWSPEC_COMP_TYPE_FLOW_LABEL = 13

# Flowspec numeric operator bits

class BGP(dpkt.Packet):
    __hdr__ = (
        ('marker', '16s', '\x01' * 16),
        ('len', 'H', 0),
        ('type', 'B', OPEN)
        )

    as4_ = True

    def __init__( self, data, addPathNegociated=False ):
        self.addPathNegociated = addPathNegociated
        super().__init__(data)

    def unpack(self, buf):
        dpkt.Packet.unpack(self, buf)
        self.data = self.data[:self.len - self.__hdr_len__]
        if self.type == OPEN:
            self.data = self.open = self.Open(self.data)
        elif self.type == UPDATE:
            self.data = self.update = self.Update(self.data, self.addPathNegociated)
        elif self.type == NOTIFICATION:
            self.data = self.notifiation = self.Notification(self.data)
        elif self.type == KEEPALIVE:
            self.data = self.keepalive = self.Keepalive(self.data)
        elif self.type == ROUTE_REFRESH:
            self.data = self.route_refresh = self.RouteRefresh(self.data)

    class Open(dpkt.Packet):
        __hdr__ = (
            ('v', 'B', 4),
            ('asn', 'H', 0),
            ('holdtime', 'H', 0),
            ('identifier', 'I', 0),
            ('param_len', 'B', 0)
            )

        def unpack(self, buf):
            dpkt.Packet.unpack(self, buf)
            if self.param_len:
               param = self.Parameter(self.data)
               self.data = self.parameters = param

        def __len__(self):
            return self.__hdr_len__ + len( self.data )

        class Parameter(dpkt.Packet):
            __hdr__ = (
                ('type', 'B', 0),
                ('len', 'B', 0)
                )

            def unpack(self, buf):
               dpkt.Packet.unpack(self, buf)
               capList = []
               while self.len > 0:
                  cap = self.Capability(self.data)
                  self.data = self.data[len(cap):]
                  self.len -= len(cap)
                  capList.append(cap)
               self.data = self.capabilities = capList

            def __len__(self):
               return self.__hdr_len__ + \
                  sum( map( len, self.data ) )

            class Capability(dpkt.Packet):
                __hdr__ = (
                    ('code', 'B', 0),
                    ('len', 'B', 0)
                    )

                def unpack(self, buf):
                    dpkt.Packet.unpack(self, buf)
                    self.data = self.data[:self.len]


    class Update(dpkt.Packet):
        __hdr_defaults__ = {
            'withdrawn': [],
            'attributes': [],
            'announced': [],
            'announcedAddPathIds': [],
            'withdrawnAddPathIds': [],
            }

        def __init__( self, data, addPathNegociated=False ):
            self.addPathNegociated = addPathNegociated
            super().__init__( data )

        def unpack(self, buf):
            self.data = buf

            # Withdrawn Routes
            wlen = struct.unpack('>H', self.data[:2])[0]
            self.data = self.data[2:]
            l = []
            addPathIds = []
            while wlen > 0:
                if self.addPathNegociated:
                    addPathId = struct.unpack('!I', self.data[:4])[0]
                    self.data = self.data[4:]
                    addPathIds.append(addPathId)
                    wlen -= 4
                route = RouteIPV4(self.data)
                self.data = self.data[len(route):]
                wlen -= len(route)
                l.append(route)
            self.withdrawn = l
            self.withdrawnAddPathIds = addPathIds

            # Path Attributes
            plen = struct.unpack('>H', self.data[:2])[0]
            self.data = self.data[2:]
            l = []
            while plen > 0:
                attr = self.Attribute(self.data, self.addPathNegociated )
                self.data = self.data[len(attr):]
                plen -= len(attr)
                l.append(attr)
            self.attributes = l

            # Announced Routes
            l = []
            addPathIds = []
            while self.data:
                if self.addPathNegociated:
                    addPathId = struct.unpack('!I', self.data[:4])[0]
                    self.data = self.data[4:]
                    addPathIds.append(addPathId)
                route = RouteIPV4(self.data)
                self.data = self.data[len(route):]
                l.append(route)
            self.announced = l
            self.announcedAddPathIds = addPathIds

        def __len__(self):
            return 2 + sum(map(len, self.withdrawn)) + \
                   2 + sum(map(len, self.attributes)) + \
                   sum(map(len, self.announced)) + \
                   len( self.withdrawnAddPathIds ) * 4 + \
                   len( self.announcedAddPathIds ) * 4

        def __str__(self):
            return struct.pack('>H', sum(map(len, self.withdrawn))) + \
                   ''.join(map(str, self.withdrawn)) + \
                   struct.pack('>H', sum(map(len, self.attributes))) + \
                   ''.join(map(str, self.attributes)) + \
                   ''.join(map(str, self.announced))
 
        class Attribute(dpkt.Packet):
            __hdr__ = (
                ('flags', 'B', 0),
                ('type', 'B', 0)
                )

            def __init__( self, data, addPathNegociated=False ):
               self.addPathNegociated = addPathNegociated
               super().__init__( data )

            def _get_o(self):
                return (self.flags >> 7) & 0x1
            def _set_o(self, o):
                self.flags = (self.flags & ~0x80) | ((o & 0x1) << 7)
            optional = property(_get_o, _set_o)

            def _get_t(self):
                return (self.flags >> 6) & 0x1
            def _set_t(self, t):
                self.flags = (self.flags & ~0x40) | ((t & 0x1) << 6)
            transitive = property(_get_t, _set_t)

            def _get_p(self):
                return (self.flags >> 5) & 0x1
            def _set_p(self, p):
                self.flags = (self.flags & ~0x20) | ((p & 0x1) << 5)
            partial = property(_get_p, _set_p)

            def _get_e(self):
                return (self.flags >> 4) & 0x1
            def _set_e(self, e):
                self.flags = (self.flags & ~0x10) | ((e & 0x1) << 4)
            extended_length = property(_get_e, _set_e)

            def unpack(self, buf):
                dpkt.Packet.unpack(self, buf)

                if self.extended_length:
                    self.len = struct.unpack('>H', self.data[:2])[0]
                    self.data = self.data[2:]
                else:
                    self.len = struct.unpack('B', self.data[:1])[0]
                    self.data = self.data[1:]
                
                self.data = self.data[:self.len]

                if self.type == ORIGIN:
                    self.data = self.origin = self.Origin(self.data)
                elif self.type == AS_PATH:
                   if ( BGP.as4_ ):
                      self.data = self.as_path = self.AS4Path(self.data)
                   else:
                      self.data = self.as_path = self.ASPath(self.data)
                elif self.type == NEXT_HOP:
                    self.data = self.next_hop = self.NextHop(self.data)
                elif self.type == MULTI_EXIT_DISC:
                    self.data = self.multi_exit_disc = self.MultiExitDisc(self.data)
                elif self.type == LOCAL_PREF:
                    self.data = self.local_pref = self.LocalPref(self.data)
                elif self.type == ATOMIC_AGGREGATE:
                    self.data = self.atomic_aggregate = \
                                self.AtomicAggregate(self.data)
                elif self.type == AGGREGATOR:
                    self.data = self.aggregator = self.Aggregator(self.data)
                elif self.type == COMMUNITIES:
                    self.data = self.communities = self.Communities(self.data)
                elif self.type == ORIGINATOR_ID:
                    self.data = self.originator_id = self.OriginatorID(self.data)
                elif self.type == CLUSTER_LIST:
                    self.data = self.cluster_list = self.ClusterList(self.data)
                elif self.type == MP_REACH_NLRI:
                    self.data = self.mp_reach_nlri = self.MPReachNLRI( self.data,
                                                            self.addPathNegociated )
                elif self.type == MP_UNREACH_NLRI:
                    self.data = self.mp_unreach_nlri = self.MPUnreachNLRI( self.data,
                                                            self.addPathNegociated )
                elif self.type == EXTENDED_COMMUNITIES:
                    self.data = self.communities = self.ExtCommunities(self.data)
                elif self.type == EXTENDED_COMMUNITIESIP6:
                    self.data = self.communities_ip6 = self.ExtCommunitiesIp6(
                       self.data)
                elif self.type == LARGE_COMMUNITIES:
                    self.data = self.large_communities = self.LargeCommunities(
                       self.data)

            def __len__(self):
                if self.extended_length:
                    attr_len = 2
                else:
                    attr_len = 1
                return self.__hdr_len__ + \
                       attr_len + \
                       len(self.data)

            def __str__(self):
                if self.extended_length:
                    attr_len_str = struct.pack('>H', self.len)
                else:
                    attr_len_str = struct.pack('B', self.len)
                return self.pack_hdr() + \
                       attr_len_str + \
                       str(self.data)

            class Origin(dpkt.Packet):
                __hdr__ = (
                    ('type', 'B', ORIGIN_IGP),
                )

            class ASPath(dpkt.Packet):
                __hdr_defaults__ = {
                    'segments': []
                    }

                def unpack(self, buf):
                    self.data = buf
                    l = []
                    while self.data:
                        seg = self.ASPathSegment(self.data)
                        self.data = self.data[len(seg):]
                        l.append(seg)
                    self.data = self.segments = l

                def __len__(self):
                    return sum(map(len, self.data))

                def __str__(self):
                    return ''.join(map(str, self.data))
 
                class ASPathSegment(dpkt.Packet):
                    __hdr__ = (
                        ('type', 'B', 0),
                        ('len', 'B', 0)
                        )

                    def unpack(self, buf):
                        dpkt.Packet.unpack(self, buf)
                        l = []
                        for i in range(self.len):
                            AS = struct.unpack('>H', self.data[:2])[0]
                            self.data = self.data[2:]
                            l.append(AS)
                        self.data = self.path = l

                    def __len__(self):
                        return self.__hdr_len__ + \
                               2 * len(self.path)

                    def __str__(self):
                        as_str = ''
                        for AS in self.path:
                            as_str += struct.pack('>H', AS)
                        return self.pack_hdr() + \
                               as_str

            class AS4Path(dpkt.Packet):
                __hdr_defaults__ = {
                    'segments': []
                    }

                def unpack(self, buf):
                    self.data = buf
                    l = []
                    while self.data:
                        seg = self.ASPathSegment(self.data)
                        self.data = self.data[len(seg):]
                        l.append(seg)
                    self.data = self.segments = l

                def __len__(self):
                    return sum(map(len, self.data))

                def __str__(self):
                    return ''.join(map(str, self.data))
 
                class ASPathSegment(dpkt.Packet):
                    __hdr__ = (
                        ('type', 'B', 0),
                        ('len', 'B', 0)
                        )

                    def unpack(self, buf):
                        dpkt.Packet.unpack(self, buf)
                        l = []
                        for i in range(self.len):
                            AS = struct.unpack('!L', self.data[:4])[0]
                            self.data = self.data[4:]
                            l.append(AS)
                        self.data = self.path = l

                    def __len__(self):
                        return self.__hdr_len__ + \
                               4 * len(self.path)

                    def __str__(self):
                        as_str = ''
                        for AS in self.path:
                            as_str += struct.pack('!L', AS)
                        return self.pack_hdr() + \
                               as_str

            class NextHop(dpkt.Packet):
                __hdr__ = (
                    ('ip', 'I', 0),
                )

            class MultiExitDisc(dpkt.Packet):
                __hdr__ = (
                    ('value', 'I', 0),
                )

            class LocalPref(dpkt.Packet):
                __hdr__ = (
                    ('value', 'I', 0),
                )

            class AtomicAggregate(dpkt.Packet):
                __hdr_defaults__ = { }
                def unpack(self, buf):
                    pass

                def __len__(self):
                    return 0

                def __str__(self):
                    return ''

            class Aggregator(dpkt.Packet):
                __hdr__ = (
                    ('asn', 'H', 0),
                    ('ip', 'I', 0)
                )

            class Communities(dpkt.Packet):
                __hdr_defaults__ = {
                    'list': []
                    }

                def unpack(self, buf):
                    self.data = buf
                    l = []
                    while self.data:
                        val = struct.unpack('>I', self.data[:4])[0]
                        if ( val >= 0x00000000 and val <= 0x0000ffff ) or \
                           ( val >= 0xffff0000 and val <= 0xffffffff ):
                            comm = self.ReservedCommunity(self.data[:4])
                        else:
                            comm = self.Community(self.data[:4])
                        self.data = self.data[len(comm):]
                        l.append(comm)
                    self.data = self.list = l

                def __len__(self):
                    return sum(map(len, self.data))

                def __str__(self):
                    return ''.join(map(str, self.data))

                class Community(dpkt.Packet):
                    __hdr__ = (
                        ('asn', 'H', 0),
                        ('value', 'H', 0)
                    )

                class ReservedCommunity(dpkt.Packet):
                    __hdr__ = (
                        ('value', 'I', 0),
                    )

            class ExtCommunities(dpkt.Packet):
                __hdr_defaults__ = {
                    'list': []
                    }

                def unpack(self, buf):
                    self.data = buf
                    l = []
                    while self.data:
                        comm = self.ExtCommunity(self.data[:8])
                        self.data = self.data[len(comm):]
                        l.append(comm)
                    self.data = self.list = l

                def __len__(self):
                    return sum(map(len, self.data))

                def __str__(self):
                    return ''.join(map(str, self.data))

                class ExtCommunity(dpkt.Packet):
                    __hdr__ = (
                       ('type', 'B', 0),
                       ('value', '7s', 0)
                    )

            class ExtCommunitiesIp6(dpkt.Packet):
                __hdr_defaults__ = {
                    'list': []
                    }

                def unpack(self, buf):
                    self.data = buf
                    l = []
                    while self.data:
                        comm = self.ExtCommunityIp6(self.data[:20])
                        self.data = self.data[len(comm):]
                        l.append(comm)
                    self.data = self.list = l

                def __len__(self):
                    return sum(map(len, self.data))

                def __str__(self):
                    return ''.join(map(str, self.data))

                class ExtCommunityIp6(dpkt.Packet):
                    __hdr__ = (
                       ('type', 'B', 0),
                       ('value', '19s', 0)
                    )

            class LargeCommunities(dpkt.Packet):
                __hdr_defaults__ = {
                    'list': []
                    }

                def unpack(self, buf):
                    self.data = buf
                    l = []
                    while self.data:
                        comm = self.LargeCommunity(self.data[:12])
                        self.data = self.data[len(comm):]
                        l.append(comm)
                    self.data = self.list = l

                def __len__(self):
                    return sum(map(len, self.data))

                def __str__(self):
                    return ''.join(map(str, self.data))

                class LargeCommunity(dpkt.Packet):
                    __hdr__ = (
                       ('global_admin', '4s', 0),
                       ('local_data', '8s', 0)
                    )

            class OriginatorID(dpkt.Packet):
                __hdr__ = (
                    ('value', 'I', 0),
                )

            class ClusterList(dpkt.Packet):
                __hdr_defaults__ = {
                    'list': []
                    }

                def unpack(self, buf):
                    self.data = buf
                    l = []
                    while self.data:
                        id = struct.unpack('>I', self.data[:4])[0]
                        self.data = self.data[4:]
                        l.append(id)
                    self.data = self.list = l

                def __len__(self):
                    return 4 * len(self.list)

                def __str__(self):
                    cluster_str = ''
                    for val in self.list:
                            cluster_str += struct.pack('>I', val)
                    return cluster_str

            class MPReachNLRI(dpkt.Packet):
                __hdr__ = (
                    ('afi', 'H', AFI_IPV4),
                    ('safi', 'B', SAFI_UNICAST),
                )

                def __init__( self, data, addPathNegociated=False ):
                   self.addPathNegociated = addPathNegociated
                   self.addPathIds = []
                   super().__init__( data )

                def unpack(self, buf):
                    dpkt.Packet.unpack(self, buf)

                    # Next Hop
                    nlen = struct.unpack('B', self.data[:1])[0]
                    self.data = self.data[1:]
                    self.next_hop = self.data[:nlen]
                    self.data = self.data[nlen:]

                    # SNPAs
                    l = []
                    num_snpas = struct.unpack('B', self.data[:1])[0]
                    self.data = self.data[1:]
                    for i in range(num_snpas):
                        snpa = self.SNPA(self.data)
                        self.data = self.data[len(snpa):]
                        l.append(snpa)
                    self.snpas = l

                    if self.afi == AFI_IPV4:
                       if self.safi == SAFI_UNICAST: 
                          Route = RouteIPV4
                       elif self.safi == SAFI_MPLS_VPN:
                          Route = RouteIPV4Vpn
                       elif self.safi == SAFI_FLOWSPEC:
                          Route = RouteIPV4Flowspec
                    elif self.afi == AFI_IPV6:
                       if self.safi == SAFI_UNICAST:
                          Route = RouteIPV6
                       elif self.safi == SAFI_LABELED_UNICAST:
                          Route = RouteIPV6Label
                       elif self.safi == SAFI_MPLS_VPN:
                          Route = RouteIPV6Vpn
                       elif self.safi == SAFI_FLOWSPEC:
                          Route = RouteIPV6Flowspec
                    else:
                       Route = RouteGeneric

                    # Announced Routes
                    l = []
                    addPathIds = []
                    while self.data:
                        if self.addPathNegociated:
                           addPathId = struct.unpack('!I', self.data[:4])[0]
                           self.data = self.data[4:]
                           addPathIds.append(addPathId)
                        route = Route(self.data)
                        self.data = self.data[len(route):]
                        l.append(route)
                    self.data = self.announced = l
                    self.addPathIds = addPathIds

                def __len__(self):
                    return self.__hdr_len__ + \
                           1 + len(self.next_hop) + \
                           1 + sum(map(len, self.snpas)) + \
                           sum(map(len, self.announced)) + \
                           len( self.addPathIds ) * 4

                def __str__(self):
                    return self.pack_hdr() + \
                           struct.pack('B', len(self.next_hop)) + \
                           str(self.next_hop) + \
                           struct.pack('B', len(self.snpas)) + \
                           ''.join(map(str, self.snpas)) + \
                           ''.join(map(str, self.announced))

                class SNPA:
                    __hdr__ = (
                        ('len', 'B', 0),
                        )

                    def unpack(self, buf):
                        dpkt.Packet.unpack(self, buf)
                        self.data = self.data[:(self.len + 1) // 2]

            class MPUnreachNLRI(dpkt.Packet):
                __hdr__ = (
                    ('afi', 'H', AFI_IPV4),
                    ('safi', 'B', SAFI_UNICAST),
                )

                def __init__( self, data, addPathNegociated=False ):
                   self.addPathNegociated = addPathNegociated
                   self.addPathIds = []
                   super().__init__( data )

                def unpack(self, buf):
                    dpkt.Packet.unpack(self, buf)

                    if self.afi == AFI_IPV4:
                       if self.safi == SAFI_UNICAST:
                          Route = RouteIPV4
                       elif self.safi == SAFI_MPLS_VPN:
                          Route = RouteIPV4Vpn
                       elif self.safi == SAFI_FLOWSPEC:
                          Route = RouteIPV4Flowspec
                    elif self.afi == AFI_IPV6:
                      if self.safi == SAFI_UNICAST:
                          Route = RouteIPV6
                      elif self.safi == SAFI_LABELED_UNICAST:
                          Route = RouteIPV6Label
                      elif self.safi == SAFI_MPLS_VPN:
                          Route = RouteIPV6Vpn
                      elif self.safi == SAFI_FLOWSPEC:
                          Route = RouteIPV6Flowspec
                    else:
                      Route = RouteGeneric

                    # Withdrawn Routes
                    l = []
                    addPathIds = []
                    while self.data:
                        if self.addPathNegociated:
                           addPathId = struct.unpack('!I', self.data[:4])[0]
                           self.data = self.data[4:]
                           addPathIds.append(addPathId)
                        route = Route(self.data)
                        self.data = self.data[len(route):]
                        l.append(route)
                    self.data = self.withdrawn = l
                    self.addPathIds = addPathIds

                def __len__(self):
                    return self.__hdr_len__ + \
                           sum(map(len, self.data)) + \
                           len( self.addPathIds ) * 4

                def __str__(self):
                    return self.pack_hdr() + \
                           ''.join(map(str, self.data))


    class Notification(dpkt.Packet):
        __hdr__ = (
            ('code', 'B', 0),
            ('subcode', 'B', 0),
            )

        def unpack(self, buf):
            dpkt.Packet.unpack(self, buf)
            self.error = self.data


    class Keepalive(dpkt.Packet):
        def unpack(self, buf):
            pass

        def __len__(self):
            return 0

        def __str__(self):
            return ''


    class RouteRefresh(dpkt.Packet):
        __hdr__ = (
            ('afi', 'H', AFI_IPV4),
            ('rsvd', 'B', 0),
            ('safi', 'B', SAFI_UNICAST)
            ) 


class RouteGeneric(dpkt.Packet):
    __hdr__ = (
        ('len', 'B', 0),
        )

    def unpack(self, buf):
        dpkt.Packet.unpack(self, buf)
        self.data = self.prefix = self.data[:(self.len + 7) // 8]

class RouteIPV4(dpkt.Packet):
    __hdr__ = (
        ('len', 'B', 0),
        )

    def unpack(self, buf):
        dpkt.Packet.unpack(self, buf)
        tmp = self.data[:(self.len + 7) // 8]
        tmp += ( 4 - len( tmp ) ) * b'\x00'
        self.data = self.prefix = tmp

    def __repr__(self):
        cidr = '%s/%d' % (socket.inet_ntop(socket.AF_INET, self.prefix), self.len)
        return f'{self.__class__.__name__}({cidr})'

    def __len__(self):
        return self.__hdr_len__ + \
               (self.len + 7) // 8

    def __str__(self):
        return self.pack_hdr() + \
               self.prefix[:(self.len + 7) // 8]

class RouteIPV6(dpkt.Packet):
    __hdr__ = (
        ('len', 'B', 0),
        )

    def unpack(self, buf):
        dpkt.Packet.unpack(self, buf)
        tmp = self.data[:(self.len + 7) // 8]
        tmp += ( 16 - len( tmp ) ) * b'\x00'
        self.data = self.prefix = tmp

    def __repr__(self):
        cidr = '%s/%d' % (socket.inet_ntop(socket.AF_INET6, self.prefix), self.len)
        return f'{self.__class__.__name__}({cidr})'

    def __len__(self):
        return self.__hdr_len__ + \
               (self.len + 7) // 8

    def __str__(self):
        return self.pack_hdr() + \
               self.prefix[:(self.len + 7) // 8]

#IPv6 flowspec route yet to be tested once a BMP packet dump is available
class RouteFlowspecIPV6(dpkt.Packet):
    __hdr__ = (
        ('len', 'B', 0),
        ('offset', 'B', 0),
        )

    def unpack(self, buf):
        dpkt.Packet.unpack(self, buf)
        tmp = bytearray( ( self.offset // 8 ) * b'\x00' )
        num_shift = self.offset % 8
        shift_begin = len( tmp )
        tmp += self.data[:( (self.len - self.offset ) + 7) // 8]
        shift_end = len( tmp )
        while shift_begin < shift_end:
            tmp[shift_begin] = tmp[shift_begin] >> num_shift
            shift_begin += 1
        tmp += ( 16 - ( ( self.len + 7 ) // 8 ) ) * b'\x00'
        self.data = self.prefix = bytes( tmp )

    def __repr__(self):
        cidr = '%s/%d' % (socket.inet_ntop(socket.AF_INET6, self.prefix), self.len)
        return f'{self.__class__.__name__}({cidr})'

    #TODO decide what is expectation from __len__
    def __len__(self):
        return self.__hdr_len__ + \
               (self.len + 7) // 8

    #TODO decide what is expectation from __str__
    def __str__(self):
        return self.pack_hdr() + \
               self.prefix[:(self.len + 7) // 8]

class RouteIPV6Label( dpkt.Packet ):
    __hdr__ = (
        ( 'len', 'B', 0 ),
        )

    def unpack( self, buf ):
        dpkt.Packet.unpack( self, buf )
        l1 = struct.unpack_from( "B", self.data, 0 )[ 0 ]
        l2 = struct.unpack_from( "B", self.data, 1 )[ 0 ]
        l3 = struct.unpack_from( "B", self.data, 2 )[ 0 ]
        label_bytes = array.array( "B", [ 0, l1, l2, l3 ] )
        label = struct.unpack_from( ">L", label_bytes )[ 0 ]
        self.label = label >> 4
        self.label_data = self.data[ :3 ]
        self.len -= 24
        self.data = self.data[ 3: ]
        tmp = self.data[ :( self.len + 7 ) // 8 ]
        tmp += ( 16 - len( tmp ) ) * b'\x00'
        self.data = self.prefix = tmp

    def __repr__( self ):
       return "[%d] %s/%d" % ( self.label,
                               socket.inet_ntop( socket.AF_INET6, self.prefix ),
                               self.len )

    def __len__( self ):
        return self.__hdr_len__ + \
           ( self.len + 24 + 7 ) // 8

    def __str__( self ):
        return ( self.pack_hdr() + \
                 self.label_data + \
               self.prefix[ :( self.len + 7 ) // 8 ] )

class RouteIPV4Vpn( dpkt.Packet ):
   __hdr__ = (
       ( 'len', 'B', 0 ),
       )

   def unpack( self, buf ):
      dpkt.Packet.unpack( self, buf )
      l1 = struct.unpack_from( "B", self.data, 0 )[ 0 ]
      l2 = struct.unpack_from( "B", self.data, 1 )[ 0 ]
      l3 = struct.unpack_from( "B", self.data, 2 )[ 0 ]
      label_bytes = array.array( "B", [ 0, l1, l2, l3 ] )
      label = struct.unpack_from( ">L", label_bytes )[ 0 ]
      self.label = label >> 4
      self.label_data = self.data[ :3 ]
      self.data = self.data[ 3: ]
      rd = Tac.Value( 'Arnet::RouteDistinguisher' )
      rd.rdNbo = struct.unpack_from( "Q", self.data, 0 )[ 0 ]
      self.rd = rd
      self.rd_data = self.data[ :8 ]
      self.len -= 88
      self.data = self.data[ 8: ]
      tmp = self.data[ :( self.len + 7 ) // 8 ]
      tmp += ( 4 - len( tmp ) ) * b'\x00'
      self.data = self.prefix = tmp

   def __repr__( self ):
      return "[%d] %s %s/%d" % ( self.label,
                              self.rd.stringValue,
                              socket.inet_ntop( socket.AF_INET, self.prefix ),
                              self.len )

   def __len__( self ):
      return self.__hdr_len__ + \
         ( self.len + 24 + 64 + 7 ) // 8

   def __str__( self ):
      return ( self.pack_hdr() + \
               self.label_data + \
               self.rd_data + \
               self.prefix[ :( self.len + 7 ) // 8 ] )

class RouteIPV6Vpn( dpkt.Packet ):
   __hdr__ = (
       ( 'len', 'B', 0 ),
       )

   def unpack( self, buf ):
      dpkt.Packet.unpack( self, buf )
      l1 = struct.unpack_from( "B", self.data, 0 )[ 0 ]
      l2 = struct.unpack_from( "B", self.data, 1 )[ 0 ]
      l3 = struct.unpack_from( "B", self.data, 2 )[ 0 ]
      label_bytes = array.array( "B", [ 0, l1, l2, l3 ] )
      label = struct.unpack_from( ">L", label_bytes )[ 0 ]
      self.label = label >> 4
      self.label_data = self.data[ :3 ]
      self.data = self.data[ 3: ]
      rd = Tac.Value( 'Arnet::RouteDistinguisher' )
      rd.rdNbo = struct.unpack_from( "Q", self.data, 0 )[ 0 ]
      self.rd = rd
      self.rd_data = self.data[ :8 ]
      self.len -= 88
      self.data = self.data[ 8: ]
      tmp = self.data[ :( self.len + 7 ) // 8 ]
      tmp += ( 16 - len( tmp ) ) * b'\x00'
      self.data = self.prefix = tmp

   def __repr__( self ):
      return "[%d] %s %s/%d" % ( self.label,
                              self.rd.stringValue,
                              socket.inet_ntop( socket.AF_INET, self.prefix ),
                              self.len )

   def __len__( self ):
      return self.__hdr_len__ + \
         ( self.len + 24 + 64 + 7 ) // 8

   def __str__( self ):
      return ( self.pack_hdr() + \
               self.label_data + \
               self.rd_data + \
               self.prefix[ : ( self.len + 7 ) // 8 ] )

class FlowspecRoute( dpkt.Packet ):
   __hdr__ = (
       ( 'length', 'B', 0 ),
       )

   def unpack( self, buf ):
      dpkt.Packet.unpack( self, buf )
      if self.length >= 0xF0:
         byte2 = struct.unpack_from( "B", self.data, 0 )[ 0 ]
         self.length = ( ( self.length & 0x0F ) << 8 ) + byte2
         self.data = self.data[1 : self.length + 1]
      else:
         self.data = self.data[ : self.length]

      fl = []
      while self.data:
         filter_type = struct.unpack_from( "B", self.data, 0 )[ 0 ]
         if filter_type == FLOWSPEC_COMP_TYPE_DEST_PREFIX:
            if self.v6:
               Filter = FilterDestPrefixIPV6
            else:
               Filter = FilterDestPrefixIPV4
         elif filter_type == FLOWSPEC_COMP_TYPE_SRC_PREFIX:
            if self.v6:
               Filter = FilterSrcPrefixIPV6
            else:
               Filter = FilterSrcPrefixIPV4
         elif filter_type == FLOWSPEC_COMP_TYPE_IP_PROTO:
            Filter = FilterIpProto
         elif filter_type == FLOWSPEC_COMP_TYPE_IP_PORT:
            Filter = FilterIpPort
         elif filter_type == FLOWSPEC_COMP_TYPE_DEST_PORT:
            Filter = FilterDestPort
         elif filter_type == FLOWSPEC_COMP_TYPE_SRC_PORT:
            Filter = FilterSrcPort
         elif filter_type == FLOWSPEC_COMP_TYPE_ICMP_TYPE:
            Filter = FilterIcmpType
         elif filter_type == FLOWSPEC_COMP_TYPE_ICMP_CODE:
            Filter = FilterIcmpCode
         elif filter_type == FLOWSPEC_COMP_TYPE_TCP_FLAGS:
            Filter = FilterTcpFlags
         elif filter_type == FLOWSPEC_COMP_TYPE_PKT_LEN:
            Filter = FilterPktLen
         elif filter_type == FLOWSPEC_COMP_TYPE_DSCP_CODE_POINT:
            Filter = FilterDscpCodePoint
         elif filter_type == FLOWSPEC_COMP_TYPE_FRAGMENT:
            Filter = FilterFragment
         elif filter_type == FLOWSPEC_COMP_TYPE_FLOW_LABEL:
            Filter = FilterFlowLabel
         else:
            raise dpkt.UnpackError('invalid filter type')
         filterInfo = Filter( self.data[1: ] )
         self.data = self.data[ 1 + len(filterInfo): ]
         fl.append( filterInfo )
      self.data = self.filter_list = fl

   def __len__( self ):
      hdr_len_plus_one = 0
      if self.length >= 240:
         hdr_len_plus_one = 1
      return self.__hdr_len__ + hdr_len_plus_one + self.length

class RouteIPV4Flowspec( FlowspecRoute ):
   def __init__(self, *args, **kwargs):
      self.v6 = False
      dpkt.Packet.__init__(self, *args, **kwargs)

class RouteIPV6Flowspec( FlowspecRoute ):
   def __init__(self, *args, **kwargs):
      self.v6 = True
      dpkt.Packet.__init__(self, *args, **kwargs)

class FilterDestPrefixIPV4( RouteIPV4):
   pass

class FilterSrcPrefixIPV4( RouteIPV4 ):
   pass

class FilterDestPrefixIPV6( RouteFlowspecIPV6 ):
   pass

class FilterSrcPrefixIPV6( RouteFlowspecIPV6 ):
   pass

def OpLenFromNumericOrBitmaskOp( numeric_op ):
   op_len = ( numeric_op & 0x30 ) >> 4
   return 1 << op_len
 
class NumOpVal( dpkt.Packet ):
   __hdr__ = (
       ( 'numeric_op', 'B', 0 ),
       )

   def unpack( self, buf ):
      dpkt.Packet.unpack( self, buf )

      op_length = OpLenFromNumericOrBitmaskOp( self.numeric_op )
      if op_length == 1:
         self.op_val = struct.unpack_from( "B", self.data, 0 )[ 0 ]
      elif op_length == 2:
         self.op_val = struct.unpack_from( ">H", self.data, 0 )[ 0 ]
      elif op_length == 4:
         # Only used by IPv6 flow label filter defined in RFC8956
         self.op_val = struct.unpack_from( ">I", self.data, 0 )[ 0 ]
      else:
         raise dpkt.UnpackError('invalid filter operand length')

   def __repr__( self ):
      num_op_dict = {
            0: 'False', 1: '==', 2: '>', 3: '>=', 4: '<', 5: '<=', 6: '!=',
            7: 'True',
      }
      strval = num_op_dict[ self.numeric_op & 0x7 ]
      return strval + str( self.op_val )

   def __len__( self ):
      return self.__hdr_len__ + \
         OpLenFromNumericOrBitmaskOp( self.numeric_op )

class BitmaskOpVal( dpkt.Packet ):
   __hdr__ = (
       ( 'bitmask_op', 'B', 0 ),
       )

   def unpack( self, buf ):
      dpkt.Packet.unpack( self, buf )

      op_length = OpLenFromNumericOrBitmaskOp( self.bitmask_op )
      if op_length == 1:
         self.op_val = struct.unpack_from( "B", self.data, 0 )[ 0 ]
      elif op_length == 2:
         self.op_val = struct.unpack_from( ">H", self.data, 0 )[ 0 ]
      else:
         # RFC8955 and 8956 haven't defined any bitmaskfilters that use
         # op_length of 4 and 8
         raise dpkt.UnpackError('invalid filter operand length')

   def __repr__( self ):
      if self.bitmask_op & 1: # if match bit set, then (data AND value) == value
         strval = "&="
      else:
         strval ="~"          # if match bit reset, then any bit set in value => True
      if self.bitmask_op & 0x2:
         strval = "!"+strval
      return strval + str( self.op_val )

   def __len__( self ):
      return self.__hdr_len__ + \
         OpLenFromNumericOrBitmaskOp( self.bitmask_op )

class NumOpValList( dpkt.Packet ):
   __hdr__ = ()

   def unpack( self, buf ):
      self.data = buf
      vl = []

      while True:
         num_op_val = NumOpVal( self.data )
         self.data = self.data[ len( num_op_val ): ]
         vl.append( num_op_val )
         if ( num_op_val.numeric_op & 0x80 ) == 0x80:
            break
      self.data = self.value_list = vl

   def __repr__( self ):
      strval = self.component + "(" + repr( self.data[0] )
      for i in self.data[1:]:
         if i.numeric_op & 0x40:
            strval += ' && '
         else:
            strval += ' || '
         strval += repr(i)
      strval += ")"
      return strval

   def __len__( self ):
      return self.__hdr_len__ + sum( len( i ) for i in self.data )

class BitmaskOpValList( dpkt.Packet ):
   __hdr__ = ()

   def unpack( self, buf ):
      self.data = buf
      vl = []

      while True:
         bitmask_op_val = BitmaskOpVal( self.data )
         self.data = self.data[ len( bitmask_op_val ): ]
         vl.append( bitmask_op_val )
         if ( bitmask_op_val.bitmask_op & 0x80 ) == 0x80:
            break
      self.data = self.value_list = vl

   def __repr__( self ):
      strval = self.component + "(" + repr( self.data[0] )
      for i in self.data[1:]:
         if i.numeric_op & 0x40:
            strval += ' && '
         else:
            strval += ' || '
         strval += repr(i)
      strval += ")"
      return strval

   def __len__( self ):
      return self.__hdr_len__ + sum( len( i ) for i in self.data )

class FilterIpProto( NumOpValList ):
   def __init__(self, *args, **kwargs):
      dpkt.Packet.__init__(self, *args, **kwargs)
      self.component = "IpProto"

class FilterIpPort( NumOpValList ):
   def __init__(self, *args, **kwargs):
      dpkt.Packet.__init__(self, *args, **kwargs)
      self.component = "IpPort"

class FilterDestPort( NumOpValList ):
   def __init__(self, *args, **kwargs):
      dpkt.Packet.__init__(self, *args, **kwargs)
      self.component = "DstPort"

class FilterSrcPort( NumOpValList ):
   def __init__(self, *args, **kwargs):
      dpkt.Packet.__init__(self, *args, **kwargs)
      self.component = "SrcPort"

class FilterIcmpType( NumOpValList ):
   def __init__(self, *args, **kwargs):
      dpkt.Packet.__init__(self, *args, **kwargs)
      self.component = "IcmpType"

class FilterIcmpCode( NumOpValList ):
   def __init__(self, *args, **kwargs):
      dpkt.Packet.__init__(self, *args, **kwargs)
      self.component = "IcmpCode"

class FilterTcpFlags( BitmaskOpValList ):
   def __init__(self, *args, **kwargs):
      dpkt.Packet.__init__(self, *args, **kwargs)
      self.component = "TcpFlags"

class FilterPktLen( NumOpValList ):
   def __init__(self, *args, **kwargs):
      dpkt.Packet.__init__(self, *args, **kwargs)
      self.component = "PktLen"


class FilterDscpCodePoint( NumOpValList ):
   def __init__(self, *args, **kwargs):
      dpkt.Packet.__init__(self, *args, **kwargs)
      self.component = "DscpCodePoint"


class FilterFragment( BitmaskOpValList ):
   def __init__(self, *args, **kwargs):
      dpkt.Packet.__init__(self, *args, **kwargs)
      self.component = "Fragment"


class FilterFlowLabel( NumOpValList ):
   def __init__(self, *args, **kwargs):
      dpkt.Packet.__init__(self, *args, **kwargs)
      self.component = "FlowLabel"


