# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import struct
from uuid import UUID
# pylint: disable=c-extension-no-member
import lzma as xz

import BiosToolsLib

# Constants from UEFI PI Spec

# Offset of 'signature' in EFI_FIRMWARE_VOLUME_HEADER struct
# ZeroVector - 16 bytes, EFI_GUID - 16 bytes, FvLength - 8 bytes
EFI_FVH_SIGNATURE_OFFSET = 16 + 16 + 8
EFI_FVH_SIGNATURE = b"_FVH"

EFI_SECTION_GUID_DEFINED = 0x02
EFI_SECTION_RAW = 0x19
EFI_FIRMWARE_FILE_SYSTEM2_GUID = "8c8ce578-8a3d-4f1c-9935-896185c32dd3"

BLOCK_HEADER_SIZE = 8
SECTION_HEADER_SIZE = 4
GUID_DEFINED_HEADER_SIZE = 20
FILE_HEADER_SIZE = 0x18
VOLUME_HEADER_SIZE = 0x38

# EDK2 MdeModulePkg/Include/Guid/LzmaDecompress.h
LZMA_CUSTOM_DECOMPRESS_GUID = "ee4e5898-3914-4259-9d6e-dc7bd79403cf"

# Aboot10 BmFv/UserFiles.fdf
# GUID - "41626f6f-742d-692e-e06e-736800000000"
ABOOT_INITRD_FREEFORM_NAME = b"Aboot-i.nsh"
# BlinkBoot BbWallabyMultiBoardPkg/Project.dec
ABOOT_VOLUME_GUID = "c1b5530c-8107-4f18-b951-20b7e5ce3e2b"


def guidToStr( guid ):
   return str( UUID( bytes_le=guid ) )

# decompiled from BaseTools/GenFds/FdfParser.py
def nameToGuid( name ):
   guid = [ '00', '00', '00', '00', '-',
            '00', '00', '-',
            '00', '00', '-',
            'e0', '00', '-',
            '00', '00', '00', '00', '00', '00' ]
   for c in name:
      for i, v in enumerate( guid ):
         if isinstance( c, str ):
            c = ord( c )
         if v == '00':
            guid[ i ] = hex( c )[ 2 : ]
            break
   return ( '' ).join( guid )

def findFileSystemV2InVolume( data ):
   fsv2 = []

   volOffset = 0
   while True:
      volOffset = data.find( EFI_FVH_SIGNATURE, volOffset )
      if volOffset < 0:
         break

      volumeData = data[ volOffset - EFI_FVH_SIGNATURE_OFFSET : ]
      try:
         header = volumeData[ : VOLUME_HEADER_SIZE ]
         # pylint: disable=unused-variable
         rsvd, guid, size, sign, atts, hdrLen, checksum, extHdrOff, rsvd2, \
               revision = struct.unpack( "<16s16sQ4sIHHHsB", header )
      except Exception: # pylint: disable=broad-except
         pass
      else:
         if sign == EFI_FVH_SIGNATURE and \
               guidToStr( guid ) == EFI_FIRMWARE_FILE_SYSTEM2_GUID:
            fsv2.append( ( volumeData[ VOLUME_HEADER_SIZE : hdrLen ],
                           volumeData[ hdrLen : size ] ) )

      volOffset += len( EFI_FVH_SIGNATURE )

   return fsv2

def parseAbootSection( sectionData, sectionType ):
   while len( sectionData ) >= SECTION_HEADER_SIZE:
      sectionHeader = sectionData[ : SECTION_HEADER_SIZE ]
      sectionSize, sectionType = struct.unpack( "<3sB", sectionHeader )
      sectionSize = struct.unpack( "<I", sectionSize + b"\x00" )[ 0 ]
      if sectionType == sectionType:
         return sectionData[ SECTION_HEADER_SIZE : sectionSize ]

      if sectionSize <= 0:
         return None

      sectionData = sectionData[ ( sectionSize + 3 ) & ( ~3 ) : ]

def findGuidInFileSystemBlocks( blocksMap, data, guidToFind ):
   res = []
   while len( blocksMap ) >= BLOCK_HEADER_SIZE:
      block = blocksMap[ : BLOCK_HEADER_SIZE ]
      blockSize, blockLength = struct.unpack( "<II", block )
      if ( blockSize, blockLength ) == ( 0, 0 ):
         # The block map ends with a (0, 0) block
         break

      blockData = data[ : blockSize * blockLength ]
      while len( blockData ) >= FILE_HEADER_SIZE and \
            blockData[ : FILE_HEADER_SIZE ] != ( b"\xff" * FILE_HEADER_SIZE ):
         # pylint: disable=unused-variable
         guid, checksum, fileType, atts, size, state = \
               struct.unpack( "<16sHBB3sB", blockData[ : FILE_HEADER_SIZE ] )
         size = struct.unpack( "<I", size + b"\x00" )[ 0 ]
         if guidToStr( guid ) == guidToFind:
            res.append( blockData[ FILE_HEADER_SIZE : size ] )

         if size < FILE_HEADER_SIZE:
            # The data were corrupted.
            break

         blockData = blockData[ ( size + 7 ) & ( ~7 ) : ]

      blocksMap = blocksMap[ BLOCK_HEADER_SIZE : ]
      data = data[ blockSize * blockLength : ]

   return res

def parseGuidDefinedSection( section ):
   # pylint: disable=unused-variable
   guid, offset, attrMask = \
      struct.unpack( "<16sHH", section[ : GUID_DEFINED_HEADER_SIZE ] )
   if guidToStr( guid ) != LZMA_CUSTOM_DECOMPRESS_GUID:
      return None

   try:
      decompressed = xz.decompress( section[ GUID_DEFINED_HEADER_SIZE : ] )
   except Exception: # pylint: disable=broad-except
      return None

   ABOOT_INITRD_GUID = nameToGuid( ABOOT_INITRD_FREEFORM_NAME )

   abootInitrd = None
   fsv2 = findFileSystemV2InVolume( decompressed )
   for blocksMap, blockData in fsv2:
      abootInitrdSections = findGuidInFileSystemBlocks( blocksMap, blockData,
                                                        ABOOT_INITRD_GUID )
      for sec in abootInitrdSections:
         abootInitrd = parseAbootSection( sec, EFI_SECTION_RAW )
         if abootInitrd:
            break

   return abootInitrd

def getUnzippedAbootImages( data ):
   images = []
   fsv2 = findFileSystemV2InVolume( data )
   for blocksMap, blockData in fsv2:
      abootSections = findGuidInFileSystemBlocks( blocksMap, blockData,
                                                  ABOOT_VOLUME_GUID )
      abootVolume = None
      for sec in abootSections:
         abootVolume = parseAbootSection( sec, EFI_SECTION_GUID_DEFINED )
         if abootVolume:
            break

      if abootVolume:
         unzippedAbootImage = parseGuidDefinedSection( abootVolume )
         if unzippedAbootImage:
            images.append( unzippedAbootImage )
   return images

def getAbootVersion( data ):
   abootImages = getUnzippedAbootImages( data )

   for abootImage in abootImages:
      version = BiosToolsLib.getVersionFromInitrd( abootImage )
      if version:
         return version

   return None
