#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac
import FpgaUtil
import os
from enum import Enum

class BmcType( Enum ):
   NONE = 0
   LITE = 1
   HEAVY = 2

def enableBmcUsb( timeout=20 ):
   print( "Enabling BMC USB interface" )
   try:
      Tac.waitFor( lambda: os.path.exists(
                   "/proc/sys/net/ipv6/conf/usb0/disable_ipv6" ),
                   description="BMC usb0 interface to show up",
                   timeout=timeout )
   except Tac.Timeout:
      print( f"usb0 device not detected after {timeout} seconds. Not enabling BMC "
             "usb interface" )
      return

   try:
      Tac.run( [ "sysctl", "--quiet", "--write",
                 "net.ipv6.conf.usb0.disable_ipv6=0" ], asRoot=True )
      Tac.run( [ "ifconfig", "usb0", "up" ], asRoot=True )
      Tac.run( [ "ip", "-6", "address", "add", "fe80::2/64", "dev", "usb0" ],
               asRoot=True )
   except Tac.SystemCommandError:
      print( "Failed to enable BMC usb interface." )
      return

   print( "Enabled BMC usb interface" )

def checkBmcModeClosure( scdPciAddr, bmcType ):
   '''
   A closure is used here so that each individual product can pass in their
   own scd/pci mapping and bmc type
   '''
   def checkBmcMode( context ):
      '''
      Reads the BmcMode bit from the pci address. Enables the BMC usb0 connection
      if the bit is set, and also sets the bmc product type if applicable.
      '''
      try:
         output = Tac.run( [ "scd", scdPciAddr, "read",
                             str( 0x8 ) ],
                             stdout=Tac.CAPTURE, asRoot=True )
      except Tac.SystemCommandError:
         context.bmcType = BmcType.NONE
         return

      val = FpgaUtil.Fpga.parseRegVal( output, 0x8 )
      if ( ( val >> 30 ) & 0x1 ) == 0:
         context.bmcType = BmcType.NONE
      else:
         context.bmcType = bmcType

      if context.bmcType != BmcType.NONE:
         enableBmcUsb()

   return checkBmcMode
