#!/usr/bin/env python3

# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import print_function
from __future__ import absolute_import

import os
import textwrap
import platform
import Tac
import socket
import subprocess

import CEosHelper

def genKickStartConfig( disabledAgents=None, filePath=None, enableCapi=False,
                        enableHostSyslog=False,
                        enableRoutingModelMultiAgent=False,
                        disableSpanningTree=False,
                        disableLldp=False,
                        enableRouting=False,
                        enableRouting6=False,
                        enableRoutedMode=False,
                        configHostname=False,
                        enableALPMMode=False,
                        enableDsfClusterConfig=False ):
   if disabledAgents is None:
      disabledAgents = []

   if not filePath:
      filePath = os.path.join( "/", "mnt", "flash", "kickstart-config" )

   with open( filePath, "w" ) as f:
      for agent in disabledAgents:
         # pylint: disable-next=consider-using-f-string
         f.write( "agent %s shutdown\n" % agent )

      if enableCapi:
         # Enable http-server on localhost
         f.write( textwrap.dedent( """ 
         management api http-commands
            no shutdown
         !
         management http-server
            protocol http localhost
         !
         """ ) )

      if enableHostSyslog:
         # Enable sync warning events to SONiC
         f.write( textwrap.dedent( """ 
         logging on
         logging trap Warning
         logging host 127.0.0.1
         match-list input string cEosEvents
            match regex Thermostat
         !
         logging policy match invert-result match-list cEosEvents discard
         """ ) )

      if enableRoutingModelMultiAgent:
         # Enable multi-agent routing protocols model
         f.write( textwrap.dedent( """
         service routing protocols model multi-agent
         """ ) )

      if disableSpanningTree:
         # Disable spanning-tree protocol.
         f.write( textwrap.dedent( """
         spanning-tree mode none
         """ ) )

      if disableLldp:
         # Disable LLDP protocol.
         f.write( textwrap.dedent( """
         no lldp-run
         """ ) )

      if enableRouting:
         f.write( textwrap.dedent( """
         ip routing
         """ ) )

      if enableRouting6:
         f.write( textwrap.dedent( """
         ipv6 unicast-routing
         """ ) )

      if enableRoutedMode:
         f.write( textwrap.dedent( """
         switchport default mode routed
         """ ) )

      if configHostname:
         # pylint: disable-next=consider-using-f-string
         f.write( textwrap.dedent( """
         hostname %s
         """ % socket.gethostname() ) )

      if enableALPMMode:
         f.write( textwrap.dedent( """
         platform trident forwarding-table partition 4
         """ ) )

      if enableDsfClusterConfig:
         # Enable DSF cluster config.
         dsfSystemId = int( os.environ.get( "SYSTEM_ID" ) )
         # We support system ids in the range [ 0 - 127 ].
         # pylint: disable-next=chained-comparison
         assert dsfSystemId >= 0 and dsfSystemId < 128

         if int( dsfSystemId ) % 2 == 0:
            internalVlanRange = "1006 2005"
         else:
            internalVlanRange = "2006 3005"

         # pylint: disable-next=consider-using-f-string
         f.write( textwrap.dedent( """
         switch cluster voq
            device system-id %d
            cpu interface Loopback0
            !
            device leaf defaults
               interface profile simulation-lyonsville
            !
            device leaf system-id 0
               name LD0
               system port range 128-132
            !
            device leaf system-id 1 
               name LD1
               system port range 133-137
            !
         !
         hostname LD%d
         !
         interface Loopback0
         !
         vlan internal order ascending range %s
         !
         end
         """ % ( dsfSystemId, dsfSystemId, internalVlanRange ) ) )


def genDmamemContainerConfig():
   filePath = os.path.join( "/", "etc", "dmamem_container_config" )
   containerId = os.environ.get( "SYSTEM_ID" )
   with open( filePath, "w" ) as f:
      f.write( "CONTAINER_ID=" + containerId + "\n" )

def genDsfConfig():
   filePath = os.path.join( "/", "etc", "dsf_config" )
   dsfSystemId = os.environ.get( "SYSTEM_ID" )
   with open( filePath, "w" ) as f:
      f.write( "DSF_FAP=" + dsfSystemId + "\n" )


## Handle config passed in by container orchestrator. Append the configuration to
## kickstart file
def handleExtKickstart( kickstartConfigFile=None, extConfigFile=None ):
   if not extConfigFile:
      extConfigFile = os.path.join( "/", "mnt", "flash", "config",
                                    "kickstart-config" )
   if not kickstartConfigFile:
      kickstartConfigFile = os.path.join( "/", "mnt", "flash",
                                          "kickstart-config" )
   with open( kickstartConfigFile, "a" ) as kickstartFd:
      with open( extConfigFile, "r" ) as extConfigFd:
         kickstartFd.write( extConfigFd.read() )

def setLinuxKernelForwarding( veosConfigFile=None ):
   if not veosConfigFile:
      veosConfigFile = os.path.join( "/", "mnt", "flash", "veos-config" )
   with open( veosConfigFile, "w+" ) as f:
      f.write( "MODE=linux" )

def runningKernelVersion():
   return platform.uname()[ 2 ]

def installedKernelVersion():
   return subprocess.check_output( [ "rpm", "-q", "EosKernel",
      "--queryformat='%{VERSION}-%{RELEASE}'" ], universal_newlines=True )

# We need to verify that cEOS is running on the same
# kernel version as the binary of the kernel that cEos
# is bundled with.
def checkKernelVersion():
   # Check the running Kernel with the binary
   versionMatch = False
   try:
      versionMatch = runningKernelVersion() == installedKernelVersion()
   finally:
      # Ignore exception due to missing rpms, etc
      pass
   return versionMatch

def applyCeosMode( mode ) :
   if mode.checkKernelVersion:
      assert checkKernelVersion(), "Kernel version check failed for cEOS"

   if mode.generateKickStart:
      blacklistedAgents = os.getenv( 'EOS_AGENT_BLACKLIST', "" )
      blacklistedAgents = [ agent for agent in blacklistedAgents.split( ',' )
                            if agent != "" ]
      disabledAgents = mode.disabledAgents + blacklistedAgents

      genKickStartConfig(
         disabledAgents=disabledAgents,
         enableCapi=mode.enableCapi,
         enableHostSyslog=mode.enableHostSyslog,
         enableRoutingModelMultiAgent=mode.enableRoutingModelMultiAgent,
         disableSpanningTree=mode.disableSpanningTree,
         disableLldp=mode.disableLldp,
         enableRouting=mode.enableRouting,
         enableRouting6=mode.enableRouting6,
         enableRoutedMode=mode.enableRoutedMode,
         configHostname=mode.configHostname,
         enableALPMMode=mode.enableALPMMode,
         enableDsfClusterConfig=mode.enableDsfClusterConfig
      )

   if mode.extKickstart:
      handleExtKickstart()

   if mode.linuxKernelForwarding:
      setLinuxKernelForwarding()

   if mode.genDmamemContainerConfig:
      genDmamemContainerConfig()

   if mode.genDsfConfig:
      genDsfConfig()

def createSaiDefaultIntfs():
   for intf in [ 'cpu', 'vxlan', 'fabric', 'internal1_1', 'internal1_2' ]:
      Tac.run( [ 'ip', 'link', 'add', intf, 'type', 'dummy' ],
               stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      Tac.run( [ 'ip', 'link', 'set', intf, 'up' ],
               stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )

   # When dummy fabric netdevice is created, mtu will be set to 1500.
   # Since fabric mtu is 1500, we will not able to configure mtu >1500
   # for SVIs. To fix this, we set mtu of dummy fabric netdevice to
   # EOS fabric mtu.
   Tac.run( [ 'ip', 'link', 'set', 'fabric', 'mtu', '9216' ],
            stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )

def CEosInit():
   if not CEosHelper.isCeos():
      return

   applyCeosMode( CEosHelper.getCeosMode() )
   if os.environ.get( "CEOS_NAMESPACE" ):
      createSaiDefaultIntfs()
