# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import hashlib
import json
import os

import SwimHelperLib

HASH_ALGOS = [ 'md5', 'sha1', 'sha256', 'sha512' ]

def getHashes( image ):
   def generateHash( image, algoObj ):
      with open( image, 'rb' ) as fyle:
         blockSize = 65536
         for block in iter( lambda: fyle.read( blockSize ), b'' ):
            algoObj.update( block )
      return algoObj.hexdigest()

   hashes = {}
   # 'getattr( hashlib, algo )()' will call hashlib.algo()
   for algo, algoObj in iter( { algo: getattr( hashlib, algo )()
                                for algo in HASH_ALGOS }.items() ):
      hashes[ algo ] = generateHash( image, algoObj )
   return hashes

def getHashesFromFile( jsonFile ):
   """Reads checksums from provided json file of full image and all optimizations
   and returns them in form of a dictionary.

   Args:
      jsonFile: str: path to checksums json file

   Returns:
      dict: { optimization: { 'sha512': value, 'md5sum': value } }
   """
   with open( jsonFile ) as f:
      jsonObj = json.load( f )

   # Checksums json will be in one of two possible formats:
   # 1) <image>.checksums.json
   #    - { 'images': <checksums> }
   # 2) <image>.json
   #    - { '<imageName>':
   #        { 'images': <checksums> }
   #      }
   if 'images' in jsonObj:
      imageChecksums = jsonObj[ 'images' ]
   elif len( jsonObj ) == 1:
      imageHdr = list( jsonObj.values() )[ 0 ]
      imageChecksums = imageHdr[ 'images' ]
   else:
      raise Exception( "Invalid format json with keys: %s" %
                       " ".join( jsonObj.keys() ) )

   opt2checksums = {}
   for optimization, c_sums in imageChecksums.items():
      opt2checksums[ optimization ] = \
            { algo: c_sums[ 'checksums' ][ algo ] for algo in HASH_ALGOS }

   return opt2checksums

def saveImageChecksums( image, checksums ):
   topLevelDescription = "This JSON contains {} checksums of the image/file.".format(
                              ", ".join( HASH_ALGOS ) )

   assert "Full" in checksums, \
      "Unexpected format of checksums data - missing 'Full' image"
   assert len( checksums ) == 1, \
      "Unexpected format of checksums data - multiple image checksums found"

   checksums[ "Full" ][ "description" ] = "The checksums of the image/file."
   data = { "images": checksums, "description": topLevelDescription }
   with open( "%s.checksums.json" % image, "w" ) as f:
      f.write( json.dumps( data, indent=4, separators=( ',', ': ' ) ) )

def saveSwimChecksums( image, checksums ):
   topLevelDescription = (
      "This JSON contains {} checksums of the Modular image. "
      "Modular EOS images contain multiple internal images, and the bits not needed "
      "by the target platform can be dropped (optimized/adapted) while the image is "
      "downloaded to the target during the 'install source <url>' CLI." ).format(
                                             ", ".join( HASH_ALGOS ) )

   for optim in sorted( checksums ):
      if optim == "Full":
         description = (
            "The full/unoptimized image containing the bits and pieces of all "
            "optimized images (the single image published on "
            "software-download page)." )
      else:
         description = "The image optimized into the %s optimization." % optim
      checksums[ optim ][ "description" ] = description

   data = { "images": checksums, "description": topLevelDescription }
   with open( "%s.checksums.json" % image, "w" ) as f:
      f.write( json.dumps( data, indent=4, separators=( ',', ': ' ) ) )

def genImageChecksums( imagePath ):
   imageChecksums = {}
   imageChecksums[ "Full" ] = { "checksums": getHashes( imagePath ) }
   saveImageChecksums( imagePath, imageChecksums )

def genSwimImageChecksums( imagePath, workDir ):
   imageChecksums = {}

   optims = SwimHelperLib.getSupportedOptimizationsFromSwi( imagePath )
   assert optims, "Error: legacy image found when SWIM format expected"
   for optim in optims:
      # Don't generate checksums for Default since
      # we don't adapt a full image into Default
      if optim.startswith( 'Default' ):
         continue
      optimImage = f"{workDir}/{optim}.swi"
      # Adapt image
      os.system( f"swadapt {imagePath} {optimImage} {optim}" )

      # Collect the checksums of the image
      imageChecksums[ optim ] = { "checksums": getHashes( optimImage ) }

   # Collect the checksums of the full image
   imageChecksums[ "Full" ] = { "checksums": getHashes( imagePath ) }
   saveSwimChecksums( imagePath, imageChecksums )
