# Copyright (c) 2005-2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno
import os
import pickle
import re
import sys
import threading
import traceback


import BasicCli
import CliCommon
import CliInterface
import CliShellLib
import FastServUtil
import MainCli
import QuickTrace
import SecMonUtil
import Syscall
import Tac
import Tracing
import TerminalUtil

traceHandle = Tracing.Handle( 'Cli' )
log = traceHandle.trace0
warn = traceHandle.trace1
info = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1
qv = QuickTrace.Var

def initCli( entityManager, callback, block=True,
             plugins=None, noPlugins=None, standalone=False, allowActiveMounts=True,
             runSessionManager=False, redundancyStatus=None, agent=None ):
   if entityManager is not None:
      # Disable profile unloading in MountFileInfo. This gives a small persistent
      # memory hit (~2MB), but prevents the profile data from thrashing every time a
      # Cli command is run. This is done on agent construction in ConfigAgent.py, but
      # it's also done here to set the same behavior in tests running without a
      # fully-fledged ConfigAgent
      entityManager.cEm_.mountFileInfoWrap.disableUnDoInit()
   info( 'cliSessionMountsComplete' )
   mainCli = MainCli.MainCli( entityManager,
                              plugins=plugins,
                              noPlugins=noPlugins,
                              standalone=standalone,
                              allowActiveMounts=allowActiveMounts,
                              runSessionManager=runSessionManager,
                              redundancyStatus=redundancyStatus,
                              agent=agent,
                              callback=callback )
   if block:
      Tac.waitFor( mainCli.pluginsComplete,
                   description='plugins to load' )

def runFrontendCmds( session, frontendSock ):
   currThread = threading.current_thread()
   tid = Syscall.gettid()
   while True:
      requestData = FastServUtil.readBytes( frontendSock )
      if not requestData:
         break
      # pylint: disable=no-member
      request = pickle.loads( requestData )
      trace( "PROCESSING METHOD", request.method_ )
      if request.method_ == 'endSession':
         FastServUtil.writeBytes(
            frontendSock,
            pickle.dumps( None, protocol=FastServUtil.PICKLE_PROTO ) )
         sys.stdout.flush()
         sys.stderr.flush()
         break
      procedure = getattr( session, request.method_ )
      try:
         trace( "running METHOD", request.method_ )
         if request.method_ == 'runCmd':
            qt1( qv( tid ), 'runCmd command', qv( request.args_[ 0 ] ) )
         response = procedure( *request.args_, **request.kwargs_ )
         trace( "done METHOD", request.method_, "response", response )
         if request.method_ == 'runCmd':
            qt1( qv( tid ), 'done runCmd command', qv( request.args_[ 0 ] ) )
      except: # pylint: disable-msg=W0702
         excInfo = sys.exc_info()
         if ( request.method_ in ( 'runCmd', 'tabComplete', 'printHelp' ) and
              excInfo[ 0 ] is not SystemExit ):
            session.handleCliException( excInfo, request.args_[ 0 ] )
            response = None
         else:
            if excInfo[ 0 ] is not SystemExit:
               os.write( 2, ( "Stack trace from processing frontend cmd " +
                             request.method_ + ": " +
                             re.sub( "Traceback[^:]*:..", "",
                                     repr( traceback.format_exc() ) ) +
                             '\n' ).encode() )
            response = sys.exc_info()[ 1 ]
         trace( "exception METHOD", request.method_, "response", response )

      if hasattr( currThread, 'killChildThreads' ) and request.method_ == "runCmd":
         # if we have a thread which can kill children, lets do it.
         # this is for backward compatability with FastClid
         currThread.killChildThreads() # pylint: disable-msg=E1103
      sys.stdout.flush()
      sys.stderr.flush()
      # pylint: disable=no-member
      responseData = pickle.dumps(
         response,
         protocol=FastServUtil.PICKLE_PROTO )
      FastServUtil.writeBytes( frontendSock, responseData )

def parseOptions( options ):
   if options.pdb:
      print( "Attaching pdb to Cli" )
      import pdb # pylint: disable=import-outside-toplevel
      pdb.set_trace() # pylint: disable=forgotten-debug-statement

   privLevel = options.privilege
   initialMode = BasicCli.UnprivMode
   if options.input_file:
      fin = options.input_file
      # force privileged mode
      privLevel = CliCommon.MAX_PRIV_LVL
      initialMode = BasicCli.EnableMode
   elif options.command is None:
      fin = sys.stdin
   else:
      from io import StringIO # pylint: disable=import-outside-toplevel
      fin = StringIO( options.command )

   if options.echoTimestamps:
      echo = 'E'
   elif options.echo:
      echo = 'e'
   else:
      echo = ''

   if options.startup_config:
      initialMode = BasicCli.GlobalConfigMode

   return privLevel, initialMode, fin, echo

def main( cli, entityManager, session, options, frontendSock=None ):
   privLevel, initialMode, fin, echo = parseOptions( options )
   session.initTerminal()
   TerminalUtil.ttyHook.notifyExtensions( session )

   errors = 0
   returnCode = 0
   if options.completions:
      session.printHelp( options.completions )
   elif fin.isatty():
      if frontendSock:
         runFrontendCmds( session, frontendSock )
      else:
         cliInterface = CliInterface.LocalCliInterface( session )
         cliShell = CliShellLib.CliShell( cliInterface )
         cliShell.startReadlineLoop()
   else:
      trace( "We don't have a TTY! Running loadConfigFromFile" )
      try:
         errors = MainCli.loadConfigFromFile( fin,
                                      entityManager,
                                      disableAaa=options.disable_aaa,
                                      initialModeClass=initialMode,
                                      echo=echo,
                                      privLevel=privLevel,
                                      disableGuards=options.disable_guards,
                                      startupConfig=options.startup_config,
                                   secureMonitor=SecMonUtil.getCliOption( options ),
                                      autoComplete=not options.disable_autocomplete,
                                      aaaUser=session.aaaUser(),
                                      abortOnError=options.abort_on_error )
      except KeyboardInterrupt:
         returnCode = errno.EINTR
      except CliCommon.LoadConfigError as e:
         print( e )
         return 1
      finally:
         trace( "Done running loadConfigFromFile" )
         sys.stdout.flush()
         sys.stderr.flush()

   if errors != 0:
      if options.startup_config:
         print( 'errors in startup-config:', errors )
      return 1
   return returnCode
