#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliParserCommon import * # pylint: disable-msg=W0401

# Some common definitions of Cli.

class JsonRpcErrorCodes:
   """ Error codes used in JSON RPC error responses """

   # DO NOT change these numbers unless you know what you are doing:
   # these are part of the public API.

   # Predefined error codes:
   PARSE_ERROR = -32700
   INVALID_REQUEST = -32600
   METHOD_NOT_FOUND = -32601
   INVALID_PARAMS = -32602
   INTERNAL_ERROR = -32603

   # Custom error codes:

   # Command execution failed:
   COMMAND_FAILED = 1000 # General error
   COMMAND_EXCEPTION = 1001 # InternalError
   COMMAND_INVALID = 1002 # String does not match any command
   COMMAND_UNCONVERTED = 1003 # Command has not been converted yet
   COMMAND_INCOMPATIBLE = 1004 # Marked as not compatible with CAPI
   COMMAND_UNAUTHORIZED = 1005 # Insufficient permissions to run the command
   COMMAND_CONFIG_LOCKED = 1006 # Configuration is locked

class EditConfigOptions:
   """ Defined the possible edit config options and the default options.
   The options below are as per RFC 6241 Section 7.2"""

   OPERATION_MERGE = 'merge'
   OPERATION_REPLACE = 'replace'

   TEST_OPTION_TEST_THEN_SET = 'testThenSet'
   TEST_OPTION_TEST_ONLY = 'testOnly'
   TEST_OPTION_SET = 'set'

   DEFAULT_OPERATION = OPERATION_MERGE
   DEFAULT_TEST_OPTION = TEST_OPTION_TEST_THEN_SET

   VALID_OPERATIONS = ( OPERATION_MERGE, OPERATION_REPLACE )
   VALID_TEST_OPTIONS = ( TEST_OPTION_TEST_THEN_SET, TEST_OPTION_SET,
                           TEST_OPTION_TEST_ONLY )

class ResponseFormats:
   """ Defines possible response formats that a client can request """

   JSON = "json"
   TEXT = "text"

class FastClidServerConstants:
   """ Unix Domain Socket address to initiate Cli connections """
   FASTCLID_HELPER_ADDRESS_FMT = "\x00/FastClidHelper/%s"

CLI_SERVER_AGENT_NAME = "ConfigAgent"
CLI_SERVER_ADDRESS_FMT = '\x00/CliServer/%s'
CLI_CTRL_ADDRESS_FMT = '\x00/CliCtrl/%s'

# jsonrpc's version is normally an integer, but we also allow it to be the exact
# string "latest", which is internally converted to 0 (an invalid version).
JSONRPC_VERSION_LATEST = 0

# Range of acceptable values for privilege levels.
MIN_PRIV_LVL = 0
DEFAULT_PRIV_LVL = 1
MAX_PRIV_LVL = 15

SECURE_MONITOR_ATTR_NAME = "secureMonitor"

PRIV_CMD_IN_UNPRIV_MODE_ERROR_MSG = "(privileged mode required)"

# Mandatory attributes that are missing in returned model instances create
# warnings on the cli, this is the warning text (test infra converts those
# into asserts).
SHOW_OUTPUT_MISSING_ATTRIBUTE_WARNING = "missing attribute"

# json attributes found in the output but not declared in the model will
# generate warnings, this is the warning text (test infra converts those
# warnings into asserts).
SHOW_OUTPUT_UNEXPECTED_ATTRIBUTE_WARNING = "unexpected attribute"

# The directory into which temporary CLI files are written
# This is mostly used to support redirection of CLI show commands
CLI_TMP_DIR = '/var/tmp/cli'

# This is the max size of the history table in cli/configHistory
HISTORY_TABLE_LEN = 32

# Number of Cli commands to execute with auto config session enabled.
# After that, Cli commands will be executed with auto config session disabled.
AUTO_CONFIG_SESSION_NUM_CMDS = 100

UNCONVERTED_MODEL_MSG = "Command cannot be used over the API at this time. " \
                        "To see ASCII output, set format='text' in your request"

skippedJsonCmds = [ 'show rib ready vrf all',
                        # these have outputs from tac.waitfor
                        'show lacp peer detailed all-ports',
                        'show lacp internal detailed all-ports',
                        'show lacp counters all-ports',
                        'show ip igmp snooping counters detail',
                        'show mld snooping counters',
                        'show mld snooping counters errors',

                        # these have output of '! BGP inactive'
                        'show ip bgp vrf all',
                        'show ipv6 bgp vrf all',
                        'show ip bgp summary vrf all',
                        'show ipv6 bgp summary vrf all',
                        'show ip bgp neighbor vrf all',
                        'show ipv6 bgp peers',
                        'show ipv6 bgp peers vrf all',
                        'show ipv6 bgp neighbor',
                        'show maintenance bgp ip all vrf all',
                        'show maintenance bgp ipv6 all vrf all',
                        'show bgp sr-te detail vrf all',
                        'show bgp sr-te summary vrf all',

                        # prints out ! Rib Agent inactive
                        'show dynamic prefix-List vrf all',

                        # prints % This is an unconverted command
                        # this is related to BUG214391
                        'show bgp statistics',

                        # these don't print out anything
                        'show ipv6 route host',
                        'show ip route host',
                        'show traffic-policy cpu',
                        'show traffic-policy cpu errors',
                        'show mac address-table',
                        'show arp',
                        'show ipv6 neighbor',
                        'show ip route vrf all detail',
                        'show ipv6 route vrf all detail',

                        # these have extra output aside from their models
                        'show interfaces switchport',
                        'show ntp associations',
                        'dir /recursive flash:', # prints Directory of flash:/

                        # thse are printing out additional stuff over eAPI
                        'show platform fap fabric detail',
                        'show platform fap ip route',
                        'show platform fap ipv6 route',
                        'show platform fap ip pbr nexthop',
                        'show platform fap subinterface mapping',

                        # not sure why these fail on a dut fu313
                        'show interfaces counters queue | nz',
                        'show vrf',
                        'show interfaces all phy detail',
                        'show interfaces phy detail',

                        # BUG394499
                        'show system systemd sockets detail',
                        'show system systemd services detail',

                        #These counters are not present in all profiles.
                        'show platform barefoot counters table \
                              tcamRaclIdCount nz',
                        'show platform barefoot counters table \
                              tcamRaclBank1Count nz'
                        'show platform barefoot counters table \
                              egressTcamV6RaclBank1Count nz',
                        'show platform barefoot counters table \
                              egressTcamRaclBank1Count nz',

                        ]

# This is used by 'show agent logs crash' when scouring log files, and also used by
# ConfigAgent to mangle log outputs that could look like a crash but are not.
crashIndications = (
      'Exception',
      'died with signal',
      'Backtrace',
      'Assertion',   # covers C++ "Assertion `foo' failed" and python AssertionError
      'panic: ',     # covers panics from any Go agent
      )

LOAD_CONFIG_ERR_MSG = "Aborted due to command error"

class LoadConfigError( Exception ):
   """Errors generated by loading config from a file or URL."""
   def __init__( self, msg=LOAD_CONFIG_ERR_MSG ):
      self.msg = msg
      Exception.__init__( self, msg )
