# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliPlugin import AleCliLib
from CliDynamicSymbol import CliDynamicPlugin

from IpLibConsts import DEFAULT_VRF
import Tac

AleL3CliModel = CliDynamicPlugin( "AleL3CliModel" )

def doShowL3AleAcknowledgementRoutes( mode, args ):
   fibRoutes = {}
   af = 'ipv4' if 'ip' in args else 'ipv6'
   argsVrfName = args.get( 'VRF' )
   allRoutes = args.get( 'all' )
   prefix = args.get( 'PREFIX' )

   if not argsVrfName:
      argsVrfName = DEFAULT_VRF

   routeStatus = AleCliLib.getRouteStatus( af, argsVrfName )
   programmedRouteStatus = AleCliLib.getProgrammedRouteStatus( af, argsVrfName )

   # Collect the prefixes to iterate over
   prefixes = set( routeStatus.route )

   if programmedRouteStatus:
      prefixes = prefixes.union( programmedRouteStatus.routeVersion )

   for routePrefix in prefixes:
      requestedVersion = None
      requestedDelete = None
      requestFailed = None
      programmedVersion = None
      programmedDeleted = None
      programmingFailed = None

      route = routeStatus.route.get( routePrefix )

      if route:
         requestedVersion = route.routeVersionId
         requestedDelete = route.getRawAttribute( 'routeVersionId' ).tombstone
         requestFailed = route.getRawAttribute( 'routeVersionId' ).nack

      if programmedRouteStatus:
         programmedStatus = programmedRouteStatus.routeVersion.get( routePrefix )

         if programmedStatus:
            programmedVersion = programmedStatus.versionId
            rawVersionId = programmedStatus.getRawAttribute( 'versionId' )
            programmedDeleted = rawVersionId.tombstone
            programmingFailed = rawVersionId.nack

      # There are mismatched versions if:
      # 1. We don't have an ack published, and we requested any versionID.
      # 2. The published ack doesn't match the requested ack.
      # 3. Requested ack not found but stale entry found in ack status.
      #    This happens when fec/route entry is deleted from the fib but stale
      #    entry found in Programmed fec/route status.
      mismatchedVersions = False
      if programmedVersion or requestedVersion:
         mismatchedVersions = requestedVersion != programmedVersion

      # Either we are showing everything, or only showing items where the
      # routeVersionId has not been acknowledged
      if allRoutes or prefix or mismatchedVersions:
         ipRoutePrefix = Arnet.IpGenPrefix( str( routePrefix ) )

         if not prefix or str( prefix ) == str( ipRoutePrefix ):
            requestedVersionId = None

            if requestedVersion is not None:
               requestedVersionId = AleL3CliModel.AcknowledgementRouteVersionId()
               requestedVersionId.versionId = requestedVersion
               requestedVersionId.requestedDelete = requestedDelete
               requestedVersionId.failed = requestFailed

            acknowledgedVersionId = None

            if programmedVersion is not None:
               acknowledgedVersionId = AleL3CliModel.AcknowledgementRouteVersionId()
               acknowledgedVersionId.versionId = programmedVersion
               acknowledgedVersionId.requestedDelete = programmedDeleted
               acknowledgedVersionId.failed = programmingFailed

            acknowledgementRouteEntry = AleL3CliModel.AcknowledgementRouteEntry()
            acknowledgementRouteEntry.requested = requestedVersionId
            acknowledgementRouteEntry.acknowledged = acknowledgedVersionId

            fibRoutes[ ipRoutePrefix ] = acknowledgementRouteEntry

   return AleL3CliModel.AcknowledgementRouteStatus( vrfName=argsVrfName,
                                                    fibRoutes=fibRoutes )

def doShowL3AleAdjAcknowledgement( mode, args ):
   fecEntries = {}
   argsVrfName = args.get( 'VRF', DEFAULT_VRF )
   allFecs = 'all' in args
   argsFecId = args.get( 'FECID' )

   def addFecAckEntry( programmedVersion, programmedDeleted, programmingFailed,
                       requestedVersion, requestedDelete, requestFailed, allFecs,
                       argsFecId, fecId, fecEntries ):
      # There are mismatched versions if:
      # 1. We don't have an ack published, and we requested any versionID
      # 2. The published ack doesn't match the requested ack.
      # 3. Requested ack not found but stale entry found in ack status.
      #    This happens when fec/route entry is deleted from the fib but stale
      #    entry found in Programmed fec/route status.
      mismatchedVersions = False
      if programmedVersion or requestedVersion:
         mismatchedVersions = requestedVersion != programmedVersion

      # Either we are showing everything, or only showing items where the
      # routeVersionId has not been acknowledged.
      # Also show routes which are failed
      if allFecs or argsFecId or mismatchedVersions:
         if not argsFecId or argsFecId == fecId:
            requestedVersionId = None

            if requestedVersion is not None:
               requestedVersionId = AleL3CliModel.AcknowledgementFecVersionId()
               requestedVersionId.versionId = requestedVersion
               requestedVersionId.requestedDelete = requestedDelete
               requestedVersionId.failed = requestFailed

            acknowledgedVersionId = None

            if programmedVersion is not None:
               acknowledgedVersionId = AleL3CliModel.AcknowledgementFecVersionId()
               acknowledgedVersionId.versionId = programmedVersion
               acknowledgedVersionId.requestedDelete = programmedDeleted
               acknowledgedVersionId.failed = programmingFailed

            acknowledgementFecEntry = AleL3CliModel.AcknowledgementFecEntry()
            acknowledgementFecEntry.requested = requestedVersionId
            acknowledgementFecEntry.acknowledged = acknowledgedVersionId

            fecEntries[ fecId ] = acknowledgementFecEntry

   # Forwarding status is stored separately for v4 and v6, but the command
   # only has an "ip" form.  Thus we need to retrieve both forwardingStatus
   # collections separately
   forwarding4Status = AleCliLib.getForwardingStatus( 'ipv4' )
   forwarding6Status = AleCliLib.getForwardingStatus( 'ipv6' )
   fecIds = set( forwarding4Status.fec )
   fecIds = fecIds.union( forwarding6Status.fec )
   programmedFecStatus = AleCliLib.programmedFecStatus
   fecIds = fecIds.union( programmedFecStatus.fecVersion )

   for fecId in fecIds:
      requestedVersion = None
      requestedDelete = None
      requestFailed = None
      programmedVersion = None
      programmedDeleted = None
      programmingFailed = None

      fecEntry = forwarding4Status.fec.get( fecId )

      if not fecEntry:
         fecEntry = forwarding6Status.fec.get( fecId )

      if fecEntry:
         requestedVersion = fecEntry.fecVersionId
         requestedDelete = fecEntry.getRawAttribute( 'fecVersionId' ).tombstone
         requestFailed = fecEntry.getRawAttribute( 'fecVersionId' ).nack

      fecVersion = programmedFecStatus.fecVersion.get( fecId )

      if fecVersion:
         programmedVersion = fecVersion.versionId
         programmedDeleted = fecVersion.getRawAttribute( 'versionId' ).tombstone
         programmingFailed = fecVersion.getRawAttribute( 'versionId' ).nack

      addFecAckEntry( programmedVersion, programmedDeleted, programmingFailed,
                      requestedVersion, requestedDelete, requestFailed, allFecs,
                      argsFecId, fecId, fecEntries )

   # read the tunnelFibEntry and tunnelStatus in order to dump the tunnel fec ack
   # information
   tunnelProgrammingStatus = AleCliLib.tunnelProgrammingStatus
   tunnelFib = AleCliLib.tunnelFib
   # iterate through the fib entries
   tunnelIds = set( tunnelFib.entry )
   for tunId in tunnelIds:
      requestedVersion = None
      requestedDelete = None
      requestFailed = None
      programmedVersion = None
      programmedDeleted = None
      programmingFailed = None

      tunEntry = tunnelFib.entry[ tunId ]

      if tunEntry:
         requestedVersion = tunEntry.seqNo
         requestedDelete = False
         requestFailed = False

      tunStatus = tunnelProgrammingStatus.tunnelStatus.get( tunId )

      if tunStatus:
         programmedVersion = tunStatus.seqNo
         programmingFailed = tunStatus.nack
         programmedDeleted = False

      # Convert tunnelId to fecId
      fecId = Tac.Type( "Smash::Fib::FecId" ).tunnelIdToFecId( tunId )

      addFecAckEntry( programmedVersion, programmedDeleted, programmingFailed,
                      requestedVersion, requestedDelete, requestFailed, allFecs,
                      argsFecId, fecId, fecEntries )

   # nhg Fec Ack is notified by nhgProgrammedFecStream.event and will be reset
   # imediately after notifying. To get the latest Acked nhg Fec we need to look at
   # NHG hardware status.
   nhgSmashStatus = AleCliLib.nhgSmashStatus
   nhgVrfHwStatus = AleCliLib.nhgHwStatus.vrfStatus.get( DEFAULT_VRF )
   for entry in nhgSmashStatus.nexthopGroupEntry.values():
      requestedVersion = entry.configuredVersion
      requestedDelete = False
      requestFailed = False
      programmedVersion = None
      programmedDeleted = None
      programmingFailed = None

      nhgHwEntry = nhgVrfHwStatus.nexthopGroupAdjacency.get( entry.nhgId )
      # We can't support showing deleted/failed state currently because nhg hw entry
      # will be removed in such cases and we don't keep nhg fec versionId in anywhere
      # inside our system.
      if nhgHwEntry and nhgHwEntry.configuredVersion:
         programmedVersion = nhgHwEntry.configuredVersion
         programmedDeleted = False
         programmingFailed = False

      # convert NhgId to FecId
      fecId = Tac.Type( "Smash::Fib::FecId" ).nexthopGroupIdToFecId( entry.nhgId )
      addFecAckEntry( programmedVersion, programmedDeleted, programmingFailed,
                      requestedVersion, requestedDelete, requestFailed, allFecs,
                      argsFecId, fecId, fecEntries )

   return AleL3CliModel.AcknowledgementFecStatus( vrfName=argsVrfName,
                                                  fecEntries=fecEntries )
