# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliUtil
from Intf import IntfRange
import Arnet
from CliPlugin.AlePortGroupCli import (
   isStartupConfig,
   portGroupDir,
   portGroupCliConfig,
   t0,
)

def getPortList( groupName, groupMode ):
   group = portGroupDir.portGroup.get( groupName )
   if group:
      portList = group.mode.get( groupMode )
      return portList
   return None

def getDefaultGroupMode( groupName ):
   groupPort = portGroupDir.portGroup.get( groupName )
   if groupPort:
      return groupPort.defaultMode
   return None

def promptConfirmed( mode, portGroup, portMode ):
   warning = '''
                  WARNING!
   Changing the port-group mode will cause all
   interfaces on the switch to flap.
   '''
   prompt = "Do you wish to proceed with this command? [y/N]"

   portGroup = portGroupDir.portGroup.get( portGroup )
   if portMode in portGroup.warnings:
      if portGroup.warnings[ portMode ]:
         warning = warning.replace( "the switch", portGroup.warnings[ portMode ] )

   mode.addWarning( warning )
   return BasicCliUtil.confirm( mode, prompt, answerForReturn=False )

def isInDefaultMode( portGroup ):
   ''' is the portGroup currently in it's default mode?  '''
   currentPortMode = portGroupCliConfig.config.get( portGroup )
   currentPortList = getPortList( portGroup, currentPortMode ) if currentPortMode \
                     else None
   return not currentPortList or currentPortMode == getDefaultGroupMode( portGroup )

def handleConfigPortGroup( mode, args ):
   portGroup = args[ 'PORT_GROUP' ]
   portMode = args[ 'PORT_MODE' ]

   # at startup, any portGroup or portMode strings are allowed:
   if isStartupConfig( mode ):
      portGroupCliConfig.config[ portGroup ] = portMode
      return

   # if the portMode stays the same:
   currentPortMode = portGroupCliConfig.config.get( portGroup )
   if currentPortMode == portMode:
      return

   # the portGroup may be in default because of a bad setting string.
   # do not prompt in that case
   skipPrompt = ( portMode == getDefaultGroupMode( portGroup ) and
                  isInDefaultMode( portGroup ) )
   if skipPrompt or promptConfirmed( mode, portGroup, portMode ):
      t0( f"Setting cli port group config for group { portGroup } to { portMode }" )
      portGroupCliConfig.config[ portGroup ] = portMode

def handleNoConfigPortGroup( mode, args ):
   portGroup = args[ 'PORT_GROUP' ]
   portMode = getDefaultGroupMode( portGroup )

   # if we are already in default mode, do not prompt
   if ( isInDefaultMode( portGroup ) or
         promptConfirmed( mode, portGroup, portMode ) ):
      del portGroupCliConfig.config[ portGroup ]

def handleHardwarePortGroup( mode, args ):
   pgHeader = "\n%-14s\tActive Ports: %4s"
   pgLimitationHeader = "%-14s"
   port = "Port"
   state = "State"
   pgColumn = f"{port:<14}\t{state:<12}"
   pgFooter = "------------------------------------------"
   portHeader = "%-14s\t%-12s"

   for groupName, portGroup in portGroupDir.portGroup.items():
      portGroupModeName = portGroupCliConfig.config.get( groupName )
      portGroupList = portGroup.mode.get( portGroupModeName ) if \
                        portGroupModeName else None
      if not portGroupModeName or not portGroupList:
         portGroupModeName = getDefaultGroupMode( groupName )
         portGroupList = portGroup.mode[ portGroupModeName ]

      portStates = {}
      activePorts = []
      for modeName, _mode in portGroup.mode.items():
         if modeName == portGroupModeName:
            continue
         for port in _mode.ports:
            portStates[ port ] = "Inactive"
      for port in portGroup.mode[ portGroupModeName ].ports:
         portStates[ port ] = "Active"
         activePorts.append( port )

      portGroupName = f"Portgroup: { groupName }"
      if portGroupDir.generateActivePortsList:
         print( pgHeader % ( portGroupName, IntfRange.intfListToCanonical(
                                 Arnet.sortIntf( activePorts ) )[ 0 ] ) )
      else:
         print( pgHeader % ( portGroupName, portGroupModeName ) )

      if portGroupModeName in portGroup.limitations:
         lim = f'Limitations: { portGroup.limitations[ portGroupModeName ] }'
         print( pgLimitationHeader % lim )
      print( pgColumn )
      print( pgFooter )
      if portGroupDir.noActiveInactiveDuplicates:
         for port in Arnet.sortIntf( portStates ):
            print( portHeader % ( port, portStates[ port ] ) )
      else:
         for modeName, _mode in portGroup.mode.items():
            if modeName == portGroupModeName:
               state = "Active"
            else:
               state = "Inactive"
            for port in Arnet.sortIntf( _mode.ports ):
               print( portHeader % ( port, state ) )
