# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import io
import json

import AgentCommandRequest
from CliPlugin import AleCliLib
from CliPlugin.AleVerifyCliLib import VerifyPlatformStateHelper
from CliPlugin.AleVerifyCli import (
   agentConfigDir,
)
from CliDynamicSymbol import CliDynamicPlugin

AleVerifyCliModel = CliDynamicPlugin( "AleVerifyCliModel" )

def allAgentNames( mode ):
   agents = [ agentName[ 0 ] for agentName in agentConfigDir.items() ]
   return agents

def _getPendingStateCommand( quiet='quiet' ):
   cmd = 'AATS#' + quiet
   return cmd

def handlePlatformPendingState( mode, args ):
   quiet = args.get( 'quiet', '' )
   agent = args.get( 'AGENT' )
   agents = [ agent ] if agent else allAgentNames( mode )
   if not agents:
      print( 'No agents support this command' )
      return
   cmd = _getPendingStateCommand( quiet )
   for agent in agents:
      AgentCommandRequest.runSocketCommand( AleCliLib.entityManager, 'ale',
                                             ( f'aleVerify::{ agent }' ), cmd )

def _runVerifyCliSocketCommand( agent, checkPending=True ):
   if checkPending:
      cmd = _getPendingStateCommand()
   else:
      # need a ending '#' to prevent agent name being
      # substr of another agent name. we skipped agents that have '#'
      cmd = 'VTS#' + agent + '#'
   cmdBuf = io.StringIO()
   AgentCommandRequest.runSocketCommand( AleCliLib.entityManager, 'ale',
                                         ( f'aleVerify::{ agent }' ), cmd,
                                         stringBuff=cmdBuf )
   return cmdBuf.getvalue()

def handlePlatformAgentStateVerifyAll( mode, args ):
   agent = args.get( 'AGENT' )
   agents = []
   results = {}
   if agent:
      agents.append( agent )
   else:
      agents.extend( allAgentNames( mode ) )
   if not agents:
      mode.addError( 'No agents support this command' )
   for agent in agents:
      if '#' in agent:
         mode.addError( f'Skipping agent: { agent }' )
         continue
      pendingStateOutput = _runVerifyCliSocketCommand( agent )
      if pendingStateOutput.strip():
         output = { agent: [ { '_': VerifyPlatformStateHelper.retry() } ] }
      else:
         output = _runVerifyCliSocketCommand( agent, checkPending=False )
         try:
            output = json.loads( output )
         except ValueError:
            mode.addError( VerifyPlatformStateHelper.garbled( agent ) )
            output = { agent:
                  [ { '_': VerifyPlatformStateHelper.crash( output ) } ] }
      results.update( output )
   capiResults = AleVerifyCliModel.VerifyPlatformStateCapiModel()
   capiResults.setDict( results )
   return capiResults
