# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ConfigMount
import Tac
from TypeFuture import TacLazyType

from ArfaAegisLib import (
   HardwareDefinitionJsonValidator,
   TcamKeySizeError,
   UnsupportedHardwareDefinitionError,
   readJson,
)
hardwareDefinitionConfig = None

UniqueId = TacLazyType( 'Ark::UniqueId' )

def urlChanged( newUrl ):
   if newUrl:
      if not hardwareDefinitionConfig.currCfg:
         return True
      return hardwareDefinitionConfig.currCfg.url != newUrl
   return hardwareDefinitionConfig.currCfg is not None

# --------------------------------------------------
# Handler for
# 'traffic-policy interface hardware definition URL'
# in `platform tfa` configuration mode.
# --------------------------------------------------
def handleHardwareDefinition( mode, args ):
   url = args[ 'URL' ]
   if not urlChanged( url.url ):
      # Avoid changing the currCfg if the URL hasn't changed.
      # If the URL contents need to be refreshed, use `no` command
      # and reconfigure.
      return

   errMessage = ''
   validator = HardwareDefinitionJsonValidator()

   # XXX BUG764647 tracks task to only allow a single config session to
   # update the hardware definition at the same time.
   # For now, let's use the actual entity as a disposable scratchpad.
   # BUG765134 tracks adding copy and isEqual to the TAC type so that
   # we can parse the configuration into a scratchpad first.
   version = UniqueId.generateUniqueId()
   subCfg = hardwareDefinitionConfig.subconfig.newMember(
      version, url.url )

   try:
      hardwareDefinitionJson = readJson( url.localFilename() )
      validator.validate( hardwareDefinitionJson, subCfg )
   except OSError as err: # File not found
      errMessage = "Error in opening file: %s." % str( err )
   except ValueError as err: # JSON decode errors
      errMessage = "Error in parsing file: %s." % str( err )
   except KeyError as err: # JSON missing required keys
      errMessage = "Expected key %s not found." % err.args[ 0 ]
   except ( UnsupportedHardwareDefinitionError,
            TcamKeySizeError, ) as err:
      # JSON values or groups of values are not supported
      # by the platform.
      errMessage = err.message
   except Exception as err:
      # Precaution for now as we may encounter other errors we don't know about.
      del hardwareDefinitionConfig.subconfig[ version ]
      raise err

   # BUG765135 tracks the task of handling config-replace rollback.
   # For now, only handle CLI and config session rollback.
   if errMessage:
      del hardwareDefinitionConfig.subconfig[ version ]
      mode.addError( errMessage )
   else:
      hardwareDefinitionConfig.currCfg = subCfg
      mode.addWarning( 'ArfaAegis will restart' )

# --------------------------------------------------------------
# Handler for
# '(no|default) traffic-policy interface hardware definition URL'
# in `platform tfa` configuration mode.
# --------------------------------------------------------------
def handleNoOrDefaultHardwareDefinition( mode, args ):
   if not urlChanged( newUrl=None ):
      return
   hardwareDefinitionConfig.subconfig.clear()
   hardwareDefinitionConfig.currCfg = None
   mode.addWarning( 'ArfaAegis will restart' )

def Plugin( em ):
   global hardwareDefinitionConfig
   hardwareDefinitionConfig = ConfigMount.mount(
      em,
      'trafficPolicies/hardware/intf/definition',
      'Aegis::HardwareDefinitionConfig', 'w' )
