# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from ArnetModel import IpGenericAddress
from CliModel import GeneratorList
from CliModel import Bool
from CliModel import Enum
from CliModel import Int
from CliModel import Model
from CliModel import Str

# Make sure the following definitions are in sync with the definitions in
# /src/gated/gated-ctk/src/ami_common/rib_common.h
# rib_common.h holds the values as Enums (ints)
REASON_NOT_AGGREGATED = [ 'noComplement' ]

REASON_NOT_AGGREGATED_MAP = dict( enumerate( REASON_NOT_AGGREGATED, start=0 ) )

REASON_NOT_AGGREGATED_OUTPUT_MAP = {
   'noComplement' : 'Complement prefix is not available'
}

class Contributor( Model ):
   address = IpGenericAddress( help='Contributor address' )
   maskLength = Int( help='Contributor mask length' )

   def processData( self, data ):
      self.address = data.pop( 'addr' )
      self.maskLength = data.pop( 'mask' )

   def render( self ):
      # pylint: disable-next=consider-using-f-string
      print( '      %s/%d' % ( self.address, self.maskLength ) )

class Prefix( Model ):
   __streamable__ = True

   address = IpGenericAddress( help='Prefix address' )
   maskLength = Int( help='Prefix mask length' )
   autoAggregate = Bool( help='Whether this prefix is an auto aggregate' )
   notAggregatedReason = Enum( values=REASON_NOT_AGGREGATED, optional=True,
                               help='Reason this prefix is not auto aggregated' )
   contributors = GeneratorList( valueType=Contributor,
                                 help='Auto aggregate contributors' )

   def processData( self, data ):
      self.address = data.pop( 'addr' )
      self.maskLength = data.pop( 'mask' )
      self.autoAggregate = bool( ord( data.pop( 'is_auto_aggr' ) ) )
      reasonVal = data.pop( 'non_aggr_reason', None )
      if reasonVal:
         self.notAggregatedReason = REASON_NOT_AGGREGATED_MAP[ ord( reasonVal ) ]

   def render( self ):
      reason = ''
      if self.notAggregatedReason:
         reason = ': ' + REASON_NOT_AGGREGATED_OUTPUT_MAP[ self.notAggregatedReason ]

      # pylint: disable-next=consider-using-f-string
      print( '  %s/%d, %s-aggregate%s' % ( self.address, self.maskLength,
                                          'auto' if self.autoAggregate else 'non',
                                          reason ) )
      first = True
      for model in self.contributors:
         if first:
            print( '    Contributing routes:' )
            first = False
         model.render()

class PrefixBlock( Model ):
   __streamable__ = True

   firstAddress = IpGenericAddress( help='Block first address' )
   lastAddress = IpGenericAddress( help='Block last address' )
   prefixes = GeneratorList( valueType=Prefix,
                             help='Prefixes that are active in the RIB and eligible '
                                  'for auto aggregation' )

   def processData( self, data ):
      self.firstAddress = data.pop( 'start' )
      self.lastAddress = data.pop( 'end' )

   def render( self ):
      legend = 'BGP aggregation contiguous prefix block entry for %s to %s'
      print( legend % ( self.firstAddress, self.lastAddress ) )

      for model in self.prefixes:
         model.render()

      print()

class Domain( Model ):
   __streamable__ = True

   address = IpGenericAddress( help='Domain address' )
   maskLength = Int( help='Domain mask length' )
   contiguousPrefixBlocks = GeneratorList( valueType=PrefixBlock,
                                           help='List of contiguous prefix blocks' )

   def processData( self, data ):
      self.address = data.pop( 'addr' )
      self.maskLength = data.pop( 'mask' )

   def render( self ):
      # pylint: disable-next=consider-using-f-string
      print( 'BGP aggregation domain is %s/%d' % ( self.address, self.maskLength ) )

      model = None
      for model in self.contiguousPrefixBlocks:
         model.render()

      if model is None:
         print()

class AutoAggregation( Model ):
   __streamable__ = True

   domains = GeneratorList( valueType=Domain, help='Auto aggregation domains' )
   _vrf = Str( help='VRF name' )

   def render( self ):
      print( 'BGP automatic aggregation information for VRF', self._vrf )

      model = None
      for model in self.domains:
         model.render()

      if model is None:
         print()
