# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliGlobal
import LazyMount
from CliMode.Avt import ( RouterAdaptiveVirtualTopologyConfigMode,
                          RouterAvtPolicyAppProfileConfigMode,
                          RouterAvtPolicyConfigMode,
                          RouterAvtProfileConfigMode,
                          RouterAvtVrfConfigMode,
                          DEFAULT_OUTLIER_THRESH_JITTER,
                          DEFAULT_OUTLIER_THRESH_LATENCY,
                          DEFAULT_OUTLIER_THRESH_LOSSRATE,
                          DEFAULT_OUTLIER_THRESH_LOAD,
                        )
from CliMode.Dps import ( JITTER_SCALE,
                          LATENCY_SCALE,
                          LOSS_RATE_SCALE,
                          LOSS_RATE_ADJUSTMENT,
                          LOAD_SCALE,
                        )
from CliDynamicPlugin.AvtCliLib import ( AvtProfileConfigContext,
                                  AvtPolicyConfigContext,
                                  AvtVrfConfigContext,
                                  AvtPolicyAppProfileConfigContext,
                        )
from TypeFuture import TacLazyType
import Tac

gv = CliGlobal.CliGlobal( dict( avtCliConfig=None ) )
AvtRole = TacLazyType( 'Avt::AvtRole' )
TrafficClass = TacLazyType( "WanTE::TrafficClass" )
Dscp = TacLazyType( "WanTE::Dscp" )
MetricOrder = TacLazyType( 'Avt::MetricOrder' )
# pkgdeps: library SfeInternetExitTypes
PolicyConfig = TacLazyType( 'SfeInternetExit::PolicyConfig' )

def RouterAdaptiveVirtualTopologyConfigCmd( mode, args ):
   gv.avtCliConfig.avtConfigured = True
   childMode = mode.childMode( RouterAdaptiveVirtualTopologyConfigMode )
   mode.session_.gotoChildMode( childMode )

def RouterAdaptiveVirtualTopologyConfigDefaultCmd( mode, args ):
   gv.avtCliConfig.role = AvtRole.unknown
   gv.avtCliConfig.region = ""
   gv.avtCliConfig.regionId = 0
   gv.avtCliConfig.zone = ""
   gv.avtCliConfig.zoneId = 0
   gv.avtCliConfig.site = ""
   gv.avtCliConfig.siteId = 0
   gv.avtCliConfig.policyConfig.clear()
   gv.avtCliConfig.wanTEPolicyConfig.clear()
   gv.avtCliConfig.wanTEVrfConfig.clear()
   gv.avtCliConfig.avtProfile.clear()
   gv.avtCliConfig.vrfConfig.clear()
   gv.avtCliConfig.avtConfigured = False

def AvtRoleConfigCmdHandler( mode, args ):
   gv.avtCliConfig.role = AvtRole.unknown
   gv.avtCliConfig.vxlanGateway = ( 'gateway' in args and 'vxlan' in args )
   if 'edge' in args:
      gv.avtCliConfig.role = AvtRole.edge
   elif 'region' in args:
      gv.avtCliConfig.role = AvtRole.regionBR
   elif 'zone' in args:
      gv.avtCliConfig.role = AvtRole.transit
   elif 'pathfinder' in args:
      gv.avtCliConfig.role = AvtRole.pathfinder

def AvtRoleConfigCmdDefaultHandler( mode, args ):
   gv.avtCliConfig.role = AvtRole.unknown
   gv.avtCliConfig.vxlanGateway = False

def AvtRegionConfigCmdHandler( mode, args ):
   name = args.get( 'REGION_NAME' )
   idd = int( args.get( 'REGION_ID' ) )
   gv.avtCliConfig.region = name
   gv.avtCliConfig.regionId = idd

def AvtRegionConfigCmdDefaultHandler( mode, args ):
   gv.avtCliConfig.region = ""
   gv.avtCliConfig.regionId = 0

def AvtZoneConfigCmdHandler( mode, args ):
   name = args.get( 'ZONE_NAME' )
   idd = int( args.get( 'ZONE_ID' ) )
   gv.avtCliConfig.zone = name
   gv.avtCliConfig.zoneId = idd

def AvtZoneConfigCmdDefaultHandler( mode, args ):
   gv.avtCliConfig.zone = ""
   gv.avtCliConfig.zoneId = 0

def AvtSiteConfigCmdHandler( mode, args ):
   name = args.get( 'SITE_NAME' )
   idd = int( args.get( 'SITE_ID' ) )
   gv.avtCliConfig.site = name
   gv.avtCliConfig.siteId = idd

def AvtSiteConfigCmdDefaultHandler( mode, args ):
   gv.avtCliConfig.site = ""
   gv.avtCliConfig.siteId = 0

def AvtProfileConfigCmdHandler( mode, args ):
   avtName = args.get( 'AVT_NAME' )
   context = AvtProfileConfigContext( gv.avtCliConfig, avtName )
   context.addAvtProfile( avtName )
   if context.isIePolicyEmpty():
      context.setIePolicy( PolicyConfig.defaultName )
   childMode = mode.childMode( RouterAvtProfileConfigMode, context=context )
   mode.session_.gotoChildMode( childMode )

def AvtProfileConfigCmdDefaultHandler( mode, args ):
   avtName = args.get( 'AVT_NAME' )
   context = AvtProfileConfigContext( gv.avtCliConfig, avtName )
   context.delAvtProfile( avtName )

def AvtProfileLoadBalanceConfigCmdHandler( mode, args ):
   name = args.get( 'LB_POLICY' )
   mode.context.setLBPolicy( name )

def AvtProfileLoadBalanceConfigCmdDefaultHandler( mode, args ):
   mode.context.setLBPolicy( "" )

def AvtProfileInternetExitConfigCmdHandler( mode, args ):
   name = args.get( 'IE_POLICY', '' )
   mode.context.setIePolicy( name )

def AvtProfileInternetExitConfigCmdDefaultHandler( mode, args ):
   # When deleting existing IE policy then create default IE policy
   mode.context.setIePolicy( PolicyConfig.defaultName )

def AvtPolicyConfigCmdHandler( mode, args ):
   policyName = args.get( 'POLICY_NAME' )
   context = AvtPolicyConfigContext( gv.avtCliConfig, policyName )
   context.addPolicyConfig( policyName )
   childMode = mode.childMode( RouterAvtPolicyConfigMode, context=context )
   mode.session_.gotoChildMode( childMode )

def AvtPolicyConfigCmdDefaultHandler( mode, args ):
   policyName = args.get( 'POLICY_NAME' )
   context = AvtPolicyConfigContext( gv.avtCliConfig, policyName )
   context.delPolicyConfig( policyName )

def AvtPolicyAppProfileConfigCmdHandler( mode, args ):
   mode.ruleChanged = True
   profileName = args.get( 'PROFILE_NAME' )
   order = args.get( 'ORDER' )
   profileName2 = args.get( 'PROFILE_NAME_2' )
   context = AvtPolicyAppProfileConfigContext( gv.avtCliConfig,
         mode.context.policyName(),
         profileName,
         mode )
   context.addProfileConfig( profileName, order, profileName2 )
   childMode = mode.childMode( RouterAvtPolicyAppProfileConfigMode,
         context=context )
   mode.session_.gotoChildMode( childMode )

def AvtPolicyAppProfileConfigCmdDefaultHandler( mode, args ):
   mode.ruleChanged = True
   profileName = args.get( 'PROFILE_NAME' )
   context = AvtPolicyAppProfileConfigContext( gv.avtCliConfig,
         mode.context.policyName(),
         profileName,
         mode )
   context.delProfileConfig( profileName )

def AvtPolicyAppProfileActionNameConfigCmdHandler( mode, args ):
   avtName = args.get( 'AVT_NAME' )
   mode.context.setActionName( avtName )

def AvtPolicyAppProfileActionNameConfigCmdDefaultHandler( mode, args ):
   mode.context.setActionName( '' )

def AvtPolicyAppProfileTrafficClassConfigCmdHandler( mode, args ):
   trafficClass = args.get( 'TRAFFIC_CLASS_VALUE', TrafficClass.null )
   mode.context.setTrafficClass( trafficClass )

def AvtPolicyAppProfileDscpConfigCmdHandler( mode, args ):
   dscp = args.get( 'DSCP_VALUE', Dscp.null )
   mode.context.setDscp( dscp )

def AvtVrfConfigCmdHandler( mode, args ):
   vrfName = args.get( 'VRF_NAME' )
   context = AvtVrfConfigContext( gv.avtCliConfig, vrfName )
   context.addVrfConfig( vrfName )
   childMode = mode.childMode( RouterAvtVrfConfigMode, context=context )
   mode.session_.gotoChildMode( childMode )

def AvtVrfConfigCmdDefaultHandler( mode, args ):
   vrfName = args.get( 'VRF_NAME' )
   context = AvtVrfConfigContext( gv.avtCliConfig, vrfName )
   context.delVrfConfig( vrfName )

def AvtVrfPolicyConfigCmdHandler( mode, args ):
   name = args.get( 'AVT_POLICY_NAME' )
   mode.context.setPolicy( name )

def AvtVrfPolicyConfigCmdDefaultHandler( mode, args ):
   mode.context.setPolicy( "" )

def AvtVrfProfileConfigCmdHandler( mode, args ):
   name = args.get( 'PROFILE_NAME' )
   idd = int( args.get( 'ID' ) )
   mode.context.setAvtNameToId( name, idd )

def AvtVrfProfileConfigCmdDefaultHandler( mode, args ):
   name = args.get( 'PROFILE_NAME' )
   mode.context.delAvtNameToId( name )

def AvtMetricOrderConfigCmdHandler( mode, args ):
   # lossrate is default metricorder
   metricOrder = MetricOrder.lossrate
   if 'latency' in args:
      metricOrder = MetricOrder.latency
   elif 'jitter' in args:
      metricOrder = MetricOrder.jitter
   elif 'load' in args:
      metricOrder = MetricOrder.load
   mode.context.setMetricOrder( metricOrder )

def AvtMetricOrderConfigCmdDefaultHandler( mode, args ):
   mode.context.setMetricOrder( MetricOrder.lossrate )

def AvtOutlierEnabledConfigCmdHandler( mode, args ):
   if 'disabled' in args:
      mode.context.setAvtOutlierDisabled( True )
   else:
      mode.context.setAvtOutlierDisabled( False )

def AvtOutlierEnabledConfigCmdDefaultHandler( mode, args ):
   mode.context.setAvtOutlierDisabled( False )

def AvtOutlierLatencyThresholdConfigCmdHandler( mode, args ):
   timeArg = args.get( '<latency>' )
   latency = int( timeArg ) * LATENCY_SCALE
   mode.context.setOutlierThreshold( latency=latency )

def AvtOutlierLatencyThresholdConfigCmdDefaultHandler( mode, args ):
   mode.context.setOutlierThreshold(
         latency=DEFAULT_OUTLIER_THRESH_LATENCY )

def AvtOutlierJitterThresholdConfigCmdHandler( mode, args ):
   timeArg = args.get( '<jitter>' )
   jitter = int( timeArg ) * JITTER_SCALE
   mode.context.setOutlierThreshold( jitter=jitter )

def AvtOutlierJitterThresholdConfigCmdDefaultHandler( mode, args ):
   mode.context.setOutlierThreshold(
         jitter=DEFAULT_OUTLIER_THRESH_JITTER )

def AvtOutlierLossrateThresholdConfigCmdHandler( mode, args ):
   percentArg = args.get( '<percentage>' )
   loss = int( float( percentArg ) * LOSS_RATE_SCALE )
   loss += LOSS_RATE_ADJUSTMENT
   mode.context.setOutlierThreshold( loss=loss )

def AvtOutlierLossrateThresholdConfigCmdDefaultHandler( mode, args ):
   mode.context.setOutlierThreshold( loss=DEFAULT_OUTLIER_THRESH_LOSSRATE )

def AvtOutlierLoadThresholdConfigCmdHandler( mode, args ):
   percentArg = args.get( '<percentage>' )
   load = int( float( percentArg ) * LOAD_SCALE )
   mode.context.setOutlierThreshold( load=load )

def AvtOutlierLoadThresholdConfigCmdDefaultHandler( mode, args ):
   mode.context.setOutlierThreshold( load=DEFAULT_OUTLIER_THRESH_LOAD )

def Plugin( entityManager ):
   gv.avtCliConfig = LazyMount.mount( entityManager, 'avt/input/cli',
                                      'Avt::AvtCliConfig', 'w' )
