# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
from TypeFuture import TacLazyType

AvtConstants = Tac.Value( "Dps::DpsConstants" )
maxRuleKey = AvtConstants.maxRuleKey
defaultRuleKey = AvtConstants.defaultRuleKey
AvtPathOutlierThreshold = TacLazyType( 'Avt::AvtPathOutlierThreshold' )

t2 = Tracing.trace2

class AvtProfileConfigContext:
   def __init__( self, config, avtName ):
      self.config = config
      self._avtName = avtName
      self._avtProfile = None

   def avtName( self ):
      return self._avtName

   def avtProfile( self ):
      return self._avtProfile

   def isIePolicyEmpty( self ):
      if self._avtProfile:
         return self._avtProfile.internetExitPolicyName == ''
      return False

   def addAvtProfile( self, avtName ):
      avtProfile = self.config.avtProfile.get( avtName )
      if not avtProfile:
         avtProfile = self.config.newAvtProfile( avtName )
      self._avtProfile = avtProfile

   def delAvtProfile( self, avtName ):
      avtProfile = self.config.avtProfile.get( avtName )
      if avtProfile:
         del self.config.avtProfile[ avtName ]
      self._avtProfile = None

   def setLBPolicy( self, name ):
      self._avtProfile.loadBalanceGroupName = name

   def setIePolicy( self, name ):
      self._avtProfile.internetExitPolicyName = name

   def setTopology( self, topology ):
      self._avtProfile.avtTopology = topology

   def setMetricOrder( self, metricOrder ):
      self._avtProfile.metricOrder = metricOrder

   def setAvtOutlierDisabled( self, disabled ):
      self._avtProfile.outlierDisabled = disabled

   def setOutlierThreshold( self, loss=None, latency=None,
                            jitter=None, load=None ):
      if loss is None:
         loss = self._avtProfile.outlierThreshold.loss
      if latency is None:
         latency = self._avtProfile.outlierThreshold.latency
      if jitter is None:
         jitter = self._avtProfile.outlierThreshold.jitter
      if load is None:
         load = self._avtProfile.outlierThreshold.load
      self._avtProfile.outlierThreshold = \
            AvtPathOutlierThreshold( loss, latency, jitter, load )

class AvtPolicyConfigContext:
   def __init__( self, config, name ):
      self.config = config
      self._policyName = name
      self._policy = None

   def policyName( self ):
      return self._policyName

   def addPolicyConfig( self, policyName ):
      policy = self.config.policyConfig.get( policyName )
      if not policy:
         policy = self.config.newPolicyConfig( policyName )
         self.config.wanTEPolicyConfig.addMember( policy )
      self._policy = policy

   def delPolicyConfig( self, policyName ):
      policy = self.config.policyConfig.get( policyName )
      if policy:
         del self.config.policyConfig[ policyName ]
         del self.config.wanTEPolicyConfig[ policyName ]
      self._policy = None

   def policyConfig( self ):
      return self._policy

class AvtPolicyAppProfileConfigContext:
   def __init__( self, config, policyName, profileName, mode ):
      self.config = config
      self._policyName = policyName
      self._profileName = profileName
      self._profile = None
      self.mode = mode
      policy = self.config.policyConfig.get( policyName )
      if policy is None:
         t2( 'Policy name %s is not found' % policyName )
      self.policyConfig = policy

      self.appProfilePolicyRuleList = policy.appProfilePolicyRuleList
      self.profileNameToRuleKey = None
      self.constructNameToIdMap()
      self.appProfileRuleConfig = None
      self._ruleKey = None

   def constructNameToIdMap( self ):
      self.profileNameToRuleKey = {}
      self.profileNameToRuleKey[ 'default' ] = defaultRuleKey
      for ruleKey, ruleConfig in self.appProfilePolicyRuleList.items():
         self.profileNameToRuleKey[ ruleConfig.appProfileName ] = ruleKey

   def policyName( self ):
      return self._policyName

   def ruleKey( self ):
      return self._ruleKey

   def profileName( self ):
      return self._profileName

   def addProfileConfigNaive( self, profileName ):
      ruleKey = self.profileNameToRuleKey.get( profileName )
      if not ruleKey:
         curKeys = list( self.appProfilePolicyRuleList )
         if curKeys and curKeys[ -1 ] == defaultRuleKey:
            curKeys.pop( -1 )
         ruleKey = 1000 if not curKeys else curKeys[ -1 ] + 1000
         # restrict the rule key to max of 0x7FFF-FFFF
         # we will be duplicating the rule keys in default vrf
         # as follows:
         # ( vrf == default; vni == 0 ) = ruleKey + maxRuleKey
         # ( vrf == default; vni > 0 ) = ruleKey
         if ruleKey >= ( maxRuleKey - 1 ):
            self.scatterAppProfile()
            ruleKey = 1000 if not curKeys else curKeys[ -1 ] + 1000
         self.profileNameToRuleKey[ profileName ] = ruleKey

      appProfileRuleConfig = \
               self.policyConfig.appProfilePolicyRuleList.get( ruleKey )
      if not appProfileRuleConfig:
         appProfileRuleConfig = \
                  self.policyConfig.newAppProfilePolicyRuleList( ruleKey )
      appProfileRuleConfig.appProfileName = profileName
      self.appProfileRuleConfig = appProfileRuleConfig

   def scatterAppProfile( self ):
      # break them to n+1 chunks, leaving space both in the front and end
      ruleMax = maxRuleKey - 1
      chunkLength = ruleMax // ( len( self.appProfilePolicyRuleList ) + 1 )
      scattered = {}
      cur = chunkLength
      for ruleKey, ruleConfig in self.appProfilePolicyRuleList.items():
         if ruleConfig.appProfileName == 'default':
            continue
         # assuming it's ordered since it's an ordered tac collection
         scattered[ cur ] = ruleConfig
         cur += chunkLength

      self.appProfilePolicyRuleList.clear()
      self.profileNameToRuleKey.clear()
      self.profileNameToRuleKey[ 'default' ] = defaultRuleKey
      for ruleKey, ruleConfig in scattered.items():
         cur = self.policyConfig.newAppProfilePolicyRuleList( ruleKey )
         cur.appProfileName = ruleConfig.appProfileName
         cur.actionName = ruleConfig.actionName
         self.profileNameToRuleKey[ ruleConfig.appProfileName ] = ruleKey

   def findSpot( self, anchorRuleKey, order ):
      newRuleKey = -1
      if order == 'before':
         pre = 0
         for key in self.policyConfig.appProfilePolicyRuleList:
            if key == anchorRuleKey:
               break
            pre = key
         newRuleKey = ( pre + anchorRuleKey ) // 2
      elif order == 'after':
         pre = -1
         for key in self.policyConfig.appProfilePolicyRuleList:
            if pre == anchorRuleKey:
               nex = key
               break
            pre = key
         if list( self.policyConfig.appProfilePolicyRuleList )[ -1 ] == \
               anchorRuleKey:
            nex = anchorRuleKey + 2000

         newRuleKey = ( nex + anchorRuleKey ) // 2
      return newRuleKey

   def addProfileConfigComplete( self, profileName, order, profileName2 ):
      anchorRuleKey = self.profileNameToRuleKey.get( profileName2 )
      if not anchorRuleKey:
         self.mode.addError( '%s does not exist.' % profileName2 )
         return

      oldRuleKey = self.profileNameToRuleKey.get( profileName )
      oldActionName = ''
      if oldRuleKey:
         oldActionName = \
               self.policyConfig.appProfilePolicyRuleList[ oldRuleKey ].actionName
         del self.policyConfig.appProfilePolicyRuleList[ oldRuleKey ]
         del self.profileNameToRuleKey[ profileName ]

      newRuleKey = self.findSpot( anchorRuleKey, order )
      if newRuleKey in self.policyConfig.appProfilePolicyRuleList:
         self.scatterAppProfile()
         newRuleKey = self.findSpot( anchorRuleKey, order )

      appProfileRuleConfig = \
               self.policyConfig.newAppProfilePolicyRuleList( newRuleKey )
      appProfileRuleConfig.appProfileName = profileName
      appProfileRuleConfig.actionName = oldActionName
      self.appProfileRuleConfig = appProfileRuleConfig
      self.profileNameToRuleKey[ profileName ] = newRuleKey

   def addProfileConfig( self, profileName, order, profileName2 ):
      if order is None or profileName == 'default' or profileName2 == 'default':
         self.addProfileConfigNaive( profileName )
      else:
         self.addProfileConfigComplete( profileName, order, profileName2 )
      if profileName in self.profileNameToRuleKey:
         self._ruleKey = self.profileNameToRuleKey[ profileName ]

   def delProfileConfig( self, profileName ):
      if profileName not in self.profileNameToRuleKey:
         self.mode.addError( '%s does not exist. Did you make a typo?'
               % profileName )
         return

      ruleKey = self.profileNameToRuleKey[ profileName ]
      del self.policyConfig.appProfilePolicyRuleList[ ruleKey ]
      del self.profileNameToRuleKey[ profileName ]
      self.appProfileRuleConfig = None

   def setActionName( self, actionName ):
      self.appProfileRuleConfig.actionName = actionName

   def setTrafficClass( self, trafficClass ):
      self.appProfileRuleConfig.trafficClass = trafficClass

   def setDscp( self, dscp ):
      self.appProfileRuleConfig.dscp = dscp

class AvtVrfConfigContext:
   def __init__( self, config, name ):
      self.config = config
      self._vrfName = name
      self._vrfConfig = None

   def vrfName( self ):
      return self._vrfName

   def vrfConfig( self ):
      return self._vrfConfig

   def addVrfConfig( self, name ):
      vrfConfig = self.config.vrfConfig.get( name )
      if not vrfConfig:
         vrfConfig = self.config.newVrfConfig( name )
         self.config.wanTEVrfConfig.addMember( vrfConfig )
      self._vrfConfig = vrfConfig

   def delVrfConfig( self, name ):
      vrfConfig = self.config.vrfConfig.get( name )
      if vrfConfig:
         del self.config.vrfConfig[ name ]
         del self.config.wanTEVrfConfig[ name ]
      self._vrfConfig = None

   def setPolicy( self, name ):
      self._vrfConfig.policyName = name

   def setAvtNameToId( self, name, idd ):
      self._vrfConfig.avtNameToId[ name ] = idd

   def delAvtNameToId( self, name ):
      del self._vrfConfig.avtNameToId[ name ]
