#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import (
   Dict,
   Int,
   Float,
   Model,
   Bool,
   List,
   Str,
   Submodel,
   Enum,
)
from ArnetModel import IpGenericAddress
from HumanReadable import formatValueSi
import IntfModels
import TableOutput
from Toggles.BgpCommonToggleLib import toggleEvpnL3DciWithDpsEnabled
from Toggles.WanTECommonToggleLib import toggleAvtLowestLoadMetricEnabled

fl = TableOutput.Format( justify='left' )
fr = TableOutput.Format( justify='right' )

class AvtCountersByVrfModel( Model ):

   class AvtCountersByAvtModel( Model ):

      class AvtCountersByVtepModel( Model ):

         class AvtCounterEntryModel( Model ):
            intf = IntfModels.Interface(
                      help='Local interface name for outgoing traffic' )
            flows = Int( help="Number of flows following the same path" )
            outBytes = Int( help="Number of out bytes following the same path" )
            outPkts = Int( help="Number of out pkts following the same path" )
            rate = Float( help="Throughput of the outgoing traffic" )

         # key is pathname
         paths = Dict( keyType=str,
                       valueType=AvtCounterEntryModel,
                       help="Summary of packet distribution on a path" )

      # key is vtep IP
      vteps = Dict( keyType=IpGenericAddress,
                    valueType=AvtCountersByVtepModel,
                    help="Summary of packet distribution on a VTEP IP" )

   # key is avt name
   avts = Dict( keyType=str,
                valueType=AvtCountersByAvtModel,
                help="Summary of packet distribution on an AVT" )

class AvtCountersModel( Model ):
   # vrf is key
   vrfs = Dict( keyType=str,
                 valueType=AvtCountersByVrfModel,
                 help="Summary of packet distribution on a VRF" )

   def render( self ):
      header = [ "VRF", "AVT", "First Hop", "Path", "Interfaces", "Flows *",
                 "Bytes", "Packets", "Rate" ]
      table = TableOutput.createTable( header )
      table.formatColumns( fl, fl, fl, fl, fl, fr, fr, fr, fr )

      for vrfName, countersByVrf in sorted( self.vrfs.items() ):
         for avtName, countersByAvt in sorted( countersByVrf.avts.items() ):
            for vtepIp, countersByVtep in sorted(
                                          countersByAvt.vteps.items() ):
               for nhPath, counterEntry in sorted(
                                           countersByVtep.paths.items() ):
                  rate = formatValueSi( rate=counterEntry.rate )
                  table.newRow( vrfName, avtName, vtepIp,
                                nhPath, counterEntry.intf.stringValue,
                                counterEntry.flows, counterEntry.outBytes,
                                counterEntry.outPkts, rate )

      print( table.output() )
      print( "* Flows = Active Flows" )

class AvtPathsFlags( Model ):
   __public__ = True
   valid = Bool( default=False, help='Valid path processed by forwarding engine' )
   active = Bool( default=False, help='Active path used by forwarding engine' )
   outlier = Bool( default=False,
                   help='Outlier path, not programmed in forwarding engine' )
   constraintNotMet = Bool( default=False,
                            help='Avt constraint not met for the path' )
   directPath = Bool( default=False, help='Direct path' )

class AvtSegmentLabel( Model ):
   __public__ = True
   label = Int( help='Segment label' )

class AvtPath( Model ):
   __public__ = True
   destination = IpGenericAddress( help='Destination VTEP address' )
   flags = Submodel( valueType=AvtPathsFlags, help='Path flags' )
   latency = Float( help='Latency of path in msecs', optional=True )
   jitter = Float( help='Jitter of path in msecs', optional=True )
   loss = Float( help='Loss in path in percentage', optional=True )
   if toggleAvtLowestLoadMetricEnabled():
      load = Float( help='Load in path in percentage', optional=True )
      totalCapacity = Int( help='Total capacity in path in Mbps', optional=True )
      availableCapacity = Int( help="Available capacity in path in Mbps",
                               optional=True )
   pathMtu = Int( help='Maximum transmission unit of path in bytes', optional=True )
   priority = Int( help='Priority of path', optional=True )
   nexthopAddr = IpGenericAddress( help='Immediate nexthop VTEP address',
                                   optional=True )
   multihopPathId = Int( help="Multihop path ID", optional=True )
   directPathId = Int( help="Direct path ID", optional=True )
   egressPathName = Str( help="Name of the egress multihop path",
                        optional=True )
   labelStack = List( valueType=AvtSegmentLabel,
                      help='Label stack of the multihop path', optional=True )

class AvtPathsByAvtModel( Model ):
   __public__ = True
   avtId = Int( help="Adaptive virtual topology identifier" )
   avtApplicationProfiles = List(
      valueType=str,
      help="Adaptive virtual topology application profiles" )
   avtPaths = Dict(
      keyType=str,
      valueType=AvtPath,
      help="Adaptive virtual topology paths, keyed by path type and ID" )

class AvtPathsByVrfModel( Model ):
   __public__ = True
   vrf = Str( help='VRF name' )
   vni = Int( help='VxLAN network identifier for VRF' )
   avts = Dict( keyType=str,
                valueType=AvtPathsByAvtModel,
                help="Path information for an adaptive virtual topology" )

class AvtPathsModel( Model ):
   __public__ = True
   ''' Model for show adaptive-virtual-topology paths '''
   routerId = IpGenericAddress( help='Router address', optional=True )
   role = Enum( values=( 'unknown', 'edge', 'zone transit', 'region transit',
                         'pathfinder' ),
               help='Role of the node' )
   regionName = Str( help="Adaptive virtual topology region name", optional=True )
   regionId = Int( help='Adaptive virtual topology region ID', optional=True )
   zoneName = Str( help="Adaptive vurtual topology zone name", optional=True )
   zoneId = Int( help='Adaptive virtual topology zone ID', optional=True )
   siteName = Str( help='Adaptive virtual topology site name', optional=True )
   siteId = Int( help='Adaptive virtual topology site ID', optional=True )
   vrfs = Dict( keyType=str,
                valueType=AvtPathsByVrfModel,
                help="Adaptive virtual topology path information for a VRF" )

class AvtProfileByAvtModel( Model ):
   __public__ = True
   appProfileName = Str( help="Application profile name" )
   appProfileId = Int( help="Application profile ID" )
   avtName = Str( help="Adaptive virtual topology service name" )
   avtId = Int( help="Adaptive virtual topology service id" )
   state = Str( help="If the setting is active in status" )

class AvtAppProfileByVrfModel( Model ):
   __public__ = True
   vrf = Str( help='VRF name' )
   vni = Int( help='VxLAN network identifier for VRF' )
   appProfiles = Dict( keyType=str,
                      valueType=AvtProfileByAvtModel,
                      help='''Application Profile information for an
                      adaptive topolgy''' )

class AvtAppProfileModel( Model ):
   __public__ = True
   ''' Model for show adaptive-virtual-topology application-profile '''
   vrfs = Dict( keyType=str,
                valueType=AvtAppProfileByVrfModel,
                help="Application profile information for a VRF" )

class AvtNextHopByVrfModel( Model ):
   __public__ = toggleEvpnL3DciWithDpsEnabled()
   dpsNextHops = List(
      valueType=IpGenericAddress,
      help='List of next hop VTEPs with dynamic path selection encapsulation' )
   vxlanNextHops = List( valueType=IpGenericAddress,
                         help='List of next hop VTEPs with VXLAN encapsulation' )

class AvtNextHopModel( Model ):
   ''' Model for show adaptive-virtual-topology next-hop '''
   __public__ = toggleEvpnL3DciWithDpsEnabled()
   vrfs = Dict( keyType=str, valueType=AvtNextHopByVrfModel,
                help='Next hop VTEPs per VRF' )

   def render( self ):
      for vrf, entries in sorted( self.vrfs.items() ):
         print( 'VRF:', vrf )
         if entries.dpsNextHops:
            print( 'VTEPs of tunnel encapsulation type path-selection' )
            for dpsNh in sorted( entries.dpsNextHops ):
               print( '  ', dpsNh )
         if entries.vxlanNextHops:
            print( 'VTEPs of tunnel encapsulation type vxlan' )
            for vxlanNh in sorted( entries.vxlanNextHops ):
               print( '  ', vxlanNh )
         print( '' )
