# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliGlobal
import LazyMount
import SharkLazyMount
import SmashLazyMount
from CliModel import cliPrinted
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.WanTEShowCli import getVniAvtId
from AgentCommandRequest import runCliPrintSocketCommand
from IpLibConsts import DEFAULT_VRF
import Tac

AvtCliModel = CliDynamicPlugin( "AvtCliModel" )
gv = CliGlobal.CliGlobal( dict( avtCounterTable=None,
                                dpsPathStatusDir=None,
                                avtNodeStatus=None,
                                avtCountersSnapShotReq=None,
                                avtCliConfig=None,
                                evpnStatus=None,
                                vrfIdMap=None ) )

def getAvtCounters( mode, args ):
   pathStatus = gv.dpsPathStatusDir.pathStatus.get( DEFAULT_VRF )
   if pathStatus is None:
      return {}

   vrfsDict = {}
   filterVrf = args.get( 'VRF' )
   filterAvt = args.get( 'AVT' )
   filterVtepIp = args.get( 'HOP' )
   filterPath = args.get( 'PATH-ID' )

   for vniId, avtCounterByVni in gv.avtCounterTable.vniStatus.items():
      if vniId not in gv.avtNodeStatus.vniToVrfName:
         continue
      vrf = gv.avtNodeStatus.vniToVrfName[ vniId ]
      if filterVrf and filterVrf != vrf:
         continue
      vrfCounter = AvtCliModel.AvtCountersByVrfModel()

      avtCounterSnapShotByVni = gv.avtCounterTable.vniSnapShotStatus.get( vniId )
      for avtId, avtCounterByAvt in avtCounterByVni.avtStatus.items():
         if avtId not in gv.avtNodeStatus.avtNameMap[ vrf ].avtIdToName:
            continue
         avtName = gv.avtNodeStatus.avtNameMap[ vrf ].avtIdToName[ avtId ]
         if filterAvt and filterAvt != avtName:
            continue
         avtCounter = AvtCliModel.AvtCountersByVrfModel.AvtCountersByAvtModel()

         avtCounterSnapShotByAvt = avtCounterSnapShotByVni.avtStatus.get( avtId )
         for vtepIp, avtCounterStatus in avtCounterByAvt.vtepIpStatus.items():
            if filterVtepIp and filterVtepIp != vtepIp.stringValue:
               continue
            vtepCounter = AvtCliModel.AvtCountersByVrfModel.AvtCountersByAvtModel. \
                          AvtCountersByVtepModel()

            avtCounterSnapShotStatus = avtCounterSnapShotByAvt.vtepIpStatus. \
                                          get( vtepIp )
            for pathId, avtCounterEntry in avtCounterStatus.avtCounterEntry.items():
               pathName = f"path{pathId}"
               if filterPath and int( filterPath ) != pathId:
                  continue

               avtCounterSnapShotEntry = avtCounterSnapShotStatus.avtCounterEntry. \
                                            get( pathId )
               snapShotOutBytes = 0
               snapShotOutPkts = 0
               # This check is needed because there can be a delay between the
               # counterEntry creation and the snapShotEntry creation.
               # This scenario ideally is impossible in the field, but in AutoTest
               # we hit this issue as deletion /creation of entries are all
               # happening very fast.
               if avtCounterSnapShotEntry:
                  snapShotOutBytes = avtCounterSnapShotEntry.outBytes
                  snapShotOutPkts = avtCounterSnapShotEntry.outPkts

               pathCounter = AvtCliModel.AvtCountersByVrfModel. \
                             AvtCountersByAvtModel.AvtCountersByVtepModel. \
                             AvtCounterEntryModel()
               if pathId not in pathStatus.pathStatusEntry:
                  continue
               localIntf = pathStatus.pathStatusEntry[ pathId ].localIntf
               pathCounter.intf = localIntf
               pathCounter.flows = avtCounterEntry.flows
               pathCounter.outBytes = avtCounterEntry.outBytes - snapShotOutBytes
               pathCounter.outPkts = avtCounterEntry.outPkts - snapShotOutPkts
               pathCounter.rate = avtCounterEntry.throughput
               vtepCounter.paths[ pathName ] = pathCounter

            avtCounter.vteps[ vtepIp ] = vtepCounter

         vrfCounter.avts[ avtName ] = avtCounter

      vrfsDict[ vrf ] = vrfCounter

   return vrfsDict

def ShowAvtPathCountersHandler( mode, args ):
   counters = AvtCliModel.AvtCountersModel()
   counters.vrfs = getAvtCounters( mode, args )
   return counters

def ClearAvtPathCountersHandler( mode, args ):
   gv.avtCountersSnapShotReq.snapShotRequestTime = Tac.now()

def doShowAvtAgentCommand( mode, params, cliModel ):
   model = runCliPrintSocketCommand( mode.entityManager,
                                     'Sfe',
                                     'adaptive-virtual-topology',
                                     params,
                                     mode,
                                     keepalive=True,
                                     connErrMsg='Sfe agent is inactive',
                                     model=cliModel )
   return cliPrinted( model ) if model else None

def ShowAvtPathsHandler( mode, args ):
   if not gv.avtCliConfig.avtConfigured:
      mode.addError( "Adaptive virtual topology is not configured" )
      return None

   params = 'path'
   vrfName = args.get( 'VRF' )
   avtName = args.get( 'AVT' )
   detail = args.get( 'detail' )
   # default VRF may not have an VNI
   _, avtId = getVniAvtId( mode, vrfName, avtName, verifyVni=False )
   if avtName and not avtId:
      return None

   if detail:
      params += ' detail'
   if vrfName:
      params += ' vrf=%s' % vrfName
   if avtId:
      params += ' avt=%s' % avtName
   dest = args.get( 'DEST' )
   if dest:
      params += ' destination=%s' % dest

   return doShowAvtAgentCommand( mode, params, AvtCliModel.AvtPathsModel )

def ShowAvtAppProfileHandler( mode, args ):
   if not gv.avtCliConfig.avtConfigured:
      mode.addError( "Adaptive virtual topology is not configured" )
      return None

   vrfName = args.get( 'VRF' )
   avtName = args.get( 'AVT' )
   appName = args.get( 'APPLICATION' )
   params = 'application-profile'

   if vrfName:
      params += ' vrf=%s' % vrfName
   if appName:
      params += ' application=%s' % appName
   if avtName:
      params += ' avt=%s' % avtName

   return doShowAvtAgentCommand( mode, params, AvtCliModel.AvtAppProfileModel )

def ShowAvtTunnelEncapHandler( mode, args ):
   if not gv.avtCliConfig.avtConfigured:
      mode.addError( "Adaptive virtual topology is not configured" )
      return None

   filterDps = args.get( 'path-selection' )
   filterVxlan = args.get( 'vxlan' )
   filterNone = not filterDps and not filterVxlan
   data = AvtCliModel.AvtNextHopModel()
   for nh, encapType in gv.evpnStatus.nexthopEncapType.items():
      vrfEntry = gv.vrfIdMap.vrfIdToName.get( nh.vrfId )
      if vrfEntry:
         vrfName = vrfEntry.vrfName
      else:
         vrfName = str( nh.vrfId )
      vrfData = data.vrfs.get( vrfName, AvtCliModel.AvtNextHopByVrfModel() )
      if encapType == Tac.Type( 'Evpn::EncapType' ).encapDps:
         if filterDps or filterNone:
            vrfData.dpsNextHops.append( nh.ipGenAddr )
      if encapType == Tac.Type( 'Evpn::EncapType' ).encapVxlan:
         if filterVxlan or filterNone:
            vrfData.vxlanNextHops.append( nh.ipGenAddr )
      if vrfData != AvtCliModel.AvtNextHopByVrfModel():
         # do not bother to add this vrf if no entry in it
         data.vrfs[ vrfName ] = vrfData
   return data

def Plugin( em ):
   gv.avtCountersSnapShotReq = LazyMount.mount(
      em,
      'avt/cli/counters/snapshot/request',
      'AvtCounters::AvtCountersSnapShotRequest', 'w' )

   gv.avtNodeStatus = LazyMount.mount( em, 'avt/node/status',
                                       'Avt::AvtNodeStatus', 'r' )
   gv.dpsPathStatusDir = LazyMount.mount( em, 'dps/path/status',
                                          'Dps::PathStatusDir', 'r' )
   gv.avtCounterTable = SharkLazyMount.mount( em, 'avt/shark/counters',
                                              'AvtCounters::AvtCounterTable',
                                              SharkLazyMount.mountInfo( 'shadow' ),
                                              True )
   gv.avtCliConfig = LazyMount.mount( em, 'avt/input/cli', 'Avt::AvtCliConfig', 'r' )
   gv.evpnStatus = LazyMount.mount( em, 'evpn/status', 'Evpn::EvpnStatus', 'r' )

   gv.vrfIdMap = SmashLazyMount.mount( em, "vrf/vrfIdMapStatus",
                                       "Vrf::VrfIdMap::Status",
                                       SmashLazyMount.mountInfo( 'reader' ),
                                       autoUnmount=True )
