# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import (
   Ip4Address,
   Ip6Address,
   IpGenericPrefix,
)
from CliModel import (
   Bool,
   DeferredModel,
   Enum,
   GeneratorDict,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from CliPlugin.BgpCliModels import (
   BgpRouteASPathEntry,
   BgpRouteDetailEntry,
   BgpRoutePeerEntry,
   BgpRouteTypeEntry,
   BgpLsUnknownTlv,
   REASON_NOT_BEST_LIST,
)
from CliPlugin.VrfCli import generateVrfCliModel

BGP_LS_NLRI_TYPES = (
   'node',
   'link',
   'ipv4Prefix',
   'ipv6Prefix',
)

BGP_LS_PROTOCOL_IDS = (
   'isisLevel1',
   'isisLevel2',
   'bgpDps',
   'bgp', # BGP-LS EPE (see AID11677)
)

BGP_LS_IGP_ROUTER_ID_TYPES = (
   'invalid', # igp router id is invalid when used with BGP path-selection
   'isisSystemId',
   'isisPseudoNodeId',
   'bgpRouterId' # BGP-LS EPE (see AID11677)
)

class BgpLsMultiTopologyId( Model ):
   topologyId = Int( help='Multi-Topology ID' )
   topologyName = Str( optional=True,
                       help='String representation of Multi-Topology ID' )

class BgpLsNodeDescriptor( Model ):
   __revision__ = 2
   igpRouterIdType = Enum( values=BGP_LS_IGP_ROUTER_ID_TYPES,
                           help='Type of IGP router identifier' )
   # optional because in the future for Ospf this field will be absent. For now,
   # it'll always be present
   isisSystemId = Str( help='IS-IS system Identifier', optional=True )
   isisPnodeId = Str( help='IS-IS pseudonode Identifier', optional=True )
   # When we add Ospf support we can add ospfRouterId and ospfDrRouterId, ospfDrIntf
   asn = Str( help='Autonomous System Number of BGP-LS producer', optional=True )
   # optional and use with BGP path-selection. This field will be absent for IGP.
   bgpRouterId = Ip4Address( help='BGP router Identifier', optional=True )
   unknownTlvs = List( valueType=BgpLsUnknownTlv,
                       help="List of unknown Node Descriptor Sub TLVs" )

class BgpLsLinkDescriptor( Model ):
   localIpv4Addrs = List( valueType=Ip4Address,
                          help='List of local IPv4 interface addresses' )
   localIpv6Addrs = List( valueType=Ip6Address,
                          help='List of local IPv6 interface addresses' )
   remoteIpv4Addrs = List( valueType=Ip4Address,
                          help='List of remote IPv4 interface addresses' )
   remoteIpv6Addrs = List( valueType=Ip6Address,
                          help='List of remote IPv6 interface addresses' )
   multiTopology = Submodel( optional=True,
                             valueType=BgpLsMultiTopologyId,
                             help='Multi-Topology ID' )
   localWanId = Int( optional=True, help='Local WAN Identifier' )
   remoteWanId = Int( optional=True, help='Remote WAN Identifier' )
   localWanGroupId = Int( optional=True, help='Local WAN group Identifier' )
   unknownTlvs = List( valueType=BgpLsUnknownTlv,
                       help="List of unknown Link Descriptor Sub TLVs" )

class BgpLsPrefixDescriptor( Model ):
   prefix = IpGenericPrefix( help='Prefix originated in IGP' )
   multiTopology = Submodel( optional=True,
                             valueType=BgpLsMultiTopologyId,
                             help='Multi-Topology ID' )
   unknownTlvs = List( valueType=BgpLsUnknownTlv,
                       help="List of unknown Prefix Descriptor Sub TLVs" )

class BgpLsRoutePath( DeferredModel ):
   asPathEntry = Submodel( valueType=BgpRouteASPathEntry,
                           help='AS path information' )

   peerEntry = Submodel( valueType=BgpRoutePeerEntry, optional=True,
                         help='Peer information' )
   nextHop = Str( optional=True, help='Next hop address' )
   med = Int( optional=True, help='Multi Exit Discriminator' )
   localPreference = Int( optional=True, help='I-BGP Local preference indicator' )
   weight = Int( optional=True, help='Weight for the path' )
   timestamp = Int( optional=True,
                    help='UTC seconds since epoch when the path was received.' )

   routeType = Submodel( valueType=BgpRouteTypeEntry, help='Route type' )
   routeDetail = Submodel( valueType=BgpRouteDetailEntry, optional=True,
                           help='Route details' )
   reasonNotBestpath = Enum( values=REASON_NOT_BEST_LIST,
                             help='Reason route was not selected as BGP best path' )

class BgpLsRouteEntry( DeferredModel ):
   # Report the total number of paths as a separate attribute because the path list
   # may be filtered by peer.
   nlriType = Enum( values=BGP_LS_NLRI_TYPES,
                    help='NLRI type' )
   protocolId = Enum( values=BGP_LS_PROTOCOL_IDS,
                      help='IGP source protocol' )
   identifier = Int( help='IGP Instance Identifier' )
   localNodeDescriptor = Submodel( valueType=BgpLsNodeDescriptor,
                                   help='Identity of local node' )
   remoteNodeDescriptor = Submodel( valueType=BgpLsNodeDescriptor, optional=True,
                                    help='Identity of remote node for link NLRIs' )
   linkDescriptor = Submodel( valueType=BgpLsLinkDescriptor, optional=True,
                              help='Identity of link for link NLRIs' )
   # Using a sub model as in the future with MT support it'll have the MT ID
   # Also a structure containing descriptors is the natural representation of a
   # BGP-LS NLRI
   prefixDescriptor = Submodel( valueType=BgpLsPrefixDescriptor, optional=True,
                           help='Identity of prefix for IPv4 and IPv6 prefix NLRIs' )
   # Report the total number of paths as a separate attribute because the path list
   # may be filtered by peer.
   totalPaths = Int( help='Total number of paths for this BGP-LS route' )
   paths = List( valueType=BgpLsRoutePath,
                 help='BGP-LS paths' )
   _routeKey = Str( help='String representation of the BGP-LS NLRI' )

class BgpLsRoutes( DeferredModel ):
   __revision__ = 2
   vrf = Str( help='VRF name' )
   routerId = Ip4Address( help='BGP Router Identity' )
   asn = Str( help='Autonomous System Number' )
   _detail = Bool( optional=True, help='Detailed output is requested' )
   bgpLsRoutes = GeneratorDict( keyType=str, valueType=BgpLsRouteEntry,
                                help='Dictionary of BGP Link State route entries '
                                'keyed by a string representation of the NLRI' )

BgpLsRoutesVrfModel = generateVrfCliModel( BgpLsRoutes,
                                           'BGP Link State routes' )
