#!/usr/bin/env python3
# Copyright (c) 2014, 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from ArnetLib import bgpFormatAsn
from ArnetModel import Ip4Address
from CliModel import (
   Bool,
   Dict,
   GeneratorDict,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from CliPlugin.BgpCliModels import (
   _degradePeersDict,
   _degradeSubmodel,
   _degradeToIpGenAddress,
   peerKeyFromData
)
from CliPlugin.MaintenanceCliLib import maintenanceStateToStr
from CliPlugin.RouteMapCliModels import RouteMaps

indentLevel1 = " "
indentIncrement = "  "
indentLevel2 = indentLevel1 + indentIncrement
indentLevel3 = indentLevel2 + indentIncrement

class BgpPeerMaintenanceStatus( Model ):
   underMaintenance = Bool( default=False, help='Peering under maintenance' )
   appliedRouteMap = Str(
      help="Maintenance inout route map applied for the peer",
      optional=True )
   appliedRouteMapIn = Str(
      help="Maintenance inbound route map applied for the peer",
      optional=True )
   appliedRouteMapOut = Str(
      help="Maintenance outbound route map applied for the peer",
      optional=True )

   def getKey( self, data ):
      return peerKeyFromData( data )

   def processData( self, data ):
      if 'maintenance' in data:
         data[ 'underMaintenance' ] = data[ 'maintenance' ]
         del data[ 'maintenance' ]
         data[ 'underMaintenance' ] = ord( data[ 'underMaintenance' ] ) != 0
      if 'maintenanceRmIn' in data:
         data[ 'appliedRouteMapIn' ] = data[ 'maintenanceRmIn' ]
         del data[ 'maintenanceRmIn' ]
      if 'maintenanceRmOut' in data:
         data[ 'appliedRouteMapOut' ] = data[ 'maintenanceRmOut' ]
         del data[ 'maintenanceRmOut' ]
      # if both in/out rms the same, set appliedRouteMap for b/w compatibility
      if 'appliedRouteMapIn' in data and 'appliedRouteMapOut' in data and \
         data[ 'appliedRouteMapIn' ] == data[ 'appliedRouteMapOut' ]:
         data[ 'appliedRouteMap' ] = data[ 'appliedRouteMapOut' ]
      return data

   def renderEntry( self, peer ):
      print( indentLevel1, 'Neighbor: %s' % peer )
      if self.underMaintenance:
         print( indentLevel2, 'Maintenance state: %s' %
                maintenanceStateToStr[ 'underMaintenance' ] )
      else:
         print( indentLevel2, maintenanceStateToStr[ 'active' ] )

class BgpMaintenanceStatus( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         dictRepr = _degradePeersDict( dictRepr )
      return dictRepr

   asn = Int( help='Autonomous System Number' )
   _asdot = Bool( default=False, help='Asdot notation enabled' )
   routerId = Ip4Address( help='BGP Router Identity' )
   vrf = Str( help='VRF Name' )
   peers = GeneratorDict( valueType=BgpPeerMaintenanceStatus,
                          help='Dictionary of BGP peer entries indexed by the '
                          'peer address' )

   def setAsdot( self, asdot ):
      self._asdot = asdot

   def render( self ):
      print( 'BGP peer maintenance information for VRF', self.vrf )
      print( 'Router identifier %s, local AS number %s' %
             ( self.routerId,
               bgpFormatAsn( self.asn, self._asdot ) ) )
      for addr, peer in self.peers:
         peer.renderEntry( addr )

         if peer.appliedRouteMapIn:
            print( indentLevel2, 'Maintenance inbound route map: %s' %
                   peer.appliedRouteMapIn )
         if peer.appliedRouteMapOut:
            print( indentLevel2, 'Maintenance outbound route map: %s' %
                   peer.appliedRouteMapOut )

class BgpMaintenanceStatusPerPeer( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         dictRepr = _degradePeersDict( dictRepr )
      return dictRepr

   asn = Int( help='Autonomous System Number' )
   routerId = Ip4Address( help='BGP Router Identity' )
   vrf = Str( help='VRF Name' )
   peers = Dict( valueType=BgpPeerMaintenanceStatus,
                 help='Dictionary of BGP peer entries indexed by the '
                 'peer address' )
      
class BgpPeerMaintenanceInfo( Model ):
   __revision__ = 5

   def degrade( self, dictRepr, revision ):
      if revision < 3:
         dictRepr[ 'peer' ] = _degradeToIpGenAddress( dictRepr[ 'peer' ] )
         dictRepr[ 'status' ] = _degradeSubmodel( BgpMaintenanceStatusPerPeer,
                                                 dictRepr, 'status', revision - 1 )
      return dictRepr

   peer = Str( help='Neighbor address' )
   groups = List( help='List of Groups that the peer is part of',
                  valueType=str )
   selectedProfile = Str( help='Selected profile from BGP groups',
                          optional=True )
   status = Submodel( valueType=BgpMaintenanceStatusPerPeer,
                      help='BGP Maintenance status of this peer' )
   routeMapInfo = Submodel( valueType=RouteMaps,
                            help='Route map information' )
   routeMapInfoIn = Submodel( valueType=RouteMaps,
                              help='Inbound route map information' )
   routeMapInfoOut = Submodel( valueType=RouteMaps,
                               help='Outbound route map information' )
   _asdot = Bool( default=False, help='Asdot notation enabled' )

   def setAsdot( self, asdot ):
      self._asdot = asdot

   def render( self ):

      peerLinePrinted = False
      peer = None
      if self.status:

         print( 'BGP peer maintenance information for VRF', self.status.vrf )
         print( 'Router identifier %s, local AS number %s' %
                ( self.status.routerId,
                  bgpFormatAsn( self.status.asn, self._asdot ) ) )
         for addr, peer in self.status.peers.items():
            peerLinePrinted = True
            peer.renderEntry( addr )

      if not peerLinePrinted:
         print( indentLevel1, 'Neighbor: %s' % self.peer )

      if peer and peer.appliedRouteMapIn:
         print( indentLevel2, 'Maintenance inbound route map: %s' %
                peer.appliedRouteMapIn )
         if self.routeMapInfoIn:
            self.routeMapInfoIn.renderRouteMaps( indentLevel3 )
      if peer and peer.appliedRouteMapOut:
         print( indentLevel2, 'Maintenance outbound route map: %s' %
                peer.appliedRouteMapOut )
         if self.routeMapInfoOut:
            self.routeMapInfoOut.renderRouteMaps( indentLevel3 )

      if len( self.groups ):
         print( indentLevel2, 'Groups: %s' % ', '.join( self.groups ) )
      print( indentLevel2, 'Selected profile from BGP groups: %s'
             % ( self.selectedProfile or "<none>" ) )

class BgpMaintenanceDefaultProfile( Model ):
   __revision__ = 5
   profileName = Str( help='Profile name' )
   routeMap = Str( help='Default route map name', optional=True )
   routeMapIn = Str( help='Default inbound route map name', optional=True )
   routeMapOut = Str( help='Default outbound route map name', optional=True )
   routeMapInfo = Submodel( valueType=RouteMaps,
                            help='Route map information' )
   routeMapInfoIn = Submodel( valueType=RouteMaps,
                              help='Inbound route map information' )
   routeMapInfoOut = Submodel( valueType=RouteMaps,
                               help='Outbound route map information' )

   def render( self ):
      print( 'Bgp Profile: %s' % self.profileName )
      rmIn = self.routeMapIn if self.routeMapIn else "(none)"
      print( '  Inbound initiator route map: %s' % rmIn )
      if self.routeMapInfoIn:
         self.routeMapInfoIn.renderRouteMaps( indentLevel2 )
      rmOut = self.routeMapOut if self.routeMapOut else "(none)"
      print( '  Outbound initiator route map: %s' % rmOut )
      if self.routeMapInfoOut:
         self.routeMapInfoOut.renderRouteMaps( indentLevel2 )
