#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from BgpLib import (
   createKey,
   PeerConfigKey,
)
import Cell
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.MaintenanceModels import (
   BgpGroup, 
   BgpGroups, 
   BgpProfile,
   BgpProfiles,
   BgpNeighbors, 
   MaintenanceEventInfoEntry,
   MaintenanceEventInfo,
   MaintenanceDebugInfo,
   MaintenanceInterfaceBgpStatus,
)
from CliPlugin.MaintenanceCliLib import (
   toUtc,
   bgpGroupType,
   bgpProfileType,
   createGroupKey,
   defaultProfile,
   dynamicUnitComponentName,
   dynamicUnitVrfName,
   dynamicUnitName,
   isDynamicUnit,
   maintRunningShowCliCheck,
   maintEnterStageClass,
   maintExitStageClass
)
from CliPlugin.RouteMapCli import routeMapContainerModel
from CliPlugin.RoutingBgpCli import bgpNeighborConfig
from CliPlugin.RoutingBgpShowCli import (
   ArBgpShowOutput, 
   EmptyResponseException,
   showRibCapiCommand,
)
from CliPlugin.VrfCli import (
   VrfExecCmdDec, 
   generateVrfCliModel, 
   getVrfNames,
)
from DeviceNameLib import eosIntfToKernelIntf
from IpLibConsts import DEFAULT_VRF
import LazyMount
import six
import Tac

BgpMaintenanceModels = CliDynamicPlugin( "BgpMaintenanceModels" )

vrfNbrToGroupDir = None
bgpGroupStatusDir = None
allIntfStatusDir = None
bgpDefaultProfileDir = None
bgpProfileStatusDir = None
ipStatus = None
unitStatusDir = None
maintEnterInstanceLog = None
maintExitInstanceLog = None
mapConfig = None
maintenanceGroupDir = None
defaultReceiverRmDir = None
groupToUnitDir = None
defaultRmName = Tac.Type( 'BgpMaintenance::Defaults' ).rmName
defaultInitiatorRmDir = None

GroupOrigin = Tac.Type( "Group::GroupOrigin" )

def showBgpGroup( bgpGroup ):
   ''' Returns the populated BGP group
       Note: This function expects bgpGroup to be valid
   '''
   ret = BgpGroup()
   ret.origin = bgpGroup.origin
   retPeerGroup = BgpNeighbors()
   retPeerIpv4 = BgpNeighbors()
   retPeerIpv6 = BgpNeighbors()
   for bgpPeer in bgpGroup.neighbor:
      peerStr = bgpPeer.stringValue
      if bgpPeer.type == 'peerGroup':
         retPeerGroup.peers.append( peerStr )
      elif bgpPeer.type == 'peerIpv4':
         retPeerIpv4.peers.append( peerStr )
      elif bgpPeer.type in [ 'peerIpv6', 'peerIpv6Ll' ] :
         retPeerIpv6.peers.append( peerStr )
   bgpPeerDict = { 'peerGroup' : retPeerGroup,
                   'peerIpv4': retPeerIpv4 ,
                   'peerIpv6' : retPeerIpv6 }
   ret.neighbors = bgpPeerDict
   groupName = bgpGroup.groupName
   groupKey = Tac.Value( "Group::GroupKey", bgpGroupType, groupName )
   maintGroup = maintenanceGroupDir.maintenanceGroup.get( groupKey )
   if bgpDefaultProfileDir.profileName == defaultProfile:
      ret.bgpProfile = 'Default'
   else:
      ret.bgpProfile = bgpDefaultProfileDir.profileName
   if maintGroup:
      for profile in maintGroup.profile:
         if profile.type == bgpProfileType: 
            ret.bgpProfile = profile.name
   ret.vrfName = bgpGroup.vrfName
   bgpGroupKey = createGroupKey( bgpGroupType, groupName )
   bgpGroupToUnit = groupToUnitDir.groupToUnit.get( bgpGroupKey ) 
   if bgpGroupToUnit:
      for unit in bgpGroupToUnit.unitName:
         ret.units.append( unit )
   
   return ret

def showBgpGroups_( mode, bgpGroupName=None ):
   allBgpGroups = {}
   if not bgpGroupName:
      for groupStatus in bgpGroupStatusDir.status.values():
         if groupStatus.origin == GroupOrigin.dynamic:
            continue
         bgpGroup = showBgpGroup( groupStatus )
         allBgpGroups[ groupStatus.groupName ] = bgpGroup
   else:
      bgpGroupStatus = \
         bgpGroupStatusDir.status.get( bgpGroupName )
      if bgpGroupStatus:
         bgpGroup = showBgpGroup( bgpGroupStatus )
         allBgpGroups[ bgpGroupName ] = bgpGroup
   return allBgpGroups

@maintRunningShowCliCheck
def showBgpGroups( mode, bgpGroupName=None ):
   allBgpGroups = showBgpGroups_( mode, bgpGroupName )
   return BgpGroups( bgpGroups=allBgpGroups )

def showBgpGroupsHook( mode, groupsModel, bgpGroupName=None ):
   groupsModel.bgpGroups = showBgpGroups_( mode, bgpGroupName )

def ShowMaintenanceGroupsCmd_handler( mode, args ):
   return showBgpGroups( mode, bgpGroupName=args.get( "GROUP_NAME" ) )

def getBgpGroupProfileInfo( mode, peer, vrfName, bgpPeerMaintInfo ):
   peerKey = peer
   if vrfName not in vrfNbrToGroupDir.vrfBgpNeighborToGroup:
      return
   nbrToGroupDir = vrfNbrToGroupDir.vrfBgpNeighborToGroup[ vrfName ]
   if not nbrToGroupDir:
      return
   if not peerKey in nbrToGroupDir.bgpNeighborToGroup:
      # Look if the peer group has any groups associated
      peerConfig = bgpNeighborConfig( peerKey, vrfName, False )
      if peerConfig and peerConfig.isPeerGroupPeer:
         peerKey = peerConfig.peerGroupKey
         if not peerKey in nbrToGroupDir.bgpNeighborToGroup:
            return
      else:
         return
   for group in nbrToGroupDir.bgpNeighborToGroup[ peerKey ].group:
      bgpGroup = bgpGroupStatusDir.status.get( group.name )
      if bgpGroup.origin == GroupOrigin.dynamic:
         continue
      bgpPeerMaintInfo.groups.append( group.name )

   selectedProfile = nbrToGroupDir.bgpNeighborToGroup[ peerKey ].selectedProfile
   systemDefaultProfileName = defaultProfile
   # TODO: If the selected profile is not set, the best profile from the configured
   # groups needs to be shown as the best profile.
   if bgpProfileType in selectedProfile:
      profile = selectedProfile[ bgpProfileType ]
      bgpPeerMaintInfo.selectedProfile = profile.name
   if bgpPeerMaintInfo.selectedProfile == systemDefaultProfileName:
      if bgpDefaultProfileDir.profileName == defaultProfile:
         bgpPeerMaintInfo.selectedProfile = 'Default'
      else:
         bgpPeerMaintInfo.selectedProfile = bgpDefaultProfileDir.profileName

def ShowAddrVrfCmd_handler( mode, args ):
   return doShowMaintenanceBgp( mode, args[ "PEER" ], vrfName=args.get( "VRF" ) )

def populateEventInfo( maintEvents, logs ):
   for logEntry in six.itervalues( logs ):
      entry = MaintenanceEventInfoEntry()
      entry.timeStamp = toUtc( logEntry.timestamp )
      entry.eventInfo = logEntry.msg
      maintEvents.append( entry )

def populateDebugInfo( unitStatus ):
   maintenanceEventInfo = MaintenanceEventInfo()
   instanceLogs = maintEnterInstanceLog.instanceLog.get(
      unitStatus.maintEnterInstance, None )
   if instanceLogs:
      populateEventInfo( maintenanceEventInfo.maintEnterEvents,
                         instanceLogs.log )
   instanceLogs = maintExitInstanceLog.instanceLog.get(
      unitStatus.maintExitInstance, None )
   if instanceLogs:
      populateEventInfo( maintenanceEventInfo.maintExitEvents,
                         instanceLogs.log )

   return maintenanceEventInfo

@maintRunningShowCliCheck
def showMaintenanceAllBgpDebugInfo( mode, ignoreDisplay=False ):
   allDebugInfo = {}
   for unitStatus in unitStatusDir.status.values():
      if isDynamicUnit( unitStatus ):
         group = list( unitStatus.group )[ 0 ]
         peerString = createKey( dynamicUnitComponentName( unitStatus ) ).stringValue
         if group.type == bgpGroupType:
            vrf = dynamicUnitVrfName( unitStatus )
            allDebugInfo[ ( 'Bgp Peer %s(vrf-%s)' % ( peerString, vrf ) ) ] = \
                populateDebugInfo( unitStatus )
   return MaintenanceDebugInfo( maintenanceEventInfo=allDebugInfo )

def ShowMaintenanceDebugBgpCmd_handler( mode, args ):
   return showMaintenanceAllBgpDebugInfo( mode )

@maintRunningShowCliCheck
def doShowMaintenanceBgpDefaultProfile_handler( mode, args ):
   bgpDefaultProfile = BgpMaintenanceModels.BgpMaintenanceDefaultProfile()
   defaultProfileName = bgpDefaultProfileDir.profileName
   if defaultProfileName == defaultProfile:
      bgpDefaultProfile.profileName = 'Default'
      bgpDefaultProfile.routeMap = defaultRmName
      bgpDefaultProfile.routeMapIn = defaultRmName
      bgpDefaultProfile.routeMapOut = defaultRmName

      bgpDefaultProfile.routeMapInfo = routeMapContainerModel(
         defaultInitiatorRmDir, defaultRmName )
      bgpDefaultProfile.routeMapInfoIn = routeMapContainerModel(
         defaultInitiatorRmDir, defaultRmName )
      bgpDefaultProfile.routeMapInfoOut = routeMapContainerModel(
         defaultInitiatorRmDir, defaultRmName )
   else:
      bgpDefaultProfile.profileName = defaultProfileName
      bgpDefaultProfileStatus = bgpProfileStatusDir.status.get(
         defaultProfileName, None )
      if not bgpDefaultProfileStatus:
         bgpDefaultProfile.routeMapIn = ""
         bgpDefaultProfile.routeMapOut = ""
         bgpDefaultProfile.routeMap = ""
      else:
         bgpDefaultProfile.routeMapIn = bgpDefaultProfileStatus.policyIn.policyName
         bgpDefaultProfile.routeMapOut = bgpDefaultProfileStatus.policyOut.policyName
         if bgpDefaultProfile.routeMapIn == bgpDefaultProfile.routeMapOut:
            bgpDefaultProfile.routeMap = bgpDefaultProfile.routeMapIn
      bgpDefaultProfile.routeMapInfo = routeMapContainerModel(
         mapConfig, bgpDefaultProfile.routeMap )
      bgpDefaultProfile.routeMapInfoIn = routeMapContainerModel(
         mapConfig, bgpDefaultProfile.routeMapIn )
      bgpDefaultProfile.routeMapInfoOut = routeMapContainerModel(
         mapConfig, bgpDefaultProfile.routeMapOut )

   return bgpDefaultProfile

@maintRunningShowCliCheck
def showMaintenanceBgpDebugInfo( mode, peer, vrf=None ):
   allDebugInfo = {}
   if not vrf:
      vrf = 'default'
   peerKey = createKey( peer )
   peerString = peerKey.stringValue
   peerUnitName = dynamicUnitName( peerString, vrf )
   peerUnitStatus = unitStatusDir.status.get( peerUnitName )
   if peerUnitStatus:
      allDebugInfo[ ( 'Bgp Peer %s(vrf-%s)' % ( peerString, vrf ) ) ] = \
            populateDebugInfo( peerUnitStatus )
   return MaintenanceDebugInfo( maintenanceEventInfo=allDebugInfo )

def ShowPeerDebugCmd_handler( mode, args ):
   return showMaintenanceBgpDebugInfo( mode, args[ "PEER" ],
                                       vrf=args.get( "VRF" ) )

desc = "Per VRF BGP peer maintenance status information"
BgpMaintenanceStatusVrf = generateVrfCliModel(
   BgpMaintenanceModels.BgpMaintenanceStatus, desc )

def _doShowMaintenanceBgp( mode, ipv6, peer, vrfName,
                           statusModel=BgpMaintenanceModels.BgpMaintenanceStatus ):
   args = { 'ipv6' : ipv6 }
   cmd = 'MIO_DGET_BGP_MAINTENANCE'
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if peer is not None:
      peerConfigKey = PeerConfigKey( peer )
      if peerConfigKey.type == 'peerIpv4':
         args[ 'peerv4' ] = peerConfigKey.stringValue
      elif peerConfigKey.type in [ 'peerIpv6' ]:
         args[ 'peerv6' ] = peerConfigKey.stringValue
      elif peerConfigKey.type in [ 'peerIpv6Ll' ]:
         args[ 'peerv6' ] = peerConfigKey.v6Addr.stringValue + '%' + \
                             eosIntfToKernelIntf( peerConfigKey.llIntf )
   try:
      return showRibCapiCommand()( mode, statusModel, cmd, args,
                                   clientName='BGP' )
   except EmptyResponseException():
      return None

def getBgpMaintInfo( 
      mode, peer, vrfName, statusModel=BgpMaintenanceModels.BgpMaintenanceStatus ):
   ipv6 = peer.type in [ 'peerIpv6', 'peerIpv6Ll' ]
   return _doShowMaintenanceBgp( mode, ipv6, peer, vrfName, statusModel )

@maintRunningShowCliCheck
@ArBgpShowOutput( 'doShowMaintenanceBgpIpAll' )
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=BgpMaintenanceStatusVrf )
def doShowMaintenanceBgpIpAll( mode, vrfName=None ):
   return _doShowMaintenanceBgp( mode, False, None, vrfName )

def handlerShowMaintenanceBgpIpCmd( mode, args ):
   return doShowMaintenanceBgpIpAll( mode, vrfName=args.get( "VRF" ) )

@maintRunningShowCliCheck
@ArBgpShowOutput( 'doShowMaintenanceBgpIp6All' )
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=BgpMaintenanceStatusVrf )
def doShowMaintenanceBgpIpv6All( mode, vrfName=None ):
   return _doShowMaintenanceBgp( mode, True, None, vrfName )

def handlerShowMaintenanceBgpIPv6Cmd( mode, args ):
   return doShowMaintenanceBgpIpv6All( mode, vrfName=args.get( "VRF" ) )

desc = "Per VRF BGP peer maintenance information"
BgpPeerMaintenanceInfoVrf = generateVrfCliModel(
   BgpMaintenanceModels.BgpPeerMaintenanceInfo, desc, revision=5 )

@maintRunningShowCliCheck
@ArBgpShowOutput( 'doShowMaintenanceBgp' )
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=BgpPeerMaintenanceInfoVrf )
def doShowMaintenanceBgp( mode, peer, vrfName=None ):
   maintBgp = BgpMaintenanceModels.BgpPeerMaintenanceInfo()

   maintBgp.peer = peer.stringValue
   bgpPeerStatus = getBgpMaintInfo( 
      mode, peer, vrfName, 
      statusModel=BgpMaintenanceModels.BgpMaintenanceStatusPerPeer )
   rmName = None
   rmNameIn = None
   rmNameOut = None
   if bgpPeerStatus and bgpPeerStatus.peers.get( peer.stringValue ):
      maintBgp.status = bgpPeerStatus
      rmName = bgpPeerStatus.peers[ peer.stringValue ].appliedRouteMap
      rmNameIn = bgpPeerStatus.peers[ peer.stringValue ].appliedRouteMapIn
      rmNameOut = bgpPeerStatus.peers[ peer.stringValue ].appliedRouteMapOut
   if rmName:
      if rmName.lower() == defaultRmName.lower():
         maintBgp.routeMapInfo = routeMapContainerModel( defaultInitiatorRmDir,
                                                         defaultRmName )
      else:
         maintBgp.routeMapInfo = routeMapContainerModel( mapConfig, rmName )
   if rmNameIn:
      if rmNameIn.lower() == defaultRmName.lower():
         maintBgp.routeMapInfoIn = routeMapContainerModel( defaultInitiatorRmDir,
                                                           defaultRmName )
      else:
         maintBgp.routeMapInfoIn = routeMapContainerModel( mapConfig, rmNameIn )
   if rmNameOut:
      if rmNameOut.lower() == defaultRmName.lower():
         maintBgp.routeMapInfoOut = routeMapContainerModel( defaultInitiatorRmDir,
                                                            defaultRmName )
      else:
         maintBgp.routeMapInfoOut = routeMapContainerModel( mapConfig, rmNameOut )

   getBgpGroupProfileInfo( mode, peer, vrfName, maintBgp )
   return maintBgp

# For RibBgp : If maintenance mode is not enabled, routemap will not be displayed
# instead Maintenance Mode is disabled warning will be displayed.
@ArBgpShowOutput( 'doShowMaintenanceBgpReceiverRm' )
def doShowMaintenanceBgpReceiverRm( mode, args ):
   return routeMapContainerModel( defaultReceiverRmDir, defaultRmName )


def showBgpProfile( profileStatus ):
   ''' Returns the populated Bgp Profile
       Note: This function expects the profileStatus to be valid
   '''
   assert( profileStatus ) # pylint: disable=superfluous-parens
   bgpProfile = BgpProfile()
   bgpProfile.routeMapIn = profileStatus.policyIn.policyName
   bgpProfile.routeMapOut = profileStatus.policyOut.policyName
   bgpProfile.routeMapInOut = profileStatus.policyInOut.policyName

   return bgpProfile

def showBgpProfiles_( mode, bgpProfileName=None ):
   allBgpProfiles = {}
   if not bgpProfileName:
      for profile in bgpProfileStatusDir.status:
         if profile == defaultProfile:
            continue
         profileStatus = bgpProfileStatusDir.status.get( profile )
         if profileStatus:
            bgpProfile = showBgpProfile( profileStatus )
            allBgpProfiles[ profile ] = bgpProfile
   else:
      profileStatus = bgpProfileStatusDir.status.get( bgpProfileName )
      if profileStatus:
         bgpProfile = showBgpProfile( profileStatus )
         allBgpProfiles[ bgpProfileName ] = bgpProfile
   return allBgpProfiles

@maintRunningShowCliCheck
def showBgpProfiles( mode, bgpProfileName=None ):
   allBgpProfiles = showBgpProfiles_( mode, bgpProfileName )
   return BgpProfiles( bgpProfiles=allBgpProfiles )

def showBgpProfilesHook( mode, profilesModel, bgpProfileName=None ):
   profilesModel.bgpProfiles = showBgpProfiles_( mode, bgpProfileName )

def handlerShowMaintenanceProfilesCmd( mode, args ):
   return showBgpProfiles( mode, bgpProfileName=args.get( "PROFILE_NAME" ) )

def getVrf( intfId ):
   ipIntfStatus = ipStatus.ipIntfStatus.get( intfId.stringValue )
   if not ipIntfStatus:
      return None
   return ipIntfStatus.vrf

@ArBgpShowOutput( 'bgpShowMaintenanceIntf' )
# pylint: disable-next=inconsistent-return-statements
def bgpShowMaintenanceIntf( mode, intfId, status ):
   intfStatus = allIntfStatusDir.intfStatus.get( intfId )
   if not intfStatus or intfStatus.deviceName == "":
      return None
   args = { 'if_name' : intfStatus.deviceName }

   vrfName = getVrf( intfId )
   if not vrfName:
      return None
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   try:
      cmd = 'MIO_DGET_BGP_MAINTENANCE_IF'
      status.bgpStatus = showRibCapiCommand()( mode, MaintenanceInterfaceBgpStatus,
                                               cmd, args, clientName='BGP' )
   except EmptyResponseException():
      return None

def Plugin( entityManager ):
   global vrfNbrToGroupDir
   global bgpGroupStatusDir
   global maintEnterInstanceLog, maintExitInstanceLog
   global unitStatusDir
   global mapConfig
   global bgpProfileStatusDir
   global bgpDefaultProfileDir
   global defaultInitiatorRmDir
   global defaultReceiverRmDir
   global groupToUnitDir
   global maintenanceGroupDir
   global allIntfStatusDir
   global ipStatus

   vrfNbrToGroupDir = LazyMount.mount( entityManager,
         'maintenance/mapping/member/bgp',
         'BgpMaintenance::VrfBgpNeighborToGroupDir', 'r' )
   bgpGroupStatusDir = LazyMount.mount( entityManager,
                                        'group/status/bgp',
                                        'BgpGroup::StatusDir', 'r' )
   unitStatusDir = LazyMount.mount( entityManager,
                                    'maintenance/unit/status',
                                    'Maintenance::Unit::StatusDir', 'r' )
   bgpProfileStatusDir = LazyMount.mount( entityManager,
                                          'maintenance/profile/status/bgp',
                                          'BgpMaintenanceProfile::StatusDir', 'r' )
   bgpDefaultProfileDir = LazyMount.mount( entityManager,
      'maintenance/profile/config/default/bgp',
      'Maintenance::Profile::DefaultProfile', 'r' )
   defaultInitiatorRmDir = LazyMount.mount(
      entityManager, 'maintenance/profile/config/default/initiatorRM/',
      'Routing::RouteMap::Config', 'r' )
   maintEnterLogPath = Cell.path( 'stage/' + maintEnterStageClass + '/log' )
   maintExitLogPath = Cell.path( 'stage/' + maintExitStageClass + '/log' )
   maintEnterInstanceLog = LazyMount.mount( entityManager, maintEnterLogPath,
                                            'Stage::Log', 'r' )
   maintExitInstanceLog = LazyMount.mount( entityManager, maintExitLogPath,
                                           'Stage::Log', 'r' )
   mapConfig = LazyMount.mount(
      entityManager, 'routing/routemap/config', 'Routing::RouteMap::Config', 'r' )
   defaultReceiverRmDir = LazyMount.mount(
      entityManager, 'maintenance/profile/config/default/receiverRM/',
      'Routing::RouteMap::Config', 'r' )
   groupToUnitDir = LazyMount.mount( 
      entityManager, 'maintenance/mapping/group',
      'Maintenance::GroupToUnitDir', 'r' )
   maintenanceGroupDir = LazyMount.mount(
      entityManager, 'maintenance/group/config',
      'Maintenance::MaintenanceGroup::ConfigDir', 'r' )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
