# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import Ip4Address, IpGenericAddress
from CliModel import (
   Bool,
   DeferredModel,
   Dict,
   Enum,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from CliPlugin.BgpCliModels import (
   BgpRoutePeerEntry,
   originValues,
   PmsiTunnelId,
)
import TableOutput
from TypeFuture import TacLazyType

PmsiTunnelType = TacLazyType( "Routing::Multicast::PmsiTunnelType" )

class BgpVpnRouteTypeEntry( DeferredModel ):
   stale = Bool( default=False, help="Route is stale" )
   valid = Bool( default=False, help="Route is valid" )
   suppressed = Bool( default=False,
                      help="Route is suppressed from entering the RIB" )
   active = Bool( default=False, help="Route is active" )
   backup = Bool( default=False, help="Route is backup" )
   ecmpHead = Bool( default=False, help="Route is the ECMP head" )
   ecmp = Bool( default=False, help="Route is an ECMP route" )
   ucmp = Bool( optional=True, default=False, help="Route is an UCMP route" )
   ecmpContributor = Bool( default=False, help="Route contributes to ECMP" )
   atomicAggregator = Bool( default=False, help="Route is an atomic-aggregate" )
   queued = Bool( default=False, help="Route is queued for advertisement" )
   localAgg = Bool( optional=True, help="Route is locally aggregated" )
   notInstalledReason = Enum( values=( "routeBestInactive", "routeInstallDenied",
                                       "labeledRoutePresent" ),
                              optional=True,
                              help="Reason for route not being installed" )
   origin = Enum( optional=True, values=originValues, help="Route origin" )
   waitForConvergence = Bool( optional=True,
                              help="Route is pending best path selection" )
   unknownMetric = Bool( optional=True,
                         help="Route is pending resolution" )

class CustomerAttributesDetailEntry( DeferredModel ):
   originAsn = Int( help="Origin AS number" )
   origin = Enum( values=originValues, help='Route origin' )
   aggregator = Str( optional=True, help='Aggregator of the route' )
   originator = Ip4Address( optional=True,
                            help='Router ID of the originator of this route' )
   clusterList = Str( optional=True, help='Cluster list for the route' )
   domainPath = List( optional=True, valueType=str, help="Domain path attribute" )
   med = Int( optional=True, help='Multi Exit Discriminator for the route' )
   localPreference = Int( optional=True, help='I-BGP Local preference indicator' )
   communities = List( valueType=str, help='Standard community attributes' )
   extCommunities = List( valueType=str, help='Extended community attributes' )
   extCommunityListRaw = List( valueType=int,
                               help='Raw extended community attributes' )
   largeCommunities = List( valueType=str, help='Large community attributes' )

class BgpVpnRouteDetailEntry( DeferredModel ):
   communities = List( valueType=str, help="List of communities for this route" )
   extCommunities = List(
      valueType=str, help="List of extended communities for this route" )
   extCommunityListRaw = List(
      valueType=int, help="List of raw extended communities for this route" )
   largeCommunities = List(
      valueType=str, help="List of large communities for route" )
   rxSafi = Str( optional=True, help="Received SAFI of this route" )
   origin = Enum( values=originValues, help="Route origin" )
   originator = Ip4Address( optional=True,
                            help="Router ID of the originator of this route" )
   clusterList = Str( optional=True, help="Cluster list for the route" )
   peerEntry = Submodel( valueType=BgpRoutePeerEntry, optional=True,
                         help="Peer information for the route" )
   fromRRClient = Bool( default=False,
                        help="Route received from route reflector client" )
   tunnelRibEligible = Bool( default=False,
                             help="Route eligible to be installed in Tunnel RIB" )
   customerAttributeSet = Submodel( optional=True,
      valueType=CustomerAttributesDetailEntry,
      help="Customer Attributes" )

class EncapTunnelModel( Model ):
   vlans = Dict( optional=True, keyType=int, valueType=PmsiTunnelId,
                 help="Dictionary of PMSI tunnel identifiers indexed by VLAN ID" )
   pmsiTunnel = Submodel( optional=True, valueType=PmsiTunnelId,
                          help="PMSI tunnel identifier information" )

class EncapSourceModel( Model ):
   sources = Dict( keyType=IpGenericAddress, valueType=EncapTunnelModel,
                  help="PMSI tunnels indexed by multicast source address" )

class EncapGroupModel( Model ):
   groups = Dict( keyType=IpGenericAddress, valueType=EncapSourceModel,
                  help="Dictionary of multicast sources indexed by mulitcast group "
                       "address" )

class EncapVrfModel( Model ):
   vrfId = Int( optional=True, help="VRF ID" )
   tunnelTypes = Dict( keyType=str, valueType=EncapGroupModel,
                       help="Dictionary of multicast groups indexed by mulitcast "
                            "tunnel type" )

class MvpnEncapStatusModel( Model ):
   vrfs = Dict( keyType=str, valueType=EncapVrfModel,
                help="Dictionary of encapsulation status entries indexed by VRF "
                     "name" )

   def getPmsiTunnelTextOutput( self, tunnelType, tunnelId ):
      pmsiTunnelText = ""
      if tunnelType == PmsiTunnelType.mLdpP2mpLsp:
         pmsiTunnelText = ( f"Root Address: {tunnelId.mldpP2mp.rootAddress} "
                            f"{tunnelId.mldpP2mp.opaqueValue}" )
      elif tunnelType == PmsiTunnelType.rsvpTeP2mpLsp:
         pmsiTunnelText = ( f"P2MP ID: {tunnelId.rsvpP2mp.p2mpId} "
                            f"Tunnel ID: {tunnelId.rsvpP2mp.tunnelId} "
                            f"Extended Tunnel ID: {tunnelId.rsvpP2mp.extTunnelId}" )
      return pmsiTunnelText

   def render( self ):
      for vrfName, encapVrfModel in self.vrfs.items():
         for tunnelType, encapGroupModel in encapVrfModel.tunnelTypes.items():
            print( f"VRF: {vrfName}, VRF ID: {encapVrfModel.vrfId}, "
                   f"PMSI tunnel type: {tunnelType}" )
            headers = ( 'Group', 'Source', 'PMSI Tunnel' )
            table = TableOutput.createTable( headers )
            formatLeft = TableOutput.Format( justify="left" )
            formatLeft.padLimitIs( True )
            table.formatColumns( formatLeft, formatLeft, formatLeft )
            for group in sorted( encapGroupModel.groups ):
               encapSourceModel = encapGroupModel.groups[ group ]
               for source in sorted( encapSourceModel.sources ):
                  encapTunnelModel = encapSourceModel.sources[ source ]
                  pmsiTunnelText = self.getPmsiTunnelTextOutput(
                                      tunnelType, encapTunnelModel.pmsiTunnel )
                  table.newRow( group, source, pmsiTunnelText )
            print( table.output() )
