# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Toggles.BmpToggleLib
from ArnetModel import IpGenericAddress
from CliModel import Model
from CliModel import Dict
from CliModel import Float
from CliModel import Int
from CliModel import Str
from CliModel import Enum
from CliModel import Bool
from CliModel import Submodel
from HumanReadable import formatTimeInterval

stationStates = ( 'idle', 'connecting', 'up' )
stationStatesHelp = 'The state of the station: \n' \
                    'idle - Station is stopped due to invalid config, \n' \
                    'connecting - Station is awaiting for the connection ' \
                    'or attempting to connect, \n' \
                    'up - Station connection is established'
timestampModes = ( 'none', 'sendTime' )
timestampModesHelp = '''BMP Timestamp setting:
      none - No timestamp set in BGP monitoring protocol message
      sendTime - Time when BGP monitoring protocol message is sent'''

adjRibOutExportToggle = Toggles.BmpToggleLib.toggleBmpAdjRibOutExportEnabled()

class StationHostnameOrAddress( Model ):
   ip = IpGenericAddress( optional=True, help='The IP address of the station' )
   hostName = Str( optional=True, help='The hostname of the station' )
   def modelToString( self ):
      if self.ip:
         return str( self.ip )
      if self.hostName:
         return self.hostName
      return None

class BgpAdjRibInExportStatus( Model ):
   monitoringEnabled = Bool( help='BGP Monitoring Enabled' )
   def render( self ):
      monitoringEnabledString = None
      if self.monitoringEnabled: 
         monitoringEnabledString = 'enabled'
      else:
         monitoringEnabledString = 'disabled'
      print( 'BGP Monitoring status: ' + monitoringEnabledString )

class BmpGlobalConfig( Model ):
   timestampMode = Enum( values=timestampModes, help=timestampModesHelp )
   prePolicyExport = Bool( help='Export received pre-policy routes' )
   postPolicyExport = Bool( help='Export received post-policy routes' )
   if adjRibOutExportToggle:
      ribOutExport = Bool( help='Export advertised post-policy routes' )
   afiSafiExport = Dict( keyType=str, valueType=bool,
                         help=( 'Address family export enabled flag, '
                                'keyed by address family name string' ) )
   exportSixPe = Bool( help='Export 6PE paths' )
   exportIpv6LuTunnel = Bool( help='Export IPv6 LU tunnel paths' )

   def render( self ):
      policies = []
      if self.prePolicyExport:
         policies.append( 'pre-policy' )
      if self.postPolicyExport:
         policies.append( 'post-policy' )

      policies = ', '.join( policies ) or 'none'
      print( 'BGP received route export policies:', policies )
      if adjRibOutExportToggle:
         print( 'BGP advertised route export policies:', \
            'post-policy' if self.ribOutExport else 'none' )

      timestampModeStr = 'send-time' if self.timestampMode == 'sendTime' else 'none'
      print( 'BMP Timestamp mode:', timestampModeStr )

      exports = ', '.join(
         sorted( k for k, v in self.afiSafiExport.items() if v ) ) or 'none'
      print( 'Address Families exported:', exports )
      sixPeExports = []
      if self.exportSixPe:
         sixPeExports.append( '6PE' )
      if self.exportIpv6LuTunnel:
         sixPeExports.append( 'IPv6 LU Tunnel' )
      if sixPeExports:
         print( 'IPv6 Labeled Unicast exported Path Types:',
                ', '.join( sixPeExports ) )

class BmpStationPolicies( Model ):
   prePolicyExport = Bool( help='This station exports received pre-policy routes' )
   postPolicyExport = Bool( help='This station exports received post-policy routes' )
   if adjRibOutExportToggle:
      ribOutExport = Bool( help='This station export advertised post-policy routes' )
   def render( self ):
      policies = []
      if self.prePolicyExport:
         policies.append( 'pre-policy' )
      if self.postPolicyExport:
         policies.append( 'post-policy' )

      policies = ', '.join( policies ) or 'none'
      print( '  Station received route export policies:', policies )
      if adjRibOutExportToggle:
         print( '  Station advertised route export policies:',
                'post-policy' if self.ribOutExport else 'none' )


#-------------------------------------------------------------------------------
# 'show bgp monitoring passive summary'
#-------------------------------------------------------------------------------
listenerSummaryFormat = '{0:16s} {1:>4s} {2:>5s} {3:>8s} {4:s}'

class BmpPassiveSummary( Model ):
   port = Int( help='Port number that is listening for connections' )
   stationCount = Int( help='Number of stations configured for this listener' )
   listening = Bool( help='Listener is ready to accept connections' )

   def printModel( self, vrfName, addrFamily ):
      print( listenerSummaryFormat.format( vrfName, addrFamily,
                                          str( self.port ),
                                          str( self.stationCount ),
                                          str( self.listening ) ) )

class AllBmpPassiveSummary( Model ):
   __revision__ = 2
   bgpAdjRibInExportStatus = Submodel( valueType=BgpAdjRibInExportStatus,
                              help='Status of Rib Export for BMP' )
   listenersV4 = Dict( keyType=str, valueType=BmpPassiveSummary,
                       help='Collection of IPv4 BMP listeners indexed by VRF name' )
   listenersV6 = Dict( keyType=str, valueType=BmpPassiveSummary,
                       help='Collection of IPv6 BMP listeners indexed by VRF name' )
   def render( self ):
      self.bgpAdjRibInExportStatus.render()
      print()
      if self.listenersV4 or self.listenersV6:
         print( listenerSummaryFormat.format( 'VRF Name', 'AF', 'Port', 'Stations',
                                             'Listening' ) )
         print( listenerSummaryFormat.format( '----------------', '----', '-----',
                                             '--------', '----------------' ) )

         for vrf in sorted( self.listenersV4 ):
            self.listenersV4[ vrf ].printModel( vrf, 'IPv4' )

         for vrf in sorted( self.listenersV6 ):
            self.listenersV6[ vrf ].printModel( vrf, 'IPv6' )
      else:
         print( 'No monitoring stations are configured' )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # `listeners` was V4 Cli attr only
         # drop `listenersV6` and move `listenersV4` to `listeners`
         dictRepr[ 'listeners' ] = dictRepr[ 'listenersV4' ]
         del dictRepr[ 'listenersV4' ]
         del dictRepr[ 'listenersV6' ]
      return dictRepr

#-------------------------------------------------------------------------------
# 'show bgp monitoring passive [ vrf <vrfName> ]'
#-------------------------------------------------------------------------------

class BmpPassive( Model ):
   port = Int( help='Port number that is listening for connections' )
   listening = Bool( help='The state of the listener for when the listener is '
                          'ready to accept connections' )
   reason = Str( optional=True, help='The reason why listening is False' )
   stations = Dict( keyType=str, valueType=StationHostnameOrAddress,
                    help='The collection of stations and their hostname this '
                         'listener will accept when a new connection is made '
                         'indexed by the station\'s name' )
   connectionsAccepted = Int( help='Number of connections accepted' )
   connectionsRejected = Int( help='Number of connections rejected' )

   def printModel( self, vrfName, addrFamily ):
      print( 'Passive Listener VRF -', vrfName )
      print( '  Address Family:', addrFamily )
      print( '  Port:', self.port )
      print( '  Status:', 'Listening' if self.listening else 'Disabled' )
      if self.reason:
         print( '    Reason:', self.reason )
      if self.stations:
         print( '  Stations:' )
         for station in sorted( self.stations ):
            host = self.stations[ station ].modelToString()
            print( f'    {station}: ({host})' )
      print( '  Accepted:', self.connectionsAccepted )
      print( '  Rejected:', self.connectionsRejected )

class AllBmpPassive( Model ):
   __revision__ = 2
   bgpAdjRibInExportStatus = Submodel( valueType=BgpAdjRibInExportStatus,
                              help='Status of Rib Export for BMP' )
   listenersV4 = Dict( keyType=str, valueType=BmpPassive,
                       help='Collection of IPv4 BMP listeners indexed by VRF name' )
   listenersV6 = Dict( keyType=str, valueType=BmpPassive,
                       help='Collection of IPv6 BMP listeners indexed by VRF name' )
   def render( self ):
      self.bgpAdjRibInExportStatus.render()
      print()
      if self.listenersV4 or self.listenersV6:
         for vrf in sorted( self.listenersV4 ):
            self.listenersV4[ vrf ].printModel( vrf, 'IPv4' )
         for vrf in sorted( self.listenersV6 ):
            self.listenersV6[ vrf ].printModel( vrf, 'IPv6' )
      else:
         print( 'No monitoring stations are configured' )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # `listeners` was V4 Cli attr only
         # drop `listenersV6` and move `listenersV4` to `listeners`
         dictRepr[ 'listeners' ] = dictRepr[ 'listenersV4' ]
         del dictRepr[ 'listenersV4' ]
         del dictRepr[ 'listenersV6' ]
      return dictRepr

#-------------------------------------------------------------------------------
# 'show bgp monitoring active summary'
#-------------------------------------------------------------------------------

TCP_SOCKET_STATE_LIST  = ( 'NOTUSED', 'ESTABLISHED', 'SYN-SENT',
                         'SYN-RECEIVED', 'FIN-WAIT-1', 'FIN-WAIT-2',
                         'TIME-WAIT', 'CLOSED', 'CLOSE-WAIT',
                         'LAST-ACK', 'LISTEN', 'CLOSING' )

TCPI_OPT_TIMESTAMPS  = 1
TCPI_OPT_SACK        = 2
TCPI_OPT_WSCALE      = 4
TCPI_OPT_ECN         = 8 
TCPI_OPT_ECN_SEEN    = 16
TCPI_OPT_SYN_DATA    = 32

class TcpOptions( Model ):
   timestamps = Bool( default=False, help='Timestamps enabled' ) 
   selectiveAcks = Bool( default=False, help='Selective Acknowledgments enabled' )
   windowScale = Bool( default=False, help='Window Scale enabled' )
   ecn = Bool( default=False, help='Explicit Congestion Notification enabled' )

class BmpTcpStatistics( Model ):

   state = Enum( values=TCP_SOCKET_STATE_LIST, help='TCP Socket current state' )
   options = Submodel( valueType=TcpOptions, help='TCP socket options' )
   sendWindowScale = Int( help='TCP socket send window scale factor' )
   recvWindowScale = Int( help='TCP socket receive window scale factor' )
   retransmitTimeout = Int( help='TCP socket retransmission timeout ' \
                                 '(micro seconds)' )
   delayedAckTimeout = Int( help='TCP socket delayed ack timeout (micro secions)' )
   maxSegmentSize = Int( help='TCP outgoing maximum segment size (bytes)' )
   sendRtt = Int( help='TCP round-trip time (micro seconds)' )
   sendRttVariance = Int( help='TCP round-trip time variance (micro seconds)' )
   slowStartThreshold = Int( help='TCP send slow start size threshold (bytes)' )
   congestionWindow = Int( help='TCP send congestion window (bytes)' )
   recvRtt = Int( help='TCP receive round-trip time (micro seconds)' )
   recvWindow = Int( help='TCP advertised receive window (bytes)' )
   totalRetrans = Int( help='Total number of TCP retransmissions' )
   outputQueueLength = Int( help='TCP output queue length' )
   outputQueueMaxLength = Int( help='TCP output queue max length' )

   def setAttrFromDict( self, data ):

      self.options = TcpOptions()
      if 'options' in data:
         optionFlag = data[ 'options' ]
         if optionFlag & TCPI_OPT_TIMESTAMPS:
            self.options.timestamps = True
         if optionFlag & TCPI_OPT_SACK:
            self.options.selectiveAcks = True
         if optionFlag & TCPI_OPT_WSCALE:
            self.options.windowScale = True
         if optionFlag & TCPI_OPT_ECN:
            self.options.ecn = True

         del data[ 'options' ]

      if 'state' in data:
         data[ 'state' ] = TCP_SOCKET_STATE_LIST[ data[ 'state' ] ]

      for key in data:
         setattr( self, key, data[key] )

   def render( self ):
      # pylint: disable=consider-using-f-string
      if self.state is None:
         return

      outputQueueLength = self.outputQueueLength \
                if self.outputQueueLength is not None else "not available"
      outputQueueMaxLength = self.outputQueueMaxLength \
                if self.outputQueueMaxLength is not None else "not available"

      print( "  TCP Socket Information:" )
      print( "    TCP state is %s" % ( self.state ) )
      print( f"    Send-Q: {outputQueueLength}/{outputQueueMaxLength}" )
      print( "    Outgoing Maximum Segment Size (MSS): %s" %
             ( self.maxSegmentSize ) )
      print( "    Total Number of TCP retransmissions: %s" % ( self.totalRetrans ) )

      print( "    Options:" )
      print( "      Timestamps enabled: %s" %
            ( "yes" if self.options.timestamps
              else "no" ) )
      print( "      Selective Acknowledgments enabled: %s" %
            ( "yes" if self.options.selectiveAcks
              else "no" ) )
      print( "      Window Scale enabled: %s" %
            ( "yes" if self.options.windowScale
              else "no" ) )
      print( "      Explicit Congestion Notification (ECN) enabled: %s" %
            ( "yes" if self.options.ecn
              else "no" ) )
      print( "    Socket Statistics:" )
      print( "      Window Scale (wscale): %s,%s" %
             ( self.sendWindowScale, self.recvWindowScale ) )
      print( "      Retransmission Timeout (rto): %.1fms" %
             ( float( self.retransmitTimeout )/1000 ) )
      print( "      Round-trip Time (rtt/rtvar): %.1fms/%.1fms" %
             ( float( self.sendRtt )/1000, float( self.sendRttVariance )/1000 ) )
      print( "      Delayed Ack Timeout (ato): %.1fms" %
             ( float( self.delayedAckTimeout )/1000 ) )
      print( "      Congestion Window (cwnd): %s" % ( self.congestionWindow ) )

      if self.slowStartThreshold < 65535:
         print( "      Slow-start Threshold (ssthresh): %s" %
                ( self.slowStartThreshold ) )
      if self.sendRtt > 0 and self.maxSegmentSize and self.congestionWindow:
         print( "      TCP Throughput: %.2f Mbps" %
               ( float( self.congestionWindow ) *
                 ( float( self.maxSegmentSize ) * 8. / float( self.sendRtt ) ) ) )

      if self.recvRtt:
         print( "      Recv Round-trip Time (rcv_rtt): %.1fms" %
                ( float( self.recvRtt )/1000 ) )
      if self.recvWindow:
         print( "      Advertised Recv Window (rcv_space): %s" %
                ( self.recvWindow ) )

activeSummaryFormat = '{0:16s} {1:>12s} {2:>10s}'

class BmpActiveSummary( Model ):
   connected = Bool( help='The state of the connection to the station' )
   retryTime = Float( optional=True,
                      help='The timestamp in UTC of when the router will initiate '
                           'a connection to the station.' )
   def printModel( self, station ):
      if self.retryTime:
         # This is the time in the future
         time = self.retryTime - Tac.utcNow()
         timeStr = formatTimeInterval( time )
      else:
         timeStr = '-'
      status = 'connected' if self.connected else 'disconnected'
      print( activeSummaryFormat.format( station, status, timeStr ) )

class AllBmpActiveSummary( Model ):
   bgpAdjRibInExportStatus = Submodel( valueType=BgpAdjRibInExportStatus,
                              help='Status of Rib Export for BMP' )
   stations = Dict( keyType=str, valueType=BmpActiveSummary,
                    help='The collection of stations that are using active mode.' )
   def render( self ):
      self.bgpAdjRibInExportStatus.render()
      print()
      if self.stations:
         print( activeSummaryFormat.format( 'Station Name',
      'Status', 'Retry Time' ) )
         print( activeSummaryFormat.format( '----------------', '------------',
                                           '----------' ) )
         for station in sorted( self.stations ):
            self.stations[ station ].printModel( station )
      else:
         print( 'No monitoring stations are configured' )

#-------------------------------------------------------------------------------
# 'show bgp monitoring active [ station <station> ]'
#-------------------------------------------------------------------------------

class BmpActive( Model ):
   vrfName = Str( help='The VRF the station will be connected in' )
   address = Submodel( valueType=StationHostnameOrAddress,
                       help='The hostname or address of the station' )
   remotePort = Int( help='The port of the station used for active connection mode' )
   retryInterval = Int( help='The interval in seconds which the connection will be '
                             'retried if an attempt fails' )
   connected = Bool( help='The state of the connection to the station' )
   reason = Str( optional=True, help='The reason why the connection is False' )
   retryTime = Float( optional=True,
                      help='The timestamp in UTC of when the router will initiate '
                           'a connection to the station.' )
   connectionAttempts = Int( help='The number of times the router has '
                                  'attempted to connect to the station' )
   connectionSuccesses = Int( help='The number of times the router has '
                                   'successfully connected to the station' )
   connectionErrors = Int( help='The number of times the router has '
                                'failed to connect to the station' )
   def printModel( self, station ):
      # pylint: disable=consider-using-f-string
      print( 'Active Station -', station )
      print( '  VRF:', self.vrfName )
      address = self.address.modelToString()
      if address:
         print( '  Address:', address )
      print( '  Port:', self.remotePort )
      print( '  Retry interval: %ss' % self.retryInterval )
      print( '  Connection:', 'connected' if self.connected else 'disconnected' )
      if self.reason:
         print( '    Reason:', self.reason )
      if self.retryTime:
         time = self.retryTime - Tac.utcNow()
         print( '    Time until retry: %ss' % int( time ) )
      print( '  Attempts:', self.connectionAttempts )
      print( '  Successes:', self.connectionSuccesses )
      print( '  Errors:', self.connectionErrors )


class AllBmpActive( Model ):
   bgpAdjRibInExportStatus = Submodel( valueType=BgpAdjRibInExportStatus,
                              help='Status of Rib Export for BMP' )
   stations = Dict( keyType=str, valueType=BmpActive,
                    help='The collection of stations that are using active mode.' )
   def render( self ):
      self.bgpAdjRibInExportStatus.render()
      print()
      if self.stations:
         for station in sorted( self.stations ):
            self.stations[ station ].printModel( station )
      else:
         print( 'No monitoring stations are configured' )

#-------------------------------------------------------------------------------
# 'show bgp monitoring summary'
#-------------------------------------------------------------------------------

stationSummaryFormat = '{0:16s} {1:>12s} {2:>12s}'

class BmpStationSummary( Model ):
   state = Enum( values=stationStates, help=stationStatesHelp )
   upTime = Float( optional=True,
                   help='The timestamp in UTC when station transitioned to up' )
   def printModel( self, name ):
      if self.upTime:
         # This is the time in the future
         time = Tac.utcNow() - self.upTime
         timeStr = formatTimeInterval( time )
      else:
         timeStr = '-'
      print( stationSummaryFormat.format( name, self.state, timeStr ) )

class AllBmpStationSummary( Model ):
   bgpAdjRibInExportStatus = Submodel( valueType=BgpAdjRibInExportStatus,
                              help='Status of Rib Export for BMP' )
   bmpGlobalConfig = Submodel( valueType=BmpGlobalConfig,
                               help='Bmp export policy and timestamp configuration' )
   stations = Dict( keyType=str, valueType=BmpStationSummary,
                    help='Collection of station information indexed by the '
                         'station\'s name' )
   def render( self ):
      self.bgpAdjRibInExportStatus.render()
      self.bmpGlobalConfig.render()
      print()
      if self.stations:
         print( stationSummaryFormat.format( 'Station Name', 'Status', 'Uptime' ) )
         print( stationSummaryFormat.format( '----------------', '------------',
                                            '------------' ) )
         for name in sorted( self.stations ):
            self.stations[ name ].printModel( name )
      else:
         print( "No monitoring stations are configured" )

#-------------------------------------------------------------------------------
# 'show bgp monitoring [ station <station> ]'
#-------------------------------------------------------------------------------

class BmpStation( Model ):
   description = Str( optional=True, help='Station description' )
   connectionMode = Enum( values=( 'unknown', 'passive', 'active' ),
                          help='The mode used to connect to the station: \n'
                               'unknown - No mode is configured, \n'
                               'passive - Router listens for station to connect, \n'
                               'active - Router initiates the connection to '
                               'the station' )
   vrfName = Str( help='The VRF the station will be connected in' )
   address = Submodel( valueType=StationHostnameOrAddress, optional=True,
                       help='The hostname or address of the station' )
   remotePort = Int( optional=True,
                     help='The port of the station used for active connection mode' )
   connected = Bool( help='The state of the connection to the station' )
   state = Enum( values=stationStates, help=stationStatesHelp )
   upTime = Float( optional=True,
                   help='The timestamp in UTC when station transitioned to up' )
   flapCount = Int( help='Number station disconnects from the router' )
   tcpSocketStatistics = Submodel( valueType=BmpTcpStatistics, optional=True,
                    help='Peer TCP information' )
   exportPolicyStationConfig = \
         Submodel( valueType=BmpStationPolicies,
                   optional=True,
                   help='BMP station export policy configuration' )

   def printModel( self, name ):
      print( 'Station -', name )
      if self.description:
         print( '  Description:', self.description )
      if self.exportPolicyStationConfig:
         self.exportPolicyStationConfig.render()
      print( '  Connection mode:', self.connectionMode )
      print( '  VRF:', self.vrfName )

      if self.address:
         addressStr = self.address.modelToString()
         if addressStr:
            print( '  Address:', addressStr )
      if self.remotePort:
         print( '  Port:', self.remotePort )
      print( '  Connection:', 'connected' if self.connected else 'disconnected' )
      print( '  State:', self.state )
      if self.upTime:
         time = Tac.utcNow() - self.upTime
         timeStr = formatTimeInterval( time )
         print( '  Uptime:', timeStr )
      print( '  Station flap count:', self.flapCount )

      if self.tcpSocketStatistics is not None:
         self.tcpSocketStatistics.render()

class AllBmpStation( Model ):
   bgpAdjRibInExportStatus = Submodel( valueType=BgpAdjRibInExportStatus,
                              help='Status of Rib Export for BMP' )
   bmpGlobalConfig = Submodel( valueType=BmpGlobalConfig,
                               help='Bmp export policy and timestamp configuration' )
   stations = Dict( keyType=str, valueType=BmpStation,
                    help='Collection of station information indexed by the '
                         'station\'s name' )
   def render( self ):
      self.bgpAdjRibInExportStatus.render()
      self.bmpGlobalConfig.render()
      print()
      if self.stations:
         for name in sorted( self.stations ):
            self.stations[ name ].printModel( name )
      else:
         print( 'No monitoring stations are configured' )

class EorStatistic( Model ):
   __public__ = False
   peerAddr = IpGenericAddress( help='Peer address' )
   pendingCount = Int( help='Pending EOR(End-of-Rib)s for this peer' )
   initialPeer = Bool( help='Peer was established before station startup' )

class BmpStationConvergence( Model ):
   __public__ = False
   converged = Bool( help='The station has converged' )
   lastConvergedTimestamp = Float( optional=True,
      help='Timestamp when the station converged' )
   convergenceTime = Float( optional=True,
      help='Seconds taken for the BMP station to converge' )
   eorStatistics = Dict( keyType=IpGenericAddress, valueType=EorStatistic,
                         help='EOR(End-of-Rib) Statistics per Peer' )

   def render( self, name ): #pylint:disable=arguments-differ
      # pylint: disable=consider-using-f-string
      print( 'Station -', name )
      convergenceStr = "Pending"
      if self.converged:
         convergenceStr = "Converged"
      print( "   Convergence Status : %s" % convergenceStr )
      if self.lastConvergedTimestamp:
         time = Tac.utcNow() - self.lastConvergedTimestamp
         timeStr = formatTimeInterval( time ) + 's ago'
      else:
         timeStr = '-'
      print( "   Last converged : %s" % timeStr )

      if self.convergenceTime:
         timeStr = formatTimeInterval( self.convergenceTime ) + 's'
      else:
         timeStr = '-'
      print( "   Convergence Time : %s" % timeStr )

      trackedEorStr = ""
      untrackedEorStr = ""
      for peerAddr, eorStats in sorted( self.eorStatistics.items() ):
         if eorStats.pendingCount:
            if eorStats.initialPeer:
               trackedEorStr += "      %15s:%3d\n" % ( peerAddr,
                                                      eorStats.pendingCount )
            else:
               untrackedEorStr += "      %15s:%3d\n" % ( peerAddr,
                                                      eorStats.pendingCount )

      if trackedEorStr:
         print( "   Pending EORs per Peers tracked for convergence:" )
         print( trackedEorStr, end=' ' )

      if untrackedEorStr:
         print( "   Pending EORs per Peers not tracked for convergence:" )
         print( untrackedEorStr, end=' ' )

      print()

class AllBmpConvergence( Model ):
   __public__ = False
   stations = Dict( keyType=str, valueType=BmpStationConvergence,
                    help='Collection of station convergence information '
                         'indexed by the station\'s name' )

   def render( self ):
      if self.stations:
         for name in sorted( self.stations ):
            self.stations[ name ].render( name )
      else:
         print( 'No monitoring stations are configured' )
