# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os

import Arnet
import Cell
import CliCommand
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.BridgingCli import ( bridgingCheckStaticMacHook,
                                    bridgingAddStaticMacHook,
                                    bridgingDelStaticMacHook,
                                    warnMacTableUnsupportedUFTModeHook )
from CliPlugin.BridgingCli import switchIntfConfigIfEnabled
from CliPlugin import IntfCli
import CliGlobal
import ConfigMount
import EbraLib
import EbraLogMsgs
import Ethernet
import LazyMount
import Logging
import SmashLazyMount
import Tac
import UtmpDump

BridgingCliModel = CliDynamicPlugin( "BridgingCliModel" )

LabelType = Tac.Type( 'Arnet::MplsLabel' )
SwitchForwardingMode = Tac.Type( "Bridging::SwitchForwardingMode" )
SwitchportMode = Tac.Type( "Bridging::SwitchportMode" )

gv = CliGlobal.CliGlobal(
        bridgingConfig=None,
        bridgingFdbConfig=None,
        bridgingHwCapabilities=None,
        bridgingInCliConfig=None,
        bridgingStatus=None,
        bridgingSwitchIntfConfig=None,
        dynAllowedVlanDir=None,
        dynBlockedVlanDir=None,
        flushConfig=None,
        flushReply=None,
        flushReplySm=None,
        redundancyStatus=None,
        vlanTagFormatDropStatusDir=None )

# aging-time handlers
defaultAgingTime = 300

def setAgingTime( mode, args ):
   gv.bridgingInCliConfig.hostAgingTime = args.get( 'AGING_TIME', 0 )
   # Hook is populated on Trident4 platforms to warn of conflicting configuration
   warnMacTableUnsupportedUFTModeHook.notifyExtensions( mode )

def noAgingTime( mode, args ):
   gv.bridgingInCliConfig.hostAgingTime = defaultAgingTime

# "[no] mac address-table static" command handlers
def setStatic( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if intfNamesOrDrop := args.get( 'INTFS' ):
      # Syntax has "{ INTFS }" so we have an iterable of intf range matchers.
      intfNamesOrDrop = { intf
                          for intfRange in intfNamesOrDrop
                          for intf in intfRange }
   else:
      intfNamesOrDrop = args.get( 'drop' )

   forwardingEligible = 'forwarding' in args
   setStaticHelper( mode, args, intfNamesOrDrop, no=no,
                    forwardingEligible=forwardingEligible )
   # Hook is populated on Trident4 platforms to warn of conflicting configuration
   warnMacTableUnsupportedUFTModeHook.notifyExtensions( mode )

def setStaticHelper(
      mode, args, intfNamesOrDrop, label=LabelType.null, no=False,
      forwardingEligible=False ):
   macAddr = args[ 'MACADDR' ]
   vlanId = args[ 'VLAN_ID' ]

   intfNames = set()
   if intfNamesOrDrop is not None and intfNamesOrDrop != 'drop':
      intfNames = intfNamesOrDrop

   macAddr = Ethernet.convertMacAddrToCanonical( macAddr )

   if Ethernet.isBroadcast( macAddr ) \
      or Ethernet.isIPMulticast( macAddr, allowIanaReserved=True ) \
      or macAddr == '00:00:00:00:00:00':
      mode.addError( 'Only unicast or non-IP multicast '
                     'addresses can be configured statically.' )
      return
   elif Ethernet.isUnicast( macAddr ) and \
        intfNamesOrDrop is not None and \
        ( intfNamesOrDrop != 'drop' and len( intfNames ) > 1 ):
      mode.addError( 'Only one interface is allowed for unicast addresses.' )
      return
   elif Ethernet.isMulticast( macAddr ) and intfNamesOrDrop == 'drop':
      mode.addError( 'Cannot drop on a multicast address.' )
      return

   fdbConfig = gv.bridgingInCliConfig.fdbConfig.newMember( vlanId.id )

   def staticAddrNotFound( vlanId, macAddr ):
      # Ebra may not be aware of this mac address. So check
      # with the registered cli hooks, if anyone is aware.
      delStatus = False
      for hook in bridgingDelStaticMacHook.extensions():
         status = hook( macAddr, vlanId )
         delStatus |= status
      if not delStatus:
         # None of the hooks were successful
         mode.addWarning( 'MAC address could not be removed: Address not found' )

   def delStaticAddrAndNotify( vlanId, macAddr ):
      del table[ macAddr ]
      for hook in bridgingDelStaticMacHook.extensions():
         hook( macAddr, vlanId )

   # Unicast case
   # pylint: disable=too-many-nested-blocks
   if Ethernet.isUnicast( macAddr ):
      table = fdbConfig.configuredHost
      if no:
         a = table.get( macAddr )
         if not a:
            staticAddrNotFound( vlanId, macAddr )
            return
         else:
            if intfNamesOrDrop is None:
               delStaticAddrAndNotify( vlanId, macAddr )
            elif intfNamesOrDrop != 'drop' and a.intf != '':
               delStaticAddrAndNotify( vlanId, macAddr )
            elif intfNamesOrDrop == 'drop' and a.intf == '':
               delStaticAddrAndNotify( vlanId, macAddr )
            else:
               staticAddrNotFound( vlanId, macAddr )
            return

      # Address needs to be added to the collection of configuredHost
      dropMode = 'dropModeNone'
      if intfNamesOrDrop == 'drop':
         intf = ''
         if 'address' in args and 'destination' in args:
            dropMode = 'dropModeDst'
         else:
            dropMode = 'dropModeSrcAndDst'
      else:
         for intfIn in intfNames:
            intf = intfIn

      # Confirm from the hooks, if it is OK to add static mac entry. Don't
      # add if any of the hooks returns False.
      accept = True
      for hook in bridgingCheckStaticMacHook.extensions():
         if not hook( mode, macAddr, vlanId, intfNamesOrDrop ):
            accept = False

      if accept:
         table.addMember( Tac.Value(
            "Bridging::ConfiguredHost", address=macAddr, intf=intf,
            forwardingEligible=forwardingEligible, entryType='configuredStaticMac',
            label=label, dropMode=dropMode ) )
         # Notify add through the add CliHook
         for hook in bridgingAddStaticMacHook.extensions():
            hook( macAddr, vlanId, intfNamesOrDrop )

   # Multicast case, exceptions addressed above
   else:
      if mode.session_.guardsEnabled() \
         and not gv.bridgingHwCapabilities.staticMcastSupported:
         mode.addError( 'Adding a static multicast MAC address is not '
                        'supported on this platform' )
         return
      table = fdbConfig.ethGroup
      if no:
         if not macAddr in table:
            staticAddrNotFound( vlanId, macAddr )
            return
         else:
            a = table.get( macAddr )
            if intfNamesOrDrop is None:
               delStaticAddrAndNotify( vlanId, macAddr )
            else:
               for intf in intfNames:
                  intfList = a.intf
                  if intf in intfList:
                     del a.intf[ intf ]
                  else:
                     mode.addError( 'Interface ' + intf + ' could not be removed '
                                    'from entry ' + str( macAddr ) )
                     mode.addError( 'The interface is not configured '
                                    'for the address' )
               # If there are no more interfaces remaining, delete the multicast
               # table entry
               if len( a.intf ) == 0:
                  delStaticAddrAndNotify( vlanId, macAddr )
            return

      # Again confirm from the Cli hooks, if it is OK to add this static mac.
      proceed = True
      for hook in bridgingCheckStaticMacHook.extensions():
         if not hook( mode, macAddr, vlanId, intfNamesOrDrop ):
            proceed = False
      if not proceed:
         return

      # Address needs to be added to the collection ethGroup
      # 'drop' is not a valid option (enforced above)
      entry = table.get( macAddr )
      if not entry:
         entry = fdbConfig.newEthGroup( macAddr )
      for intf in intfNames:
         entry.intf[ intf ] = True

# clear mac address-table handler
def doMacAddressTableClear( mode, args ):
   key = f"{ os.getpid() }.{ Tac.now() }"
   value = Tac.Value( "Bridging::HostTableFlushRequest" )
   vlanLog = "all vlans"
   vlanId = args.get( 'VLANID' )
   if vlanId is not None:
      vlanLog = f"vlan { vlanId.id }"
      value.vlanId = vlanId.id
   intfLog = "all interfaces"
   intfName = None
   for intfField in [ 'ETHINTF', 'ETHSUBINTF', 'VXLANINTF' ]:
      intf = args.get( intfField )
      if intf:
         intfName = intf.name
         break
   if not intfName:
      intfName = args.get( 'INTF_NAME' )
   if intfName is not None:
      intfLog = f"interface { intfName }"
      value.intf = intfName
   addrLog = "all addresses"
   macAddr = args.get( 'MACADDR' )
   if macAddr is not None:
      addrLog = f"address { macAddr }"
      value.addr = macAddr
   gv.flushConfig.hostTableFlushRequest[ key ] = value

   info = UtmpDump.getUserInfo()
   Logging.log( EbraLogMsgs.ETH_MAC_ADDR_TABLE_CLEAR,
                intfLog, vlanLog, addrLog,
                info[ 'user' ], info[ 'tty' ], info[ 'ipAddr' ] )

# show dot1q-tunnel handlers
def doShowDot1QTunnel( mode, args ):
   ret = BridgingCliModel.Dot1QTunnel()
   intfs = IntfCli.Intf.getAll( mode, args.get( 'INTERFACE' ) )
   if intfs is None:
      return ret

   # Note that if no interface name is specified, we only show the interfaces that
   # are configured as switchports.  However, if an interface name is specified, we
   # show that interface regardless of whether it is configured as a switchport.
   def intfIsDot1QTunnel( i ):
      switchIntfConfig = switchIntfConfigIfEnabled( mode, i )
      return switchIntfConfig and switchIntfConfig.switchportMode == 'dot1qTunnel'
   intfs = [ x for x in intfs if intfIsDot1QTunnel( x ) ]

   for intfName in Arnet.sortIntf( i.name for i in intfs ):
      ret.interfaces.append( intfName )

   return ret

# switch forwarding-mode handlers
def setForwardingMode( mode, args ):
   if 'cut-through' in args:
      setCutThrough( mode, args )
   elif 'store-and-forward' in args:
      setStoreAndForward( mode, args )
   else:
      assert False, f'Unknown forwarding mode {args}'

def setCutThrough( mode, args ):
   if gv.bridgingHwCapabilities.supportedForwardingMode and \
          gv.bridgingHwCapabilities.supportedForwardingMode[ 0 ] == \
          SwitchForwardingMode.cutThrough:
      gv.bridgingInCliConfig.forwardingMode = SwitchForwardingMode.defaultMode
   else:
      gv.bridgingInCliConfig.forwardingMode = SwitchForwardingMode.cutThrough

def setStoreAndForward( mode, args ):
   if gv.bridgingHwCapabilities.supportedForwardingMode and \
          gv.bridgingHwCapabilities.supportedForwardingMode[ 0 ] == \
          SwitchForwardingMode.storeAndForward:
      gv.bridgingInCliConfig.forwardingMode = SwitchForwardingMode.defaultMode
   else:
      gv.bridgingInCliConfig.forwardingMode = SwitchForwardingMode.storeAndForward

# "switch forwarding-mode store-and-forward multicast' handlers
def setMcastStoreAndForward( mode, args ):
   gv.bridgingInCliConfig.l3McastForwardingMode = \
      SwitchForwardingMode.storeAndForward

# no forwarding-mode handler
def noForwardingMode( mode, args ):
   if 'store-and-forward' in args and 'multicast' in args:
      gv.bridgingInCliConfig.l3McastForwardingMode = SwitchForwardingMode.defaultMode
   else:
      gv.bridgingInCliConfig.forwardingMode = SwitchForwardingMode.defaultMode

# [no|default] switchport default mode handler
def setSwitchportDefaultMode( mode, args ):
   modeType = getattr( SwitchportMode, args.get( 'MODE_TYPE', 'access' ) )
   gv.bridgingInCliConfig.defaultSwitchportMode = modeType
   display = ( "Default Switchport mode changed. This shall impact all the "
               "interfaces that are in default switchport configuration state." )
   mode.addWarning( display )

# "[no|default] mac address-table reserved forward" command handlers
def shouldSetIeeeReservedMac( token ):
   # Check to make sure macAddrAll and individual entries are mutually exclusive
   ethAddrAll = Arnet.EthAddr( EbraLib.ieeeReservedConfigAll )
   if token == EbraLib.ieeeReservedTokenAll:
      if gv.bridgingInCliConfig.ieeeReservedForwarding:
         return False
   else:
      if ethAddrAll in gv.bridgingInCliConfig.ieeeReservedForwarding:
         return False
   return True

def shouldNoIeeeReservedMac( token ):
   # Check to make sure macAddrAll and individual entries are mutually exclusive
   ethAddrAll = Arnet.EthAddr( EbraLib.ieeeReservedConfigAll )
   if token == EbraLib.ieeeReservedTokenAll:
      if ethAddrAll not in gv.bridgingInCliConfig.ieeeReservedForwarding:
         return False
   else:
      if ethAddrAll in gv.bridgingInCliConfig.ieeeReservedForwarding:
         return False
   return True

def _ieeeReservedAddrForConfToken( mode, token ):
   if token == EbraLib.ieeeReservedTokenAll:
      macAddr = EbraLib.ieeeReservedConfigAll
   elif EbraLib.isIeeeReservedAddressToken( token ):
      macAddr = EbraLib.ieeeReservedAddressTokenToConfig( token )
   elif EbraLib.isIeeeReservedGroupToken( token ):
      macAddr = EbraLib.ieeeReservedGroupTokenToConfig( token )
   else:
      assert False, "Invalid reserved address token: " + repr( token )
   return Arnet.EthAddr( macAddr )

def setIeeeReservedMac( mode, token ):
   # For address ranges (except the special range 01-0f), the platform will
   # accept any address in the range.
   macAddr = _ieeeReservedAddrForConfToken( mode, token )
   gv.bridgingInCliConfig.ieeeReservedForwarding[ macAddr ] = True
   # Hook is populated on Trident4 platforms to warn of conflicting configuration
   warnMacTableUnsupportedUFTModeHook.notifyExtensions( mode )

# pylint: disable=inconsistent-return-statements
def noIeeeReservedMac( mode, token ):
   macAddr = _ieeeReservedAddrForConfToken( mode, token )
   if macAddr in gv.bridgingInCliConfig.ieeeReservedForwarding:
      del gv.bridgingInCliConfig.ieeeReservedForwarding[ macAddr ]

def setIeeeReservedMacCommon( mode, args ):
   mac = args.get( 'MAC_ADDR' )
   if mac is None or not shouldSetIeeeReservedMac( mac ):
      return
   return setIeeeReservedMac( mode, mac )

def noIeeeReservedMacCommon( mode, args ):
   mac = args.get( 'MAC_ADDR' )
   if mac is None or not shouldNoIeeeReservedMac( mac ):
      return
   return noIeeeReservedMac( mode, mac )

# "[no|default] mac pause-frame pass-through" command handlers
def setPauseFramePassThrough( mode, args ):
   gv.bridgingInCliConfig.pausePassThrough = True

def noPauseFramePassThrough( mode, args ):
   gv.bridgingInCliConfig.pausePassThrough = False

# "show mac pause-frame" command handlers
def doShowPauseFramePassThrough( mode, args ):
   ret = BridgingCliModel.PauseFramePassThrough()
   ret.pauseFramePassThrough = gv.bridgingConfig.pausePassThrough
   return ret

# mac address-table flushing interface vlan aggregation disabled handler
def disableVlanMacFlushAgg( mode, args ):
   gv.bridgingInCliConfig.vlanMacFlushAggregationDisabled = True
   # Hook is populated on Trident4 platforms to warn of conflicting configuration
   warnMacTableUnsupportedUFTModeHook.notifyExtensions( mode )

def noDisableVlanMacFlushAgg( mode, args ):
   gv.bridgingInCliConfig.vlanMacFlushAggregationDisabled = False

def initFlushReplySm():
   if gv.flushReplySm is None:
      gv.flushReplySm = Tac.newInstance( "Ebra::HostTableFlushReplySm",
                                         gv.flushConfig, gv.flushReply,
                                         gv.redundancyStatus )

def Plugin( entityManager ):

   mount = LazyMount.mount
   gv.bridgingConfig = mount( entityManager, "bridging/config",
                              "Bridging::Config", "r" )
   gv.bridgingFdbConfig = mount( entityManager, "bridging/fdbConfigDir",
                                 "Bridging::FdbConfigDir", "r" )
   gv.bridgingHwCapabilities = mount( entityManager, "bridging/hwcapabilities",
                                      "Bridging::HwCapabilities", "r" )
   gv.bridgingInCliConfig = ConfigMount.mount( entityManager,
                                     "bridging/input/config/cli",
                                     "Bridging::Input::CliConfig", "w" )
   gv.bridgingStatus = SmashLazyMount.mount( entityManager, "bridging/status",
                                             "Smash::Bridging::Status",
                                             SmashLazyMount.mountInfo( 'reader' ) )
   gv.bridgingSwitchIntfConfig = mount( entityManager, "bridging/switchIntfConfig",
                                        "Bridging::SwitchIntfConfigDir", "r" )
   gv.dynAllowedVlanDir = mount( entityManager, "bridging/input/dynvlan/allowedvlan",
                                 "Tac::Dir", "ri" )
   gv.dynBlockedVlanDir = mount( entityManager, "bridging/input/dynBlockedVlan",
                                 "Tac::Dir", "ri" )
   gv.vlanTagFormatDropStatusDir = mount( entityManager,
                                          "bridging/vlanTagFormatDrop/status",
                                          "Tac::Dir", "ri" )

   mg = entityManager.mountGroup()
   gv.flushConfig = mg.mount( "bridging/flush/request/cli",
                              "Bridging::HostTableFlushRequestDir", "w" )
   gv.flushReply = mg.mount( "bridging/flush/reply/all",
                             "Bridging::HostTableFlushReplyDir", "r" )
   gv.redundancyStatus = mg.mount( Cell.path( "redundancy/status" ),
                                   "Redundancy::RedundancyStatus", "r" )
   mg.close( callback=initFlushReplySm, blocking=True )
