# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Bool
from CliModel import Dict
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliPlugin import IntfCli
from IntfModels import Interface
import Tac

allConfiguredVlansStr = 'allConfiguredVlans'
allowedVlansOnlyStr = 'allowedVlansOnly'
vlanForwardingEnum = ( allowedVlansOnlyStr, allConfiguredVlansStr )

class MacSourceCheckSystemMac( Model ):
   sourceCheckCPUMac = Bool( help="Source check CPU Mac" )

   def render( self ):
      if self.sourceCheckCPUMac:
         print( 'Source check for system mac enabled' )
      else:
         print( 'Source check for system mac disabled' )

class MacSourceCheckInvalid( Model ):
   sourceCheckInvalid = Bool( help="Source check invalid Mac" )

   def render( self ):
      if self.sourceCheckInvalid:
         print( 'Source check for invalid mac enabled' )
      else:
         print( 'Source check for invalid mac disabled' )

class Dot1QTunnel( Model ):
   interfaces = List( valueType=Interface,
                      help="List of interfaces in dot1q-tunnel mode" )

   def render( self ):
      print( "dot1q-tunnel mode LAN Port (s)" )
      print( "------------------------------" )

      if not self.interfaces:
         print( "No ports have been configured as dot1q-tunnel" )
      else:
         for interface in self.interfaces:
            print( IntfCli.Intf.getShortname( interface ) )

class PauseFramePassThrough( Model ):
   pauseFramePassThrough = Bool( help="Pause frame visibility" )

   def render( self ):
      if self.pauseFramePassThrough:
         print( 'Pass through: Enabled' )
      else:
         print( 'Pass through: Disabled' )

class ConfigSource( Model ):
   source = Str( help="Name of the config source" )
   config = List( valueType=str,
                  help="List of config commands published by this source" )

   def render( self ):
      print( f"\tSource: { self.source }" )
      for cmd in self.config:
         print( f"\t\t{ cmd }" )
      print()

class SwitchportsConfigSources( Model ):
   interfaces = Dict( keyType=Interface, valueType=ConfigSource,
                      help="Switchport config source indexed by interface "
                      "name" )

   def render( self ):
      for intfName in Arnet.sortIntf( self.interfaces ):
         configSource = self.interfaces[ intfName ]
         print( f"Interface: { intfName }" )
         configSource.render()
