# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliModel
from CliPlugin.CbfNhgCli import (
   IPv4,
   IPv6,
)
import TableOutput
import Tac

FecOverrideStatusEnum = Tac.Type( 'Qos::Smash::FecOverrideStatusEntry::Status' )

# PLEASE READ! This dictionary maps the current enum values of
# FecOverrideStatusEntry::Status to their previous values for CAPI backwards
# compatibility. If an enum value ever changes, its new value MUST be mapped to its
# previous value here. Reach out to cbfnhg-dev@ if there is any confusion.
StableCapiStatusEnum = {
   FecOverrideStatusEnum.unknownStatus: 'unknownStatus',
   FecOverrideStatusEnum.programmed: 'programmed',
   FecOverrideStatusEnum.outOfResource: 'outOfResource',
   FecOverrideStatusEnum.fecNotFound: 'fecNotFound',
   FecOverrideStatusEnum.hwNotReady: 'hwNotReady',
   FecOverrideStatusEnum.hwRequestPending: 'hwRequestPending',
}

# A dictionary which maps values of StableCapiStatusEnum to their prettified string
# representation displayed in the CLI output.
StableCapiStatusEnumToPrettyString = {
   'unknownStatus': 'n/a',
   'programmed': 'programmed',
   'outOfResource': 'out of resource',
   'fecNotFound': 'fec not found',
   'hwNotReady': 'not ready',
   'hwRequestPending': 'request pending',
}

# See AID/12 for CLI style guidelines.
formatText = TableOutput.Format( justify='left', wrap=True )
formatText.padLimitIs( True )
formatNumeric = TableOutput.Format( justify='right', wrap=True )
formatNumeric.padLimitIs( True )

def prettifyFecOverrideEntryStatus( status ):
   if status is None:
      return 'n/a'
   elif stableEnumValue := StableCapiStatusEnum.get( status ):
      # If for some reason stableEnumValue is not found in
      # StableCapiStatusEnumToPrettyString we will raise a KeyError as something
      # has unexpectedly gone wrong which should not be ignored.
      return StableCapiStatusEnumToPrettyString[ stableEnumValue ]
   raise ValueError( f'Unexpected enum value: { status }' )

class CbfNhgOverride( CliModel.Model ):
   defaultNhg = CliModel.Str( help='Default nexthop group name' )
   overrideNhg = CliModel.Str( help='Override nexthop group name' )
   addressFamily = CliModel.Enum( values=[ IPv4, IPv6 ], help='Address family' )
   dscp = CliModel.Int( help='DSCP value' )
   defaultFecId = CliModel.Int( help='Default nexthop group FEC ID',
                                optional=True )
   overrideFecId = CliModel.Int( help='Overriding nexthop group FEC ID',
                                 optional=True )
   ecmp = CliModel.Bool( help='ECMP group', optional=True )
   status = CliModel.Enum( values=StableCapiStatusEnum.keys(),
                           help='TCAM programming status for this FEC override',
                           optional=True )
   inPackets = CliModel.Int( help='Ingress packet count', optional=True )
   inOctets = CliModel.Int( help='Ingress octet count', optional=True )

   def render( self ):
      defaultFecId = 'n/a'
      if self.defaultFecId is not None:
         defaultFecId = self.defaultFecId
      overrideFecId = 'n/a'
      if self.overrideFecId is not None:
         overrideFecId = self.overrideFecId
      ecmp = 'n/a'
      if self.ecmp is not None:
         ecmp = 'yes' if self.ecmp else 'no'
      status = prettifyFecOverrideEntryStatus( self.status )
      # Explicitly check "is not None" because 0 (which will also evaluate to False)
      # is a valid counter value. We only want to display "n/a" when
      # inPackets/inOctets are not set.
      inPackets = 'n/a'
      if self.inPackets is not None:
         inPackets = self.inPackets
      inOctets = 'n/a'
      if self.inOctets is not None:
         inOctets = self.inOctets
      return ( self.defaultNhg,
               self.overrideNhg,
               self.addressFamily,
               self.dscp,
               defaultFecId,
               overrideFecId,
               ecmp,
               status,
               inPackets,
               inOctets, )

   @staticmethod
   def key( obj ):
      '''Convert a CbfNhgOverride object or dict representation to a tuple of
      its attribute values which can be used for sorting. The tuple is ordered in
      decreasing attribute priority.
      '''
      if isinstance( obj, dict ):
         obj = CliModel.unmarshalModel( CbfNhgOverride, obj )
      elif not isinstance( obj, CbfNhgOverride ):
         raise TypeError( f'{ type( obj ) } is not a dict or CbfNhgOverride' )
      return ( obj.defaultNhg,
               obj.overrideNhg,
               obj.addressFamily,
               obj.dscp,
               obj.defaultFecId,
               obj.overrideFecId, )

class CbfNhgOverrideTable( CliModel.Model ):
   # This attribute is not included in the CAPI dict. It is only for the render
   # functions to determine if the counter columns should be displayed.
   _renderCounters = CliModel.Bool( default=False,
                                    help='Render the InOctets/InPackets columns' )
   overrides = CliModel.List( valueType=CbfNhgOverride,
                              help='CBF nexthop group FEC overrides' )

   def headings( self ):
      return [ 'Default NHG',
               'Overriding NHG',
               'AF',
               'DSCP',
               'Default FECs',
               'Overriding FECs',
               'ECMP',
               'Status', ]

   def formats( self ):
      return [ formatText,
               formatText,
               formatText,
               formatNumeric,
               formatNumeric,
               formatNumeric,
               formatText,
               formatText, ]

   def render( self ):
      headings = self.headings()
      formats = self.formats()
      if self._renderCounters:
         headings.extend( [ 'InPackets', 'InOctets' ] )
         formats.extend( [ formatNumeric, formatNumeric ] )
      # Set tableWidth to match NexthopGroup::NameIndex::max. See
      # testLongNexthopGroupName in CbfNhgCliTest for an example of the output.
      table = TableOutput.createTable( headings, tableWidth=127 )
      table.formatColumns( *formats )
      for override in sorted( self.overrides, key=CbfNhgOverride.key ):
         row = self.renderRow( override )
         table.newRow( *row )
      print( table.output() )

   def renderRow( self, override ):
      # Exclude the last 2 columns ("InOctets" and "InPackets") when _renderCounters
      # is False.
      return override.render() if self._renderCounters else override.render()[ : -2 ]
