#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ClbCliLib
import CliGlobal
import ConfigMount
import LazyMount
import SharkLazyMount
import SmashLazyMount
import Tac

gv = CliGlobal.CliGlobal( config=None,
                          status=None,
                          checkpoint=None,
                          clbDirectFlowStatus=None,
                          directFlowStatusDir=None,
                          trafficPolicyCounters=None,
                          capabilities=None,
                          directHwFlowStatus=None,
                          directHwSmStatus=None )

def clearFlowTable( mode, args ):
   gv.config.clearLbFlows = Tac.now()

def clearCounters( mode, args ):
   if 'session' in args:
      checkpoint = mode.session.sessionData( ClbCliLib.SESSION_DATA_COUNTER )
      if not checkpoint:
         checkpoint = Tac.newInstance( "Clb::CounterCheckpoint", "" )
         mode.session.sessionDataIs( ClbCliLib.SESSION_DATA_COUNTER,
                                     checkpoint )
   else:
      checkpoint = gv.checkpoint

   # copy over all the unmatched counters
   if gv.status.counterStatus:
      for pgName, pg in gv.status.counterStatus.portGroupCounter.items():
         pgCounter = checkpoint.portGroupCounter.newMember( pgName )
         pgCounter.unmatchedNormal = pg.unmatchedNormal
         pgCounter.unmatchedOverflow = pg.unmatchedOverflow

      # delete non-existent groups
      for pgName in checkpoint.portGroupCounter:
         if pgName not in gv.status.counterStatus.portGroupCounter:
            del checkpoint.portGroupCounter[ pgName ]
   else:
      checkpoint.portGroupCounter.clear()

   trafficPolicyCache = ClbCliLib.trafficPolicyFlowStatsCache(
      gv.capabilities, gv.trafficPolicyCounters )

   if gv.status.flowStatus:
      for flowKey in gv.status.flowStatus.flow:
         if gv.capabilities.hwOffloadMethod == 'directFlow':
            counters = ClbCliLib.getDirectFlowCounters( gv.clbDirectFlowStatus,
                                                        gv.directFlowStatusDir,
                                                        flowKey )
         elif gv.capabilities.hwOffloadMethod == 'trafficPolicy':
            counters = trafficPolicyCache.get( flowKey )
         elif gv.capabilities.hwOffloadMethod == 'directHw':
            counters = ClbCliLib.getDirectHwCounters( gv.directHwFlowStatus,
                                                      gv.directHwSmStatus,
                                                      flowKey )
         else:
            counters = None
         if counters is not None:
            packetCount, byteCount = counters
            flowCounter = Tac.Value( "Clb::Counter", packetCount, byteCount )
            checkpoint.flowCounter[ flowKey ] = flowCounter

      # delete non-existent flows
      for flowKey in checkpoint.flowCounter:
         if flowKey not in gv.status.flowStatus.flow:
            del checkpoint.flowCounter[ flowKey ]
   else:
      checkpoint.flowCounter.clear()

   checkpoint.checkpointTime = Tac.now()

def Plugin( entityManager ):
   gv.config = ConfigMount.mount( entityManager, "clb/config",
                                  "Clb::Config", "w" )
   gv.status = LazyMount.mount( entityManager, "clb/status",
                                "Clb::Status", "r" )
   gv.clbDirectFlowStatus = LazyMount.mount( entityManager, 'clb/directflow/status',
                                             'Clb::DirectFlowStatus', 'r' )
   gv.directFlowStatusDir = LazyMount.mount( entityManager,
                                             'openflow/directflowhwstatusdir',
                                             'Tac::Dir', 'ri' )
   gv.directHwSmStatus = LazyMount.mount( entityManager, 'clb/directHw/smStatus',
                                          'Clb::DirectHwSmStatus', 'ri' )
   gv.trafficPolicyCounters = SharkLazyMount.mount(
      entityManager,
      'policyMap/counters/trafficPolicyInterface/input',
      'PolicyMap::Counters::PolicyMapTypeCounters',
      SharkLazyMount.mountInfo( 'shadow' ),
      autoUnmount=True )
   gv.capabilities = LazyMount.mount( entityManager, 'clb/hardware/capabilities',
                                      'Clb::Hardware::Capabilities', 'r' )
   gv.checkpoint = LazyMount.mount( entityManager, "clb/checkpoint",
                                    "Clb::CounterCheckpoint", "w" )

   gv.directHwFlowStatus = SmashLazyMount.mount(
      entityManager,
      "clb/directHw/flowStatus",
      "Clb::DirectHwFlowStatus",
      SmashLazyMount.mountInfo( 'reader' ) )
