#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ClbCliLib
import CliDynamicSymbol
import CliGlobal
import LazyMount
import SharkLazyMount
import SmashLazyMount
import Tac

import statistics

ClbFlowModel = CliDynamicSymbol.CliDynamicPlugin( "ClbFlowModel" )

gv = CliGlobal.CliGlobal( clbConfig=None,
                          clbStatus=None,
                          clbCheckpoint=None,
                          vrfIdStatus=None,
                          snoopInput=None,
                          clbDirectFlowStatus=None,
                          l3Config=None,
                          directFlowStatusDir=None,
                          trafficPolicyCounters=None,
                          requirements=None,
                          capabilities=None,
                          warningStatus=None,
                          directHwFlowStatus=None,
                          directHwFlowCounterStatus=None,
                          directHwSmStatus=None )

def getCheckpointCounters( mode ):
   sessionCheckpoint = mode.session.sessionData( ClbCliLib.SESSION_DATA_COUNTER )
   if ( sessionCheckpoint and
        sessionCheckpoint.checkpointTime > gv.clbCheckpoint.checkpointTime ):
      # pick the later one
      return sessionCheckpoint
   return gv.clbCheckpoint

def getCommGroupIdStr( groupId ):
   if groupId:
      gid = Tac.newInstance( 'ClbFlow::FlowGroupId', groupId )
      threshold = 'A' if gid.type == 'flowGroupIdThresholdTypeLow' else 'B'
      return f'{gid.gidValue()}{threshold}'
   else:
      return '0'


def makeCliFlow( mode, tacFlow, outFlow, trafficPolicyCache ):
   dstIntf = None
   vtep = None
   ipNh = None
   destType = 'unknown'
   if outFlow and outFlow.dest:
      if outFlow.dest.containsVxlan():
         destType = "bridgedVxlan"
         dstIntf = outFlow.dest.vxlan.outputIntf
         vtep = outFlow.dest.vxlan.vtepAddr
      else:
         assert outFlow.dest.containsIp()
         destType = "routed"
         dstIntf = outFlow.dest.ip.outputIntf
         ipNh = outFlow.dest.ip.ipNextHop

   ve = ( gv.vrfIdStatus.vrfIdToName.get( tacFlow.vrfId ) if
          gv.vrfIdStatus.vrfIdToName else None )

   cliFlow = ClbFlowModel.Flow()
   cliFlow.vrfName = ve.vrfName if ve else 'UNKNOWN'
   cliFlow.srcAddr = tacFlow.srcAddr
   cliFlow.dstAddr = tacFlow.dstAddr
   cliFlow.srcPort = tacFlow.srcPort
   cliFlow.dstPort = tacFlow.dstPort
   cliFlow.queuePair = tacFlow.dstQueuePair
   cliFlow.dstMac = tacFlow.dstMac
   cliFlow.vlanId = tacFlow.vlanId
   cliFlow.srcIntf = tacFlow.l2SrcPort
   cliFlow.destType = destType
   cliFlow.protocol = Tac.enumValue( 'Arnet::IpProtocolNumber', tacFlow.protocol )
   cliFlow.inHardware = tacFlow.inHardware
   cliFlow.hardwareStatus = str( tacFlow.hardwareStatus )

   # Turn Tac Enum Value into a readable string
   FlowProgStatusEnum = Tac.Type( "ClbFlow::FlowProgStatus" )
   if tacFlow.programStatus == FlowProgStatusEnum.FlowProgStatusUnprogrammed:
      cliFlow.programStatus = 'Flow Not Programmed'
   elif tacFlow.programStatus == FlowProgStatusEnum.FlowProgStatusFlowLimitExceeded:
      cliFlow.programStatus = 'Flow Limit Exceeded'
   elif tacFlow.programStatus == FlowProgStatusEnum.FlowProgStatusSrcPortNoGroup:
      cliFlow.programStatus = 'Flow No Group Configured For Source Port'
   elif tacFlow.programStatus == FlowProgStatusEnum.FlowProgStatusRouteLookupFailed:
      cliFlow.programStatus = 'Flow Route Lookup Failed'
   elif tacFlow.programStatus == FlowProgStatusEnum.FlowProgStatusSnoopGroupSuccess:
      cliFlow.programStatus = 'Flow Programming Communicator Group Id Success'
   elif tacFlow.programStatus == FlowProgStatusEnum.FlowProgStatusRouteLookupSuccess:
      cliFlow.programStatus = 'Flow Programming Success'
   cliFlow.commGroupId = getCommGroupIdStr( tacFlow.commGroupId )
   cliFlow.fieldSetId = tacFlow.routeGroupId.fieldSetId
   cliFlow.routeGroupVtep = tacFlow.routeGroupId.vtepAddr

   checkpoint = getCheckpointCounters( mode )

   if gv.capabilities.hwOffloadMethod == 'directFlow':
      cliFlow.directFlowName = ClbCliLib.getDirectFlowName( gv.clbDirectFlowStatus,
                                                            tacFlow.flowKey )
      counters = ClbCliLib.getDirectFlowCounters(
         gv.clbDirectFlowStatus, gv.directFlowStatusDir, tacFlow.flowKey )
   elif gv.capabilities.hwOffloadMethod == 'trafficPolicy':
      counters = trafficPolicyCache.get( tacFlow.flowKey )
   elif gv.capabilities.hwOffloadMethod == 'directHw':
      counters = ClbCliLib.getDirectHwCounters( gv.directHwFlowCounterStatus,
                                                gv.directHwSmStatus,
                                                tacFlow.flowKey )
   else:
      counters = None

   if counters is not None:
      packetCount, byteCount = counters
      flowCheckpoint = checkpoint.flowCounter.get( tacFlow.flowKey )
      if ( flowCheckpoint and packetCount >= flowCheckpoint.packetCount and
            byteCount >= flowCheckpoint.byteCount ):
         packetCount -= flowCheckpoint.packetCount
         byteCount -= flowCheckpoint.byteCount
      cliFlow.packetCount = packetCount
      cliFlow.byteCount = byteCount

   if dstIntf:
      cliFlow.dstIntf = dstIntf

   assert not ( vtep and ipNh )

   if vtep:
      cliFlow.vtep = vtep

   if ipNh:
      cliFlow.ipNextHop = ipNh

   return cliFlow

def showClbFlow( mode, args ):
   clbStatus = gv.clbStatus
   showFlowModel = ClbFlowModel.ShowFlowModel()
   showFlowModel.detailIs( 'detail' in args )
   showFlowModel.countersOnlyIs( 'counters' in args )

   if gv.l3Config.protocolAgentModel != 'multi-agent':
      showFlowModel.warnAboutProtocolModelIs( True )

   # this may be uninitialized
   if clbStatus.flowStatus is None:
      showFlowModel.totalFlows = 0
      return showFlowModel

   if 'VRF' in args:
      showFlowModel.filterVrfIs( args[ 'VRF' ] )

   if 'SRC_IP' in args:
      showFlowModel.filterSrcAddrIs( args[ 'SRC_IP' ] )

   if 'DST_IP' in args:
      showFlowModel.filterDstAddrIs( args[ 'DST_IP' ] )

   if 'QP' in args:
      showFlowModel.filterQueuePairIs( args[ 'QP' ] )

   if 'DST_MAC' in args:
      showFlowModel.filterDstMacIs( args[ 'DST_MAC' ] )

   if 'SRC_INTF' in args:
      for intf in args[ 'SRC_INTF' ]:
         showFlowModel.addFilterSrcIntf( intf )

   if 'DST_INTF' in args:
      for intf in args[ 'DST_INTF' ]:
         showFlowModel.addFilterDstIntf( intf )

   if 'VLAN_ID' in args:
      showFlowModel.filterVlanIdIs( args[ 'VLAN_ID' ] )

   trafficPolicyCache = ClbCliLib.trafficPolicyFlowStatsCache(
      gv.capabilities, gv.trafficPolicyCounters )

   for ( tacFlowKey, tacFlow ) in clbStatus.flowStatus.flow.items():
      outFlow = None
      flowOutput = clbStatus.flowOutput
      if flowOutput:
         outFlow = flowOutput.flow.get( tacFlowKey ) # fix display w/o outflow
      cliFlow = makeCliFlow( mode, tacFlow, outFlow, trafficPolicyCache )
      showFlowModel.maybeAddFlow( cliFlow )

   flowStatus = clbStatus.flowStatus
   showFlowModel.totalFlows = len( flowStatus.flow ) if flowStatus else 0
   return showFlowModel

def makeCliLearnedFlow( tacSnoopFlow ):
   cliFlow = ClbFlowModel.LearnedFlow()

   cliFlow.srcAddr = tacSnoopFlow.key.sip
   cliFlow.dstAddr = tacSnoopFlow.key.dip
   cliFlow.srcPort = tacSnoopFlow.key.sport.value
   cliFlow.dstPort = tacSnoopFlow.key.dport.value
   cliFlow.protocol = Tac.enumValue( 'Arnet::IpProtocolNumber',
                                     tacSnoopFlow.key.protocol )
   cliFlow.queuePair = tacSnoopFlow.key.queuePair
   cliFlow.srcIntf = tacSnoopFlow.key.ingIntfId
   cliFlow.vlanId = tacSnoopFlow.key.vlanId
   cliFlow.dstMac = tacSnoopFlow.key.dstMac
   cliFlow.firstSeen = tacSnoopFlow.firstSeen
   cliFlow.writeLength = tacSnoopFlow.writeLength
   cliFlow.remoteKey = tacSnoopFlow.remoteKey
   cliFlow.virtualAddress = tacSnoopFlow.virtualAddress

   for wl in tacSnoopFlow.writeLengthHistory.values():
      cliFlow.writeLengthHistory.append( wl )

   for rk in tacSnoopFlow.remoteKeyHistory.values():
      cliFlow.remoteKeyHistory.append( rk )

   for va in tacSnoopFlow.virtualAddressHistory.values():
      cliFlow.virtualAddressHistory.append( va )

   return cliFlow

def showClbLearnedFlow( mode, args ):
   model = ClbFlowModel.ShowLearnedFlowModel()
   model.detailIs( 'detail' in args )

   if 'SRC_IP' in args:
      model.filterSrcAddrIs( args[ 'SRC_IP' ] )

   if 'DST_IP' in args:
      model.filterDstAddrIs( args[ 'DST_IP' ] )

   if 'QP' in args:
      model.filterQueuePairIs( args[ 'QP' ] )

   if 'DST_MAC' in args:
      model.filterDstMacIs( args[ 'DST_MAC' ] )

   if 'SRC_INTF' in args:
      for intf in args[ 'SRC_INTF' ]:
         model.addFilterSrcIntf( intf )

   if 'VLAN_ID' in args:
      model.filterVlanIdIs( args[ 'VLAN_ID' ] )

   for snoopFlow in gv.snoopInput.flow.values():
      cliFlow = makeCliLearnedFlow( snoopFlow )
      model.maybeAddFlow( cliFlow )

   model.totalFlows = len( gv.snoopInput.flow )
   return model

def buildFlowsByDest():
   clbStatus = gv.clbStatus
   flowsByDest = {}
   if not clbStatus.flowStatus:
      return flowsByDest

   for flowKey in clbStatus.flowStatus.flow:
      outFlow = clbStatus.flowOutput.flow.get( flowKey )
      if outFlow and outFlow.dest:
         dest = outFlow.dest
         if dest.containsIp():
            dest = dest.ip
         else:
            assert dest.containsVxlan()
            dest = dest.vxlan
         if dest not in flowsByDest:
            flowsByDest[ dest ] = []
         flowsByDest[ dest ].append( flowKey )

   return flowsByDest

def buildFlowCounters( mode ):
   clbStatus = gv.clbStatus
   cModelDict = {}
   pgNameById = {}

   def getModel( pgName=None, gid=None ):
      assert pgName is None or gid is None

      if gid:
         pgName = pgNameById.get( gid )
         if not pgName:
            return None
         # fall through

      if pgName:
         m = cModelDict.get( pgName )
         if not m:
            m = ClbFlowModel.PortGroupCounters()
            m.initZero()
            m.pgName = pgName
            cModelDict[ pgName ] = m
         assert m
         return m

      return None

   if not clbStatus.flowGroupIdStatus:
      # we need this to build the name -> gid mappings, so without it we're done
      return cModelDict

   for ( gidInt, pgName ) in clbStatus.flowGroupIdStatus.portGroupByGroupId.items():
      gid = Tac.newInstance( 'ClbFlow::FlowGroupId', gidInt )
      pgNameById[ gid.gidValue() ] = pgName
      m = getModel( pgName=pgName )
      if clbStatus.counterStatus:
         cs = clbStatus.counterStatus.portGroupCounter.get( pgName )
         if cs:
            m.normalFallbackPacketCount = cs.unmatchedNormal.packetCount
            m.normalFallbackByteCount = cs.unmatchedNormal.byteCount
            m.overflowFallbackPacketCount = cs.unmatchedOverflow.packetCount
            m.overflowFallbackByteCount = cs.unmatchedOverflow.byteCount
            checkpoint = getCheckpointCounters( mode )
            pgCounter = checkpoint.portGroupCounter.get( pgName )
            if pgCounter:
               if ( m.normalFallbackPacketCount >=
                    pgCounter.unmatchedNormal.packetCount and
                    m.normalFallbackByteCount >=
                    pgCounter.unmatchedNormal.byteCount ):
                  m.normalFallbackPacketCount -= \
                     pgCounter.unmatchedNormal.packetCount
                  m.normalFallbackByteCount -= pgCounter.unmatchedNormal.byteCount
               if ( m.overflowFallbackPacketCount >=
                    pgCounter.unmatchedOverflow.packetCount and
                    m.overflowFallbackByteCount >=
                    pgCounter.unmatchedOverflow.byteCount ):
                  m.overflowFallbackPacketCount -= \
                     pgCounter.unmatchedOverflow.packetCount
                  m.overflowFallbackByteCount -= \
                     pgCounter.unmatchedOverflow.byteCount

   if clbStatus.flowL2SrcPortToFlowKeyDir:
      keyDir = clbStatus.flowL2SrcPortToFlowKeyDir
      for ( intfId, flowKeyMap ) in keyDir.flowL2SrcPortToFlowKeyMap.items():
         if intfId not in clbStatus.flowGroupIdStatus.groupIdByIntfId:
            continue
         gidInt = clbStatus.flowGroupIdStatus.groupIdByIntfId.get( intfId )
         gid = Tac.newInstance( 'ClbFlow::FlowGroupId', gidInt )
         m = getModel( gid=gid.gidValue() )
         if not m:
            continue
         a = len( flowKeyMap.commGroupedFlowSet )
         notA = len( flowKeyMap.nonCommGroupedFlowSet )
         m.allocatedFlows += a
         m.unallocatedFlows += notA
         m.learnedFlows += ( a + notA )

   if clbStatus.groupDir:
      for ( gidInt, flowGroup ) in clbStatus.groupDir.flowGroup.items():
         gid = Tac.newInstance( 'ClbFlow::FlowGroupId', gidInt )
         m = getModel( gid=gid.gidValue() )
         if not m:
            continue

         inHwCount = 0
         notInHwCount = 0
         for key in flowGroup.flow:
            flow = clbStatus.flowStatus.flow.get( key )
            if not flow:
               continue

            if flow.inHardware:
               inHwCount += 1
            else:
               notInHwCount += 1

         m.programmedFlows += inHwCount
         m.unprogrammedFlows += notInHwCount

   return cModelDict

def showClbStats( mode, args ):
   clbStatus = gv.clbStatus
   statsModel = ClbFlowModel.ShowStatsModel()
   statsModel.initZero()

   if( clbStatus.flowStatus and clbStatus.groupDir and clbStatus.routeGroupDir and
       clbStatus.fecIdStatus ):

      statsModel.numRawLearnedFlows = len( gv.snoopInput.flow )
      statsModel.numClbFlows = len( clbStatus.flowStatus.flow )

      statsModel.numClbCommGroups = len( clbStatus.groupDir.flowGroup )
      if statsModel.numClbCommGroups:
         commGroupSizes = list( map( lambda x: len( x.flow ),
                                     clbStatus.groupDir.flowGroup.values() ) )
         statsModel.clbCommGroupMean = float( statistics.mean( commGroupSizes ) )
         if len( commGroupSizes ) > 1:
            statsModel.clbCommGroupStDev = \
               float( statistics.stdev( commGroupSizes ) )

      statsModel.numClbRouteGroups = len( clbStatus.routeGroupDir.flowGroup )
      if statsModel.numClbRouteGroups:
         routeGroupSizes = list( map( lambda x: len( x.flow ),
                                      clbStatus.routeGroupDir.flowGroup.values() ) )
         statsModel.clbRouteGroupMean = float( statistics.mean( routeGroupSizes ) )
         if len( routeGroupSizes ) > 1:
            statsModel.clbRouteGroupStDev = \
               float( statistics.stdev( routeGroupSizes ) )

      flowsByDest = buildFlowsByDest()
      statsModel.numClbDests = len( flowsByDest )
      if statsModel.numClbDests:
         destSizes = list( map( len, flowsByDest.values() ) )
         statsModel.clbDestMean = float( statistics.mean( destSizes ) )
         if len( destSizes ) > 1:
            statsModel.clbDestStDev = float( statistics.stdev( destSizes ) )

   cModelDict = buildFlowCounters( mode )
   for m in cModelDict.values():
      statsModel.addPgCounters( m )

   return statsModel

def showClbDest( mode, args ):
   clbStatus = gv.clbStatus
   model = ClbFlowModel.ShowClbDestModel()
   model.detailIs( 'detail' in args )

   flowsByDest = buildFlowsByDest()

   vxType = Tac.Type( 'ClbFlow::FlowDestVxlan' )
   ipType = Tac.Type( 'ClbFlow::FlowDestIp' )

   trafficPolicyCache = ClbCliLib.trafficPolicyFlowStatsCache(
      gv.capabilities, gv.trafficPolicyCounters )

   for ( dest, flowList ) in flowsByDest.items():
      destModel = ClbFlowModel.Dest()
      if isinstance( dest, vxType ):
         destModel.destType = 'bridgedVxlan'
         destModel.interface = dest.outputIntf
      else:
         assert isinstance( dest, ipType )
         destModel.destType = 'routed'
         destModel.interface = dest.outputIntf
         destModel.nextHop = dest.ipNextHop

      destAndFlowsModel = ClbFlowModel.DestAndFlows()
      destAndFlowsModel.destination = destModel
      for flowKey in flowList:
         outFlow = clbStatus.flowOutput.flow.get( flowKey )
         if outFlow:
            tacFlow = clbStatus.flowStatus.flow.get( flowKey )
            if tacFlow:
               cliFlow = makeCliFlow( mode, tacFlow, outFlow, trafficPolicyCache )
               destAndFlowsModel.flows.append( cliFlow )

      model.dests.append( destAndFlowsModel )

   return model

# show load-balance cluster flows group communicator summary
def showClbComGroup( mode, args ):
   clbStatus = gv.clbStatus
   model = ClbFlowModel.ShowClbComGroupModel()
   model.summaryIs( 'summary' in args )

   if not clbStatus.groupDir:
      return model

   trafficPolicyCache = ClbCliLib.trafficPolicyFlowStatsCache(
      gv.capabilities, gv.trafficPolicyCounters )

   for groupId in clbStatus.groupDir.flowGroup:
      comGroupAndFlowsModel = ClbFlowModel.ComGroupAndFlows()
      flowGroup = clbStatus.groupDir.flowGroup[ groupId ]
      comGroupAndFlowsModel.groupId = getCommGroupIdStr( groupId )
      for flowKey in flowGroup.flow:
         outFlow = clbStatus.flowOutput.flow.get( flowKey )
         if outFlow:
            tacFlow = clbStatus.flowStatus.flow.get( flowKey )
            if tacFlow:
               cliFlow = makeCliFlow( mode, tacFlow, outFlow, trafficPolicyCache )
               comGroupAndFlowsModel.flows.append( cliFlow )
      model.comGroups.append( comGroupAndFlowsModel )
   return model

# show load-balance cluster group route
def showClbRouteGroup( mode, args ):
   clbStatus = gv.clbStatus
   model = ClbFlowModel.ShowClbRouteGroupModel()
   model.summaryIs( 'summary' in args )

   if not clbStatus.routeGroupDir:
      return model

   trafficPolicyCache = ClbCliLib.trafficPolicyFlowStatsCache(
      gv.capabilities, gv.trafficPolicyCounters )

   for routeGroupId in clbStatus.routeGroupDir.flowGroup:
      routeGroupIdModel = ClbFlowModel.RouteGroupId()

      routeGroupIdModel.groupId = getCommGroupIdStr( routeGroupId.inputFlowGroupId )
      routeGroupIdModel.fieldSetId = routeGroupId.fieldSetId
      routeGroupIdModel.vtepAddr = routeGroupId.vtepAddr
      routeGroupAndFlowsModel = ClbFlowModel.RouteGroupAndFlows()
      routeGroupAndFlowsModel.routeGroupId = routeGroupIdModel
      flowRouteGroup = clbStatus.routeGroupDir.flowGroup[ routeGroupId ]
      for flowKey in flowRouteGroup.flow:
         outFlow = clbStatus.flowOutput.flow.get( flowKey )
         if outFlow:
            tacFlow = clbStatus.flowStatus.flow.get( flowKey )
            if tacFlow:
               cliFlow = makeCliFlow( mode, tacFlow, outFlow, trafficPolicyCache )
               routeGroupAndFlowsModel.flows.append( cliFlow )
      model.routeGroups.append( routeGroupAndFlowsModel )
   return model

def showClbWarnings( mode, args ):
   model = ClbFlowModel.ShowClbWarningsModel()

   for w in gv.warningStatus.warning:
      wModel = ClbFlowModel.WarningModel()
      if w.warningType == 'warningTypeDuplicatePort':
         wModel.warningType = 'duplicatePort'
      elif w.warningType == 'warningTypeMissingPort':
         wModel.warningType = 'missingPort'
      elif w.warningType == 'warningTypeUplinkInconsistent':
         wModel.warningType = 'uplinkInconsistent'
      elif w.warningType == 'warningTypeMissingVxlanInterface':
         wModel.warningType = 'missingVxlanInterface'

      wModel.interface = w.intfId

      model.warningList.append( wModel )

   return model

def showClbStatus( mode, args ):
   model = ClbFlowModel.ShowClbStatusModel()
   assert model

   if gv.requirements.forwardingMode == 'disabled':
      model.clbStatus = 'disabled'
   elif gv.clbConfig.flowMonitor:
      model.clbStatus = 'monitor'
   else:
      model.clbStatus = 'enabled'

   if model.clbStatus == 'enabled':
      # right now we only support one global action, but in the future it
      # will be per-port group, so just replicate it for now

      for ( pgName, pg ) in gv.clbConfig.hostIntfGroup.items():
         pgStatus = ClbFlowModel.PortGroupStatus()
         pgStatus.pgName = pgName

         action = pg.flowExhaustionAction
         if action and action.dscp != action.dscpInvalid:
            pgStatus.fallbackDscp = action.dscp
         else:
            pgStatus.fallbackDscp = None

         if action and action.trafficClass != action.trafficClassInvalid:
            pgStatus.fallbackTrafficClass = action.trafficClass
         else:
            pgStatus.fallbackTrafficClass = None
         model.pgStatus.append( pgStatus )

   return model

def Plugin( entityManager ):
   gv.clbConfig = LazyMount.mount( entityManager, 'clb/config', 'Clb::Config', 'r' )
   gv.clbStatus = LazyMount.mount( entityManager, 'clb/status', 'Clb::Status', 'r' )
   gv.vrfIdStatus = SmashLazyMount.mount( entityManager, "vrf/vrfIdMapStatus",
                                          "Vrf::VrfIdMap::Status",
                                          SmashLazyMount.mountInfo( 'reader' ),
                                          autoUnmount=True )
   gv.snoopInput = SmashLazyMount.mount( entityManager, 'clb/snoopFlow',
                                         'Clb::SnoopFlowDir',
                                         SmashLazyMount.mountInfo( 'reader' ) )
   gv.clbDirectFlowStatus = LazyMount.mount( entityManager, 'clb/directflow/status',
                                             'Clb::DirectFlowStatus', 'r' )
   gv.l3Config = LazyMount.mount( entityManager, 'l3/config', 'L3::Config', 'r' )
   gv.directFlowStatusDir = LazyMount.mount( entityManager,
                                             'openflow/directflowhwstatusdir',
                                             'Tac::Dir', 'ri' )
   gv.directHwSmStatus = LazyMount.mount( entityManager, 'clb/directHw/smStatus',
                                          'Clb::DirectHwSmStatus', 'ri' )
   gv.trafficPolicyCounters = SharkLazyMount.mount(
      entityManager,
      'policyMap/counters/trafficPolicyInterface/input',
      'PolicyMap::Counters::PolicyMapTypeCounters',
      SharkLazyMount.mountInfo( 'shadow' ),
      autoUnmount=True )
   gv.capabilities = LazyMount.mount( entityManager, 'clb/hardware/capabilities',
                                      'Clb::Hardware::Capabilities', 'r' )
   gv.requirements = LazyMount.mount( entityManager, 'clb/hardware/requirements',
                                      'Clb::Hardware::Requirements', 'r' )
   gv.warningStatus = LazyMount.mount( entityManager, 'clb/warningStatus',
                                       'Clb::WarningStatus', 'r' )
   gv.clbCheckpoint = LazyMount.mount( entityManager, 'clb/checkpoint',
                                       'Clb::CounterCheckpoint', 'w' )
   gv.directHwFlowStatus = SmashLazyMount.mount(
      entityManager,
      "clb/directHw/flowStatus",
      "Clb::DirectHwFlowStatus",
      SmashLazyMount.mountInfo( 'reader' ) )
   gv.directHwFlowCounterStatus= SmashLazyMount.mount(
      entityManager,
      "clb/directHw/flowCounterStatus",
      "Clb::DirectHwFlowCounterStatus",
      SmashLazyMount.mountInfo( 'reader' ) )
