#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import ClbCliLib
import CliCommand
import CliDynamicSymbol
import CliMatcher
import CliMode.Clb
import ShowCommand
import Tac
import difflib
import io
import sys

ClbConfigHandlers = CliDynamicSymbol.CliDynamicPlugin( "ClbConfigHandlers" )

gv = ClbConfigHandlers.gv
flowKw = ClbConfigHandlers.flowKw

class ClbMatchPolicyConfigMode( CliMode.Clb.ClbMatchPolicyMode,
                                  BasicCli.ConfigModeBase ):
   # we'll register our own "show active"
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session, name ):
      self.pendingConfig_ = None
      CliMode.Clb.ClbMatchPolicyMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def validatePolicy( self ):
      return self.pendingConfig_.valid

   def initialize( self ):
      if self.pendingConfig_:
         # already initialized
         return
      # initialize scratchPad from existing policy
      config = gv.config.flowMatchPolicy.get( self.policyName )
      self.pendingConfig_ = config or Tac.Value( "Clb::TrafficPolicyMatch" )

   def abort( self ):
      self.pendingConfig_ = None

   def commit( self ):
      if self.pendingConfig_ is None:
         # not initialized or aborted, do nothing
         return

      # verify the pending config is complete
      if self.validatePolicy():
         gv.config.flowMatchPolicy[ self.policyName ] = self.pendingConfig_
      else:
         self.addError( "cannot commit policy: invalid match" )

   def _showPolicy( self, policy, output=None, forceDisplayName=False ):
      if output is None:
         output = sys.stdout
      if policy or forceDisplayName:
         output.write( f"load-balance cluster\n"
                       f"   flow match policy {self.policyName}\n" )
      if policy:
         cmd = ClbCliLib.getMatchPolicyCommand( policy )
         if cmd:
            output.write( f"      {cmd}\n" )

   def showActive( self, output=None ):
      self._showPolicy( gv.config.flowMatchPolicy.get( self.policyName ),
                        output=output )

   def showPending( self, output=None ):
      self._showPolicy( self.pendingConfig_, output=output,
                        forceDisplayName=True )

   def showDiff( self ):
      activeOutput = io.StringIO()
      self.showActive( output=activeOutput )
      pendingOutput = io.StringIO()
      self.showPending( output=pendingOutput )

      diff = difflib.unified_diff( activeOutput.getvalue( ).splitlines( ),
                                   pendingOutput.getvalue( ).splitlines( ),
                                   lineterm='' )
      print( '\n'.join( list( diff ) ) )

   def onExit( self ):
      self.commit()
      BasicCli.ConfigModeBase.onExit( self )

def enterMatchPolicyMode( mode, args ):
   childMode = mode.childMode( ClbMatchPolicyConfigMode, name=args[ 'NAME' ] )
   mode.session.gotoChildMode( childMode )

class MatchCommand( CliCommand.CliCommandClass ):
   syntax = "match rocev2 ib-bth opcode value VALUE mask MASK [ exclude { OPCODE } ]"
   noOrDefaultSyntax = "match rocev2 ib-bth opcode ..."
   data = {
      "match": "Configure match condition",
      "rocev2": "Match RoCEv2 fields",
      "ib-bth": "Match fields in the IB BTH header",
      "opcode": "Match rocev2 IB BTH opcodes",
      "value": "Specify match value",
      "VALUE": CliMatcher.IntegerMatcher( 0, 255,
                                          helpdesc="Specify match value" ),
      "mask": "Specify match mask",
      "MASK": CliMatcher.IntegerMatcher( 1, 255, helpdesc="Specify match mask" ),
      "exclude": "Exclude a list of opcodes",
      "OPCODE": CliMatcher.IntegerMatcher( 0, 255,
                                            helpdesc="Specify opcode value" ),
      }

   @staticmethod
   def handler( mode, args ):
      value = args[ 'VALUE' ]
      mask = args[ 'MASK' ]
      match = Tac.Value( "Clb::TrafficPolicyMatch" )
      match.value = value & mask
      match.mask = mask
      for e in sorted( args.get( 'OPCODE', [] ) ):
         match.exclude.push( e )
      mode.initialize()
      mode.pendingConfig_ = match

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.initialize()
      mode.pendingConfig_ = Tac.Value( "Clb::TrafficPolicyMatch" )

ClbMatchPolicyConfigMode.addCommandClass( MatchCommand )

class AbortCommand( CliCommand.CliCommandClass ):
   syntax = "abort"
   data = {
      "abort": "Exit the mode without saving"
      }

   @staticmethod
   def handler( mode, args ):
      mode.abort()
      mode.session_.gotoParentMode()

ClbMatchPolicyConfigMode.addCommandClass( AbortCommand )

class ShowActiveDiffPendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pending | active | diff'
   data = {
      'pending': 'Show pending policy',
      'active': 'Show active policy in running-config',
      'diff': 'Show the difference between active and pending policy'
   }

   @staticmethod
   def handler( mode, args ):
      if 'active' in args:
         return mode.showActive()
      elif 'diff' in args:
         return mode.showDiff()
      else:
         return mode.showPending()

ClbMatchPolicyConfigMode.addShowCommandClass( ShowActiveDiffPendingCmd )
