# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import sys

import CliCommand
import CliCommon
from CliDynamicSymbol import CliDynamicPlugin
import ConfigMount
import ShowCommand
from Syscall import gettid

import Tac
import TacUtils

CliCliModel = CliDynamicPlugin( "CliCliModel" )

cliConfig = None

#------------------------------------------------------------------------------------
# show cli pid
#------------------------------------------------------------------------------------
def showCliPid ( mode, args ):
   return CliCliModel.CliPid( pid=gettid() )

#------------------------------------------------------------------------------------
# [ no | default ] prompt PROMPT
#------------------------------------------------------------------------------------
def promptCmd( mode, args ):
   cliConfig.prompt = args.get( 'PROMPT', cliConfig.promptDefault )

#-----------------------------------------------------------------------------------
# show history [ current ]
#-----------------------------------------------------------------------------------
def showHistoryCmd( mode, args ):
   current = 'current' in args
   # We might be running from a config mode - find it instead of the current mode
   # which is always EXEC mode.
   mode = mode.session_.modeOfLastPrompt() or mode
   items = ( mode.session_.cliInput.history( mode.historyKey() ) if current else
             mode.session_.cliInput.history() )
   return CliCliModel.ShowHistory( historyItems=items )

#------------------------------------------------------------------------------------
# clear history
#------------------------------------------------------------------------------------
def clearHistory( mode, args ):
   mode.session_.cliInput.clearHistory()

#-----------------------------------------------------------------------------------
# show privilege
#-----------------------------------------------------------------------------------
def showPrivilege( mode, args ):
   return CliCliModel.PrivilegeLevel(
      privilegeLevel=mode.session.privLevel_,
      _secureMonitor=mode.session.secureMonitor() )

#------------------------------------------------------------------------------------
# [ no | default ] activity-lock-monitor
#------------------------------------------------------------------------------------
def activityLockMonitor( mode, args ):
   cliConfig.activityLockSyslogMsg = True

def activityLockMonitorNot( mode, args ):
   cliConfig.activityLockSyslogMsg = False

#------------------------------------------------------------------------------------
# cli debug parser result
#------------------------------------------------------------------------------------
def cliDebugParserResultCmd( mode, args ):
   mode.session.parserDebugIs( True )

def cliDebugParserResultCmdNot( mode, args ):
   mode.session.parserDebugIs( False )

#------------------------------------------------------------------------------------
# cli debug parser guard
#------------------------------------------------------------------------------------
def cliDebugParserGuardCmd( mode, args ):
   mode.session.guardsEnabledIs( True )

def cliDebugParserGuardCmdNot( mode, args ):
   mode.session.guardsEnabledIs( False )

#------------------------------------------------------------------------------------
# cli debug memory handler { heapcheck | mallinfo | rss | gc | log }
#------------------------------------------------------------------------------------
def cliDebugMemoryHandlerCmd( mode, args ):
   mode.session_.memoryDebugCfgIs( args.get( 'heapcheck' ),
       args.get( 'mallinfo' ), args.get( 'rss' ), args.get( 'py-objects' ),
       args.get( 'gc' ), args.get( 'log' ) )

#-----------------------------------------------------------------------------------
# run [ ignore-errors ] COMMAND
#-----------------------------------------------------------------------------------
def runCommands( mode, args ):
   if mode.session_.entityManager_.isLocalEm():
      mode.addErrorAndStop( "command does not work on standalone Cli" )
   cmds = [ x.strip() for x in args[ 'COMMAND' ].split( ';' ) ]
   ignoreErrors = 'ignore-errors' in args
   cmd = [ '/usr/bin/CliShell', '-e', '-p', str( mode.session_.privLevel_ ) ]
   if mode.session_.disableAaa_:
      cmd += [ '-A' ]
   if not ignoreErrors:
      cmd += [ '--abort-on-error' ]
   cmd += [ '-c', '\n'.join( cmds ) ]
   try:
      Tac.run( cmd, stdin=sys.stdin, stdout=sys.stdout, stderr=sys.stderr,
               env=os.environ )
   except TacUtils.SystemCommandError:
      mode.addError( "Some command(s) failed" )

#-------------------------------------------------------------------------------
# The "cli update-handler <mode> <cmd> <module> <handler>" command
# This command allows to change the code of a cli command at runtime (no restart).
# We assume the <module> with the fixed <handler> is in dir CliHandlerPatches.
#-------------------------------------------------------------------------------
def updateHandlerFunction( mode, args ):
   import CliParser # pylint: disable=import-outside-toplevel
   destModeName = args.get( "MODE" )
   cmd = args.get( "CMD" )
   moduleName = args.get( "MODULE" )
   if moduleName.endswith( ".py" ):
      moduleName = moduleName[ : -3 ]
   handlerName = args.get( "HANDLER" )
   tokens = cmd.split()

   # find the mode (TODO: what about modelets)
   destMode = CliParser.allModes.get( destModeName )
   if not destMode:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "Mode %s does not exist (modes are lazy)" % destModeName )
      return

   # import the new code
   source = "CliHandlerPatches"
   if CliCommand.isNoOrDefaultCmd( args ):
      source = "CliPlugin"
   try:
      __import__( f"{source}.{moduleName}" )
   except ImportError:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "Failed to import %s" % moduleName )
      return
   newModule = sys.modules[ f"{source}.{moduleName}" ]

   # find the new handler
   h = handlerName.split( '.' )
   try:
      if len( h ) == 2: # TODO: handle more depth
         newModule2 = getattr( newModule, h[ 0 ] )
         newHandler = getattr( newModule2, h[ 1 ] )
      else:
         newHandler = getattr( newModule, handlerName )
   except AttributeError:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "Failed to get attribute %s" % handlerName )
      return
   if isinstance( newHandler, str ):
      newHandler = CliCommand.getBoundedMethod( newHandler )

   # parse command to find cmdHandler
   try:
      parseRes = destMode.parse( tokens )
   except CliCommon.ParseError as e:
      mode.addError( f"Command '{cmd}' not found ({e})" )
      return

   # stick new handler into cmdHandler
   if isinstance( parseRes[ "cmdHandler" ].handler,
                  ShowCommand.ShowCommandHandler ):
      parseRes[ "cmdHandler" ].handler.handler_ = newHandler
   else:
      parseRes[ "cmdHandler" ].handler = newHandler
   return

#------------------------------------------------------------------------------------
# Plugin Func
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig

   cliConfig = ConfigMount.mount( entityManager, 'cli/config', 'Cli::Config', 'w' )

