#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: library DhcpServer

import Arnet
import CliPlugin.IntfCli
from CliPlugin import VrfCli
from CliPlugin import IraIpIntfCli
from CliPlugin import IraVrfCli
from CliDynamicSymbol import CliDynamicPlugin
from CliMode.DhcpServer import _maxDnsServers
from CliMode.DhcpServer import DhcpServerSubnetV4Mode
from CliMode.DhcpServer import DhcpServerReservationsInfoOptionMode
from CliMode.DhcpServer import DhcpServerMode

import ConfigMount
from HostnameCli import (
   validateHostname
)
import LazyMount
from EosDhcpServerLib import rangeContains
from EosDhcpServerLib import tacLayer2Info
from EosDhcpServerLib import tacVendorClassOption
from EosDhcpServerLib import vendorSubOptionType
from EosDhcpServerLib import OptionType
import Tac
import FileCliUtil

dhcpServerVrfConfig = None
dhcpServerVrfStatus = None
ipConfig = None

# CliDynamic Plugin
assignDynamicSubmodes = CliDynamicPlugin( "DhcpServerClientAssignMode" )
matchDynamicSubmodes = CliDynamicPlugin( "DhcpServerClientMatchMode" )
nestedMatchDynamicSubmodes = CliDynamicPlugin( "DhcpServerClientNestedMatchMode" )

def clearAssignedClientClass( config ):
   config.assignedClientClass.clear()

def setAssignedClientClass( config, args ):
   clearAssignedClientClass( config )
   clientClassNames = args[ 'CLIENT_CLASS' ]
   for clientClassName in clientClassNames:
      config.assignedClientClass[ clientClassName ] = True

def leaseTimeIs( mode, config, args ):
   if mode.session.hasError():
      return
   leaseTime = args[ 'TIME' ]
   config.leaseTime = leaseTime

def leaseTimeDel( config ):
   config.leaseTime = 0.0

def doEnterDhcpServerNestedMatchModeBaseHandler( mode, args,
                                                 getOrCreateChildModeFunc ):
   matchAny = 'any' in args
   childMode = getOrCreateChildModeFunc( mode, args, matchAny )
   matchName = args[ 'NAME' ]
   childMode.matchCriteria = mode.matchCriteria.newSubMatchCriteria( matchName )
   childMode.matchCriteria.matchAny = matchAny
   mode.session_.gotoChildMode( childMode )

def noEnterDhcpServerNestedMatchModeBaseHandler( mode, args,
                                                 getOrCreateChildModeFunc ):
   matchName = args[ 'NAME' ]
   subMatchCriteria = mode.matchCriteria.subMatchCriteria
   if matchName in subMatchCriteria:
      matchAny = subMatchCriteria[ matchName ].matchAny
      childMode = getOrCreateChildModeFunc( mode, args, matchAny )
      childMode.removeComment()

      del subMatchCriteria[ matchName ]

def doDhcpServerClientClassMatchAristaL2InfoBaseHandler( mode, args ):
   mode.matchCriteria.l2Info[ args[ 'l2Info' ] ] = True

def noDhcpServerClientClassMatchAristaL2InfoBaseHandler( mode, args ):
   del mode.matchCriteria.l2Info[ args[ 'l2Info' ] ]

def nonGlobalClientClassHandler( mode, args, currConfig ):
   clientClassName = args[ 'CLIENT_CLASS' ]
   clientClassMode = args[ 'clientClassAfMode' ]
   clientClassConfig = currConfig.newClientClassConfig( clientClassName )
   childMode = mode.childMode( clientClassMode,
                                 clientClassName=clientClassName )
   childMode.clientClassConfig = clientClassConfig
   mode.session_.gotoChildMode( childMode )

# get CliDynamicPlugin submode helper
def getSubmodeHelper( arg, submodes ):
   className = str( arg ).rsplit( '.', maxsplit=1 )[ -1 ]
   className = ''.join( [ i for i in className if i.isalnum() ] )
   return getattr( submodes, className )

# Assign mode
def getOrCreateChildClientClassAssignMode( mode, args ):
   subMode = getSubmodeHelper( args[ 'clientClassAssignAfMode' ],
                               assignDynamicSubmodes )
   return mode.childMode( subMode,
                          clientClassName=mode.clientClassName )

def getOrCreateChildSubnetClientClassAssignMode( mode, args ):
   subMode = getSubmodeHelper( args[ 'clientClassAssignAfMode' ],
                               assignDynamicSubmodes )
   return mode.childMode( subMode,
                          subnet=mode.subnetConfig.subnetId,
                          clientClassName=mode.clientClassName )

def getOrCreateChildRangeClientClassAssignMode( mode, args ):
   subMode = getSubmodeHelper( args[ 'clientClassAssignAfMode' ],
                               assignDynamicSubmodes )
   return mode.childMode( subMode,
                          subnet=mode.subnetConfig.subnetId,
                          subnetRange=mode.rangeConfig.range,
                          clientClassName=mode.clientClassName, )

# Match mode
def getOrCreateChildClientClassMatchMode( mode, args, matchAny ):
   subMode = getSubmodeHelper( args[ 'clientClassMatchMode' ],
                               matchDynamicSubmodes )
   return mode.childMode( subMode,
                          clientClassName=mode.clientClassName,
                          matchAny=matchAny )

def getOrCreateChildRangeClientClassMatchMode( mode, args, matchAny ):
   subMode = getSubmodeHelper( args[ 'clientClassMatchMode' ],
                               matchDynamicSubmodes )
   return mode.childMode( subMode,
                          subnetConfig=mode.subnetConfig,
                          rangeConfig=mode.rangeConfig,
                          clientClassName=mode.clientClassName,
                          matchAny=matchAny )

def getOrCreateChildSubnetClientClassMatchMode( mode, args, matchAny ):
   subMode = getSubmodeHelper( args[ 'clientClassMatchMode' ],
                               matchDynamicSubmodes )
   return mode.childMode( subMode,
                          subnetPrefix=mode.subnetConfig.subnetId,
                          clientClassName=mode.clientClassName,
                          matchAny=matchAny )

# Nested match mode
def getOrCreateChildClientClassNestedMatchMode( mode, args, matchAny ):
   subMode = getSubmodeHelper( args[ 'nestedMatchMode' ],
                               nestedMatchDynamicSubmodes )
   return mode.childMode( subMode,
                          clientClassName=mode.clientClassName,
                          matchAny=matchAny,
                          matchName=args[ 'NAME' ] )

def getOrCreateChildRangeClientClassNestedMatchMode( mode, args, matchAny ):
   subMode = getSubmodeHelper( args[ 'nestedMatchMode' ],
                               nestedMatchDynamicSubmodes )
   return mode.childMode( subMode,
                          subnetConfig=mode.subnetConfig,
                          rangeConfig=mode.rangeConfig,
                          clientClassName=mode.clientClassName,
                          matchAny=matchAny,
                          matchName=args[ 'NAME' ] )

def getOrCreateChildSubnetClientClassNestedMatchMode( mode, args, matchAny ):
   subMode = getSubmodeHelper( args[ 'nestedMatchMode' ],
                               nestedMatchDynamicSubmodes )
   return mode.childMode( subMode,
                          subnetPrefix=mode.subnetPrefix,
                          clientClassName=mode.clientClassName,
                          matchAny=matchAny,
                          matchName=args[ 'NAME' ] )

def tftpServerOption150Is( config, args ):
   servers = args.get( 'SERVER_ADDR', [] )
   config.tftpServerOption150.clear()
   for i, _ in enumerate( servers ):
      config.tftpServerOption150[ i ] = servers[ i ]

def tftpServerOption66Is( config, args ):
   config.tftpServerOption66 = args.get( 'NAME', config.tftpServerDefault )

def privateOptionDelByKey( config, optionKey ):
   del config.stringPrivateOption[ optionKey ]
   del config.ipAddrPrivateOption[ optionKey ]
   del config.keyToTypeMap[ optionKey ]

def privateOptionIs( privateOptionConfig, optionType, optionFunc, args ):
   code = args[ 'CODE' ]
   optionKey = Tac.Value( 'DhcpServer::PrivateOptionKey', code )

   if optionKey in privateOptionConfig.keyToTypeMap:
      privateOptionDelByKey( privateOptionConfig, optionKey )

   keyToTypeMap = privateOptionConfig.newKeyToTypeMap( optionKey )
   keyToTypeMap.type = optionType

   # Set private option data
   privateOption = optionFunc( optionType, optionKey )

   # Set always-send flag
   privateOption.alwaysSend = 'always-send' in args
   return privateOption

def privateOptionDel( privateOptionConfig, args ):
   optionKey = Tac.Value( 'DhcpServer::PrivateOptionKey', args[ 'CODE' ] )
   if privateOptionConfig:
      privateOptionDelByKey( privateOptionConfig, optionKey )

def stringPrivateOptionIs( privateOptionConfig, args ):
   optionType = OptionType.optionString
   privateOption = privateOptionIs(
                           privateOptionConfig, optionType,
                           privateOptionConfig.newStringPrivateOption, args )
   privateOption.data = args[ 'STRING' ]

def ipAddrPrivateOptionIs( privateOptionConfig, args ):
   optionType = args[ 'ipAddrType' ]
   privateOption = privateOptionIs(
                        privateOptionConfig, optionType,
                        privateOptionConfig.newIpAddrPrivateOption, args )
   ipAddresses = args[ 'IP' ]
   for i, _ in enumerate( ipAddresses ):
      privateOption.data[ i ] = ipAddresses[ i ]
   privateOption.isArray = len( ipAddresses ) > 1

def arbitraryOptionIs( arbitraryOptionConfig, optionType, optionFunc, args ):
   code = args[ 'CODE' ]
   optionKey = Tac.Value( 'DhcpServer::ArbitraryOptionKey', code )
   if optionKey in arbitraryOptionConfig.keyToTypeMap:
      del arbitraryOptionConfig.stringArbitraryOption[ optionKey ]
      del arbitraryOptionConfig.fqdnArbitraryOption[ optionKey ]
      del arbitraryOptionConfig.ipAddrArbitraryOption[ optionKey ]
      del arbitraryOptionConfig.hexArbitraryOption[ optionKey ]
      del arbitraryOptionConfig.keyToTypeMap[ optionKey ]
   keyToTypeMap = arbitraryOptionConfig.newKeyToTypeMap( optionKey )
   keyToTypeMap.type = optionType

   # Set arbitrary option data
   arbitraryOption = optionFunc( optionType, optionKey )

   # Set always-send flag
   arbitraryOption.alwaysSend = 'always-send' in args
   return arbitraryOption

def hexArbitraryOptionIs( arbitraryOptionConfig, args ):
   optionType = OptionType.optionHex
   arbitraryOption = arbitraryOptionIs(
                           arbitraryOptionConfig, optionType,
                           arbitraryOptionConfig.newHexArbitraryOption, args )
   arbitraryOption.data = args[ 'HEX' ]

def hexArbitraryOptionDel( arbitraryOptionConfig, args ):
   optionKey = Tac.Value( 'DhcpServer::ArbitraryOptionKey', args[ 'CODE' ] )
   if arbitraryOptionConfig:
      del arbitraryOptionConfig.hexArbitraryOption[ optionKey ]
      del arbitraryOptionConfig.keyToTypeMap[ optionKey ]

def stringArbitraryOptionIs( arbitraryOptionConfig, args ):
   optionType = OptionType.optionString
   arbitraryOption = arbitraryOptionIs(
                           arbitraryOptionConfig, optionType,
                           arbitraryOptionConfig.newStringArbitraryOption, args )
   arbitraryOption.data = args[ 'STRING' ]

def stringArbitraryOptionDel( arbitraryOptionConfig, args ):
   optionKey = Tac.Value( 'DhcpServer::ArbitraryOptionKey', args[ 'CODE' ] )
   if arbitraryOptionConfig:
      del arbitraryOptionConfig.stringArbitraryOption[ optionKey ]
      del arbitraryOptionConfig.keyToTypeMap[ optionKey ]

def fqdnArbitraryOptionIs( arbitraryOptionConfig, args ):
   optionType = OptionType.optionFqdn
   arbitraryOption = arbitraryOptionIs(
                           arbitraryOptionConfig, optionType,
                           arbitraryOptionConfig.newFqdnArbitraryOption, args )
   arbitraryOption.data = args[ 'FQDN' ]

def fqdnArbitraryOptionDel( arbitraryOptionConfig, args ):
   optionKey = Tac.Value( 'DhcpServer::ArbitraryOptionKey', args[ 'CODE' ] )
   if arbitraryOptionConfig:
      del arbitraryOptionConfig.fqdnArbitraryOption[ optionKey ]
      del arbitraryOptionConfig.keyToTypeMap[ optionKey ]

def ipAddrArbitraryOptionIs( arbitraryOptionConfig, args ):
   optionType = args[ 'ipAddrType' ]
   arbitraryOption = arbitraryOptionIs(
                           arbitraryOptionConfig, optionType,
                           arbitraryOptionConfig.newIpAddrArbitraryOption, args )
   ipAddresses = args[ 'IP' ]
   arbitraryOption.data.update( enumerate( ipAddresses ) )
   arbitraryOption.isArray = len( ipAddresses ) > 1

def ipAddrArbitraryOptionDel( arbitraryOptionConfig, args ):
   optionKey = Tac.Value( 'DhcpServer::ArbitraryOptionKey', args[ 'CODE' ] )
   if arbitraryOptionConfig:
      del arbitraryOptionConfig.ipAddrArbitraryOption[ optionKey ]
      del arbitraryOptionConfig.keyToTypeMap[ optionKey ]

def tftpBootFileNameIs( mode, config, args ):
   if mode.session.hasError():
      return
   config.tftpBootFileName = args.get( 'FILENAME',
                                       config.tftpBootFileNameDefault )

def defaultGatewayIs( config, args ):
   config.defaultGateway = args[ 'IP_ADDR' ]

def clearDhcpServerConfig( config ):
   bootFileDefault = config.tftpBootFileNameDefault

   config.dhcpServerMode = config.dhcpServerModeDefault
   config.disabled = config.disabledDefault

   config.leaseTimeIpv4 = config.leaseTimeIpv4Default
   config.domainNameIpv4 = ''
   config.dnsServersIpv4.clear()
   config.subnetConfigIpv4.clear()
   config.debugLogPath = config.debugLogPathDefault
   config.debugLogUrl = config.debugLogPathDefault
   config.echoClientIdIpv4 = config.echoClientIdIpv4Default

   config.tftpServerOption66Ipv4 = config.tftpServerDefault
   config.tftpBootFileNameIpv4 = bootFileDefault
   config.tftpBootFileNameIpv6 = bootFileDefault
   config.tftpServerOption150Ipv4.clear()

   config.leaseTimeIpv6 = config.leaseTimeIpv6Default
   config.domainNameIpv6 = ''
   config.dnsServersIpv6.clear()
   config.subnetConfigIpv6.clear()
   config.vendorOptionIpv4.clear()
   config.clientClassConfigIpv4.clear()
   config.clientClassConfigIpv6.clear()
   config.globalPrivateOptionIpv4 = None
   config.globalPrivateOptionIpv6 = None
   config.globalArbitraryOptionIpv4 = None

def doEnterDhcpServerVrfMode( mode, args ):
   vrfName = args[ 'VRFNAME' ]
   dhcpServerVrfConfig.vrfConfig.newMember( vrfName )
   childMode = mode.childMode( DhcpServerMode, vrf=vrfName )
   childMode.dhcpServerConfig = dhcpServerVrfConfig.vrfConfig[ vrfName ]
   childMode.dhcpServerConfig.dhcpServerMode = True
   mode.session_.gotoChildMode( childMode )

def noEnterDhcpServerVrfMode( mode, args ):
   vrfName = args[ 'VRFNAME' ]
   vrfConfigDir = dhcpServerVrfConfig.vrfConfig
   vrfConfig = vrfConfigDir.get( vrfName )
   if vrfConfig and (
         vrfConfig.interfacesIpv4 or vrfConfig.interfacesIpv6 ):
      clearDhcpServerConfig( vrfConfig )
   else:
      if vrfConfig:
         clearDhcpServerConfig( vrfConfig )
      del vrfConfigDir[ vrfName ]

def doEnterDhcpServerVendorOptionMode( mode, args ):
   vendorId = args[ 'VENDOR_ID' ]
   vendorMode = args[ 'vendorMode' ]
   childMode = mode.childMode( vendorMode, vendorId=vendorId )
   mode.session_.gotoChildMode( childMode )

def noEnterDhcpServerVendorOptionMode( mode, args ):
   vendorId = args[ 'VENDOR_ID' ]
   if args[ 'af' ] == 'ipv4':
      del mode.dhcpServerConfig.vendorOptionIpv4[ vendorId ]

def setDhcpServerVendorOptionSubOption( mode, args ):
   code = args[ 'NUMBER' ]
   subOption = mode.vendorOption.subOptionConfig.newMember( code )
   if 'string' in args:
      subOption.type = vendorSubOptionType.string
      subOption.dataString = args[ 'STRING' ]
   else:
      subOption.type = vendorSubOptionType.ipAddress
      ipAddresses = args[ 'IP_ADDR' ]
      subOption.isArray = 'array' in args
      subOption.dataIpAddress.clear()
      for i, ip in enumerate( ipAddresses ):
         subOption.dataIpAddress[ i ] = ip

def noDhcpServerVendorOptionSubOption( mode, args ):
   code = args[ 'NUMBER' ]
   del mode.vendorOption.subOptionConfig[ code ]

def setDhcpServerDisabled( mode, args ):
   mode.dhcpServerConfig.disabled = True

def noDhcpServerDisabled( mode, args ):
   mode.dhcpServerConfig.disabled = mode.dhcpServerConfig.disabledDefault

def setDhcpServerGlobalLeaseTime( mode, args ):
   if mode.session.hasError():
      return
   leaseTime = args[ 'TIME' ]
   ipv4 = 'ipv4' in args
   if ipv4:
      mode.dhcpServerConfig.leaseTimeIpv4 = leaseTime
   else:
      mode.dhcpServerConfig.leaseTimeIpv6 = leaseTime

def noDhcpServerGlobalLeaseTime( mode, args ):
   ipv4 = 'ipv4' in args
   if ipv4:
      mode.dhcpServerConfig.leaseTimeIpv4 = (
            mode.dhcpServerConfig.leaseTimeIpv4Default )
   else:
      mode.dhcpServerConfig.leaseTimeIpv6 = (
            mode.dhcpServerConfig.leaseTimeIpv6Default )

def setDhcpServerSubnetLeaseTime( mode, args ):
   leaseTimeIs( mode, mode.subnetConfig, args )

def noDhcpServerSubnetLeaseTime( mode, args ):
   leaseTimeDel( mode.subnetConfig )

def setDhcpServerClientClassLeaseTime( mode, args ):
   leaseTimeIs( mode, mode.clientClassConfig, args )

def noDhcpServerClientClassLeaseTime( mode, args ):
   leaseTimeDel( mode.clientClassConfig )

def setDhcpServerSubnetClientClassLeaseTime( mode, args ):
   leaseTimeIs( mode, mode.clientClassConfig, args )

def noDhcpServerSubnetClientClassLeaseTime( mode, args ):
   leaseTimeDel( mode.clientClassConfig )

def setDhcpServerRangeClientClassLeaseTime( mode, args ):
   leaseTimeIs( mode, mode.clientClassConfig, args )

def noDhcpServerRangeClientClassLeaseTime( mode, args ):
   leaseTimeDel( mode.clientClassConfig )

def setDhcpServerDnsDomainName( mode, args ):
   domain = args[ 'NAME' ]
   if not validateHostname( domain ):
      mode.addError( f"Invalid domain name {domain}" )
      return
   ipv4 = 'ipv4' in args
   if ipv4:
      mode.dhcpServerConfig.domainNameIpv4 = domain
   else:
      mode.dhcpServerConfig.domainNameIpv6 = domain

def noDhcpServerDnsDomainName( mode, args ):
   ipv4 = 'ipv4' in args
   if ipv4:
      mode.dhcpServerConfig.domainNameIpv4 = ""
   else:
      mode.dhcpServerConfig.domainNameIpv6 = ""

def setDhcpServerClientClassDnsDomainName( mode, args ):
   domain = args[ 'NAME' ]
   if not validateHostname( domain ):
      mode.addError( f"Invalid domain name {domain}" )
      return
   mode.clientClassConfig.domainName = domain

def noDhcpServerClientClassDnsDomainName( mode, args ):
   mode.clientClassConfig.domainName = ""

def setDhcpServerDnsServer( mode, args ):
   ipv4 = 'ipv4' in args
   if ipv4:
      servers = args[ 'SERVERS' ]
      mode.dhcpServerConfig.dnsServersIpv4.clear()
      for i in range( min( _maxDnsServers, len( servers ) ) ):
         mode.dhcpServerConfig.dnsServersIpv4[ i ] = Arnet.IpAddr( servers[ i ] )
   else:
      servers = args[ 'SERVERS6' ]
      mode.dhcpServerConfig.dnsServersIpv6.clear()
      for i in range( min( _maxDnsServers, len( servers ) ) ):
         mode.dhcpServerConfig.dnsServersIpv6[ i ] = Arnet.Ip6Addr( servers[ i ] )

def noDhcpServerDnsServer( mode, args ):
   ipv4 = 'ipv4' in args
   if ipv4:
      mode.dhcpServerConfig.dnsServersIpv4.clear()
   else:
      mode.dhcpServerConfig.dnsServersIpv6.clear()

def setDhcpServerDebugLog( mode, args ):
   debugFilePath = args[ 'FILE' ]
   FileCliUtil.checkUrl( debugFilePath )
   if debugFilePath.isdir():
      mode.addError( "Cannot specify a directory" )
      return
   mode.dhcpServerConfig.debugLogPath = debugFilePath.localFilename()
   # localFilename: /tmp/localfile
   # url: file:/tmp/localfile
   # Save the url passed in because when testing a save plugin with which uses
   # Url.filenameToUrl(), it doesn't convert properly back to file:/
   mode.dhcpServerConfig.debugLogUrl = debugFilePath.url

def noDhcpServerDebugLog( mode, args ):
   mode.dhcpServerConfig.debugLogPath = mode.dhcpServerConfig.debugLogPathDefault
   mode.dhcpServerConfig.debugLogUrl = mode.dhcpServerConfig.debugLogPathDefault

def disableEchoClientId( mode, args ):
   mode.dhcpServerConfig.echoClientIdIpv4 = False

def noDisableEchoClientId( mode, args ):
   mode.dhcpServerConfig.echoClientIdIpv4 = True

def setDhcpServerTftpOption66( mode, args ):
   mode.dhcpServerConfig.tftpServerOption66Ipv4 = args.get(
         'NAME', mode.dhcpServerConfig.tftpServerDefault )

def setDhcpServerTftpOption150( mode, args ):
   servers = args.get( 'SERVER_ADDR' )
   mode.dhcpServerConfig.tftpServerOption150Ipv4.clear()
   for i, _ in enumerate( servers ):
      mode.dhcpServerConfig.tftpServerOption150Ipv4[ i ] = servers[ i ]

def noDhcpServerTftpOption150( mode, args ):
   mode.dhcpServerConfig.tftpServerOption150Ipv4.clear()

def setDhcpServerTftpBootFile( mode, args ):
   if mode.session.hasError():
      return
   bootFile = args.get( 'FILENAME',
                        mode.dhcpServerConfig.tftpBootFileNameDefault )
   ipv4 = 'ipv4' in args
   if ipv4:
      mode.dhcpServerConfig.tftpBootFileNameIpv4 = bootFile
   else:
      mode.dhcpServerConfig.tftpBootFileNameIpv6 = bootFile

def setDhcpServerSubnetTftpOption66( mode, args ):
   tftpServerOption66Is( mode.subnetConfig, args )

def setDhcpServerClientClassTftpOption66( mode, args ):
   tftpServerOption66Is( mode.clientClassConfig, args )

def setDhcpServerSubnetTftpOption150( mode, args ):
   tftpServerOption150Is( mode.subnetConfig, args )

def setDhcpServerClientClassTftpOption150( mode, args ):
   tftpServerOption150Is( mode.clientClassConfig,
                                                      args )

def setDhcpServerSubnetTftpBootFile( mode, args ):
   tftpBootFileNameIs( mode, mode.subnetConfig, args )

def setDhcpServerClientClassTftpBootFile( mode, args ):
   tftpBootFileNameIs( mode, mode.clientClassConfig, args )

def doEnterDhcpServerSubnetMode( mode, args ):
   prefix = args[ 'SUBNET' ]
   if not args[ 'prefixCheck' ]( mode, prefix ):
      return
   childMode = mode.childMode( args[ 'subnetMode' ], param=prefix )
   mode.session_.gotoChildMode( childMode )

def noEnterDhcpServerSubnetMode( mode, args ):
   prefix = args[ 'SUBNET' ]
   if not args[ 'prefixCheck' ]( mode, prefix ):
      return
   if args[ 'af' ] == 'ipv4':
      prefix = Arnet.Prefix( prefix )
      del mode.dhcpServerConfig.subnetConfigIpv4[ prefix ]
   else:
      prefix = Arnet.Ip6Prefix( prefix )
      del mode.dhcpServerConfig.subnetConfigIpv6[ prefix ]

def setDhcpServerDnsServerBase( mode, args ):
   mode.dnsServersIs( args )

def noDhcpServerDnsServerBase( mode, args ):
   mode.dnsServersDel( args )

def setDhcpServerSubnetV4DefaultGateway( mode, args ):
   defaultGatewayIs( mode.subnetConfig, args )

def setDhcpServerClientClassV4DefaultGateway( mode, args ):
   defaultGatewayIs( mode.clientClassConfig, args )

def doEnterDhcpServerClientClassBaseMode( mode, args ):
   clientClassName = args[ 'CLIENT_CLASS' ]
   clientClassMode = args[ 'clientClassAfMode' ]
   childMode = mode.childMode( clientClassMode,
                               clientClassName=clientClassName )
   mode.session_.gotoChildMode( childMode )

def noEnterDhcpServerClientClassBaseMode( mode, args ):
   clientClassName = args[ 'CLIENT_CLASS' ]
   clientClassConfig = args[ 'clientClassConfig' ]
   del clientClassConfig[ clientClassName ]

def clearClientClassOptions( clientClassConfig, serverConfig, af ):
   clientClassConfig.domainName = ''
   if clientClassConfig.privateOptionConfig:
      clientClassConfig.privateOptionConfig.stringPrivateOption.clear()
      clientClassConfig.privateOptionConfig.ipAddrPrivateOption.clear()
      clientClassConfig.privateOptionConfig = None
   if clientClassConfig.arbitraryOptionConfig:
      clientClassConfig.arbitraryOptionConfig.stringArbitraryOption.clear()
      clientClassConfig.arbitraryOptionConfig.fqdnArbitraryOption.clear()
      clientClassConfig.arbitraryOptionConfig.ipAddrArbitraryOption.clear()
      clientClassConfig.arbitraryOptionConfig.hexArbitraryOption.clear()
      clientClassConfig.arbitraryOptionConfig = None
   clientClassConfig.dnsServers.clear()
   if af == 'ipv4':
      clientClassConfig.leaseTime = serverConfig.leaseTimeIpv4Default
      clientClassConfig.defaultGateway = Arnet.IpAddr( '0.0.0.0' )
      clientClassConfig.tftpServerOption66 = clientClassConfig.tftpServerDefault
      clientClassConfig.tftpBootFileName = clientClassConfig.tftpBootFileNameDefault
   else:
      clientClassConfig.leaseTime = serverConfig.leaseTimeIpv6Default

def clearClientClassMatchConfig( clientClassConfig ):
   clientClassConfig.matchCriteriaModeConfig = None

def doEnterDhcpServerAssignModeBase( mode, args, getOrCreateChildModeFunc ):
   childMode = getOrCreateChildModeFunc( mode, args )
   mode.session_.gotoChildMode( childMode )

def noEnterDhcpServerAssignModeBase( mode, args ):
   clearClientClassOptions( mode.clientClassConfig, mode.dhcpServerConfig,
                            args[ 'af' ] )

def doEnterDhcpServerClientClassAssignMode( mode, args ):
   doEnterDhcpServerAssignModeBase(
         mode, args, getOrCreateChildClientClassAssignMode )

def noEnterDhcpServerClientClassAssignMode( mode, args ):
   noEnterDhcpServerAssignModeBase( mode, args )

def doEnterDhcpServerSubnetClientClassAssignMode( mode, args ):
   doEnterDhcpServerAssignModeBase(
         mode, args, getOrCreateChildSubnetClientClassAssignMode )

def noEnterDhcpServerSubnetClientClassAssignMode( mode, args ):
   noEnterDhcpServerAssignModeBase( mode, args )

def doEnterDhcpServerRangeClientClassAssignMode( mode, args ):
   doEnterDhcpServerAssignModeBase(
         mode, args, getOrCreateChildRangeClientClassAssignMode )

def noEnterDhcpServerRangeClientClassAssignMode( mode, args ):
   noEnterDhcpServerAssignModeBase( mode, args )
   mode.clientClassConfig.assignedIp = mode.clientClassConfig.ipAddrDefault

def doEnterDhcpServerMatchModeBase( mode, args, getOrCreateChildModeFunc ):
   matchAny = 'any' in args
   childMode = getOrCreateChildModeFunc( mode, args, matchAny )
   if not childMode.clientClassConfig.matchCriteriaModeConfig:
      childMode.clientClassConfig.matchCriteriaModeConfig = ()
   childMode.matchCriteria = childMode.clientClassConfig.matchCriteriaModeConfig
   childMode.matchCriteria.matchAny = matchAny
   mode.session_.gotoChildMode( childMode )

def noEnterDhcpServerMatchModeBase( mode, args, getOrCreateChildModeFunc ):
   clientClassConfig = mode.clientClassConfig
   matchCriteriaModeConfig = clientClassConfig.matchCriteriaModeConfig
   if matchCriteriaModeConfig:
      matchAny = 'any' in args
      childMode = getOrCreateChildModeFunc( mode, args, matchAny )
      childMode.removeComment()

      childMode.matchCriteria = None
      clearClientClassMatchConfig( clientClassConfig )

def doEnterDhcpServerClientClassMatchMode( mode, args ):
   doEnterDhcpServerMatchModeBase(
         mode, args, getOrCreateChildClientClassMatchMode )

def noEnterDhcpServerClientClassMatchMode( mode, args ):
   noEnterDhcpServerMatchModeBase(
         mode, args, getOrCreateChildClientClassMatchMode )

def doEnterDhcpServerSubnetClientClassMatchMode( mode, args ):
   doEnterDhcpServerMatchModeBase(
         mode, args, getOrCreateChildSubnetClientClassMatchMode )

def noEnterDhcpServerSubnetClientClassMatchMode( mode, args ):
   noEnterDhcpServerMatchModeBase(
         mode, args, getOrCreateChildSubnetClientClassMatchMode )

def doEnterDhcpServerRangeClientClassMatchMode( mode, args ):
   doEnterDhcpServerMatchModeBase(
         mode, args, getOrCreateChildRangeClientClassMatchMode )

def noEnterDhcpServerRangeClientClassMatchMode( mode, args ):
   noEnterDhcpServerMatchModeBase(
         mode, args, getOrCreateChildRangeClientClassMatchMode )

def doEnterDhcpServerClientClassNestedMatchMode( mode, args ):
   doEnterDhcpServerNestedMatchModeBaseHandler(
         mode, args, getOrCreateChildClientClassNestedMatchMode )

def noEnterDhcpServerClientClassNestedMatchMode( mode, args ):
   noEnterDhcpServerNestedMatchModeBaseHandler(
         mode, args, getOrCreateChildClientClassNestedMatchMode )

def doEnterDhcpServerSubnetClientClassNestedMatchMode( mode, args ):
   doEnterDhcpServerNestedMatchModeBaseHandler(
         mode, args, getOrCreateChildSubnetClientClassNestedMatchMode )

def noEnterDhcpServerSubnetClientClassNestedMatchMode( mode, args ):
   noEnterDhcpServerNestedMatchModeBaseHandler(
         mode, args, getOrCreateChildSubnetClientClassNestedMatchMode )

def doEnterDhcpServerRangeClientClassNestedMatchMode( mode, args ):
   doEnterDhcpServerNestedMatchModeBaseHandler(
         mode, args, getOrCreateChildRangeClientClassNestedMatchMode )

def noEnterDhcpServerRangeClientClassNestedMatchMode( mode, args ):
   noEnterDhcpServerNestedMatchModeBaseHandler(
         mode, args, getOrCreateChildRangeClientClassNestedMatchMode )

def setDhcpServerClientClassMatchVendorId( mode, args ):
   mode.matchCriteria.vendorId[ args[ 'VENDOR_ID' ] ] = True

def noDhcpServerClientClassMatchVendorId( mode, args ):
   del mode.matchCriteria.vendorId[ args[ 'VENDOR_ID' ] ]

def tacVendorClassFromArgs( args ):
   enterpriseId = args.get( 'ENTERPRISE_ID', 0 )
   anyEnterpriseId = not enterpriseId
   return tacVendorClassOption( enterpriseId, anyEnterpriseId,
                                args[ 'QUOTED_STRING' ] )

def setDhcpServerClientClassMatchVendorClass( mode, args ):
   vendorClass = tacVendorClassFromArgs( args )
   mode.matchCriteria.vendorClass[ vendorClass ] = True

def noDhcpServerClientClassMatchVendorClass( mode, args ):
   vendorClass = tacVendorClassFromArgs( args )
   del mode.matchCriteria.vendorClass[ vendorClass ]

def setDhcpServerClientClassMatchMacAddr( mode, args ):
   mode.matchCriteria.hostMacAddress[ args[ 'MAC_ADDR' ] ] = True

def noDhcpServerClientClassMatchMacAddr( mode, args ):
   del mode.matchCriteria.hostMacAddress[ args[ 'MAC_ADDR' ] ]

def setDhcpServerClientClassMatchDuid( mode, args ):
   mode.matchCriteria.duid[ args[ 'DUID' ] ] = True

def noDhcpServerClientClassMatchDuid( mode, args ):
   del mode.matchCriteria.duid[ args[ 'DUID' ] ]

def setDhcpServerClientClassMatchAristaInfoOption( mode, args ):
   doDhcpServerClientClassMatchAristaL2InfoBaseHandler( mode, args )

def noDhcpServerClientClassMatchAristaInfoOption( mode, args ):
   noDhcpServerClientClassMatchAristaL2InfoBaseHandler( mode, args )

def setDhcpServerClientClassMatchAristaRemoteId( mode, args ):
   doDhcpServerClientClassMatchAristaL2InfoBaseHandler( mode, args )

def noDhcpServerClientClassMatchAristaRemoteId( mode, args ):
   noDhcpServerClientClassMatchAristaL2InfoBaseHandler( mode, args )

def setDhcpServerClientClassMatchInfoOptionHexOrStr( mode, args ):
   mode.matchCriteria.circuitIdRemoteIdHexOrStr[ args[ 'infoOpt' ] ] = True

def noDhcpServerClientClassMatchInfoOptionHexOrStr( mode, args ):
   del mode.matchCriteria.circuitIdRemoteIdHexOrStr[ args[ 'infoOpt' ] ]

def setDhcpServerClientClassMatchRemoteIdHexOrStr( mode, args ):
   mode.matchCriteria.remoteIdHexOrStr[ args[ 'remoteId' ] ] = True

def noDhcpServerClientClassMatchRemoteIdHexOrStr( mode, args ):
   del mode.matchCriteria.remoteIdHexOrStr[ args[ 'remoteId' ] ]

def setDhcpServerGlobalV4StringPrivateOption( mode, args ):
   if not mode.dhcpServerConfig.globalPrivateOptionIpv4:
      mode.dhcpServerConfig.globalPrivateOptionIpv4 = ()
   stringPrivateOptionIs( mode.dhcpServerConfig.globalPrivateOptionIpv4, args )

def noDhcpServerGlobalV4StringPrivateOption( mode, args ):
   privateOptionDel( mode.dhcpServerConfig.globalPrivateOptionIpv4, args )

def setDhcpServerGlobalV6StringPrivateOption( mode, args ):
   if not mode.dhcpServerConfig.globalPrivateOptionIpv6:
      mode.dhcpServerConfig.globalPrivateOptionIpv6 = ()
   stringPrivateOptionIs( mode.dhcpServerConfig.globalPrivateOptionIpv6, args )

def noDhcpServerGlobalV6StringPrivateOption( mode, args ):
   privateOptionDel( mode.dhcpServerConfig.globalPrivateOptionIpv6, args )

def setDhcpServerClientClassStringPrivateOption( mode, args ):
   if not mode.clientClassConfig.privateOptionConfig:
      mode.clientClassConfig.privateOptionConfig = ()
   stringPrivateOptionIs( mode.clientClassConfig.privateOptionConfig, args )

def noDhcpServerClientClassStringPrivateOption( mode, args ):
   privateOptionDel( mode.clientClassConfig.privateOptionConfig, args )

def setDhcpServerGlobalIpv4AddrPrivateOption( mode, args ):
   if not mode.dhcpServerConfig.globalPrivateOptionIpv4:
      mode.dhcpServerConfig.globalPrivateOptionIpv4 = ()
   ipAddrPrivateOptionIs( mode.dhcpServerConfig.globalPrivateOptionIpv4, args )

def setDhcpServerGlobalIpv6AddrPrivateOption( mode, args ):
   if not mode.dhcpServerConfig.globalPrivateOptionIpv6:
      mode.dhcpServerConfig.globalPrivateOptionIpv6 = ()
   ipAddrPrivateOptionIs( mode.dhcpServerConfig.globalPrivateOptionIpv6, args )

def setDhcpServerClientClassIpv4AddrPrivateOption( mode, args ):
   if not mode.clientClassConfig.privateOptionConfig:
      mode.clientClassConfig.privateOptionConfig = ()
   ipAddrPrivateOptionIs( mode.clientClassConfig.privateOptionConfig, args )

def setDhcpServerClientClassIpv6AddrPrivateOption( mode, args ):
   if not mode.clientClassConfig.privateOptionConfig:
      mode.clientClassConfig.privateOptionConfig = ()
   ipAddrPrivateOptionIs( mode.clientClassConfig.privateOptionConfig, args )

def doEnterDhcpServerRangeModeBase( mode, args ):
   # TODO (BUG573597): When toggle is on, SuperServer code should look at
   # subnetConfig.rangeConfig.keys() instead of subnetConfig.ranges
   mode.subnetConfig.ranges[ args[ 'tacRange' ] ] = True
   rangeConfig = mode.subnetConfig.newRangeConfig( args[ 'tacRange' ] )
   childMode = mode.childMode( args[ 'rangeMode' ],
         param=( mode.subnetConfig, args[ 'tacRange' ], mode.vrf ) )
   childMode.rangeConfig = rangeConfig
   mode.session_.gotoChildMode( childMode )

def noEnterDhcpServerRangeModeBase( mode, args ):
   del mode.subnetConfig.ranges[ args[ 'tacRange' ] ]
   del mode.subnetConfig.rangeConfig[ args[ 'tacRange' ] ]

def setDhcpServerRangeClientClassAssignment( mode, args ):
   setAssignedClientClass( mode.rangeConfig, args )

def noDhcpServerRangeClientClassAssignment( mode, args ):
   clearAssignedClientClass( mode.rangeConfig )

def setDhcpServerSubnetClientClassAssignment( mode, args ):
   setAssignedClientClass( mode.subnetConfig, args )

def noDhcpServerSubnetClientClassAssignment( mode, args ):
   clearAssignedClientClass( mode.subnetConfig )

def setDhcpServerSubnetRangeBase( mode, args ):
   ipv4 = isinstance( mode, DhcpServerSubnetV4Mode )
   ipAddrFn = Arnet.IpAddr if ipv4 else Arnet.Ip6Addr
   subnetRange = "DhcpServer::Range" if ipv4 else "DhcpServer::Range6"
   start = ipAddrFn( args[ 'START' ] )
   end = ipAddrFn( args[ 'END' ] )
   if start > end:
      mode.addError( 'Start range cannot be larger than end of range' )
      return
   subnetRange = Tac.Value( subnetRange, start, end )
   mode.subnetConfig.ranges[ subnetRange ] = True

def noDhcpServerSubnetRangeBase( mode, args ):
   ipv4 = isinstance( mode, DhcpServerSubnetV4Mode )
   ipAddrFn = Arnet.IpAddr if ipv4 else Arnet.Ip6Addr
   subnetRange = "DhcpServer::Range" if ipv4 else "DhcpServer::Range6"
   start = ipAddrFn( args[ 'START' ] )
   end = ipAddrFn( args[ 'END' ] )
   if start > end:
      mode.addError( 'Start range cannot be larger than end of range' )
      return
   subnetRange = Tac.Value( subnetRange, start, end )
   del mode.subnetConfig.ranges[ subnetRange ]

def setDhcpServerSubnetName( mode, args ):
   name = args[ 'NAME' ]
   mode.subnetConfig.subnetName = name

def noDhcpServerSubnetName( mode, args ):
   mode.subnetConfig.subnetName = ""

def getInterfaceVrf( mode, args ):
   ipIntfConfig = ipConfig.ipIntfConfig.get( mode.intf.name )
   vrf = ( VrfCli.DEFAULT_VRF if ipIntfConfig is None or ipIntfConfig.vrf == ""
           else ipIntfConfig.vrf )
   return vrf

def setInterfaceEnterDhcpServerEnable( mode, args ):
   vrf = getInterfaceVrf( mode, args )
   if vrf not in dhcpServerVrfConfig.vrfConfig:
      dhcpServerVrfConfig.vrfConfig.newMember( vrf )
   config = dhcpServerVrfConfig.vrfConfig[ vrf ]
   ipv4 = 'ipv4' in args
   interfaces = config.interfacesIpv4 if ipv4 else \
                config.interfacesIpv6
   interfaces[ mode.intf.name ] = True

def noInterfaceEnterDhcpServerEnable( mode, args ):
   vrf = getInterfaceVrf( mode, args )
   ipv4 = 'ipv4' in args
   if vrf in dhcpServerVrfConfig.vrfConfig:
      config = dhcpServerVrfConfig.vrfConfig[ vrf ]
      interfaces = config.interfacesIpv4 if ipv4 else \
                   config.interfacesIpv6
      del interfaces[ mode.intf.name ]

# Clear lease functions
def clearAllLeases( config, status, args ):
   if 'ipv4' in args:
      config.clearIdIpv4 = max( status.lastClearIdIpv4,
                                config.clearIdIpv4 ) + 1
   else:
      config.clearIdIpv6 = max( status.lastClearIdIpv6,
                                config.clearIdIpv6 ) + 1

def afClearLeaseIdInc( config, args, leaseIp ):
   if args[ 'af' ] == 'ipv4':
      clearId = config.clearLeaseIpv4.clearId
      config.clearLeaseIpv4 = Tac.Value( 'DhcpServer::ClearLeaseIpv4',
                                         leaseIp, clearId + 1 )
   else:
      clearId = config.clearLeaseIpv6.clearId
      config.clearLeaseIpv6 = Tac.Value( 'DhcpServer::ClearLeaseIpv6',
                                         leaseIp, clearId + 1 )

# Helper function to get vrfs
def getVrfs( mode, args ):
   vrf = args.get( 'VRFNAME' ) or VrfCli.vrfMap.lookupCliModeVrf( mode, None )
   if vrf == 'all':
      vrfs = dhcpServerVrfConfig.vrfConfig
   else:
      vrfs = [ v for v in dhcpServerVrfConfig.vrfConfig if v == vrf ]
   return vrfs

def setDhcpServerClearLeasesVrf( mode, args ):
   vrfs = getVrfs( mode, args )
   for vrf in vrfs:
      config = dhcpServerVrfConfig.vrfConfig[ vrf ]
      status = dhcpServerVrfStatus.vrfStatus[ vrf ]
      clearAllLeases( config, status, args )

def setDhcpServerClearIpv4LeaseByAddrVrf( mode, args ):
   leaseIp = args[ 'leaseAddr' ]
   if Arnet.IpAddr( leaseIp ) == Arnet.IpAddr( '0.0.0.0' ):
      return
   vrfs = getVrfs( mode, args )
   for vrf in vrfs:
      config = dhcpServerVrfConfig.vrfConfig[ vrf ]
      afClearLeaseIdInc( config, args, leaseIp )

def setDhcpServerClearIpv6LeaseByAddrVrf( mode, args ):
   leaseIp = args[ 'leaseAddr' ]
   if Arnet.Ip6Addr( leaseIp ) == Arnet.Ip6Addr( '::' ):
      return
   vrfs = getVrfs( mode, args )
   for vrf in vrfs:
      config = dhcpServerVrfConfig.vrfConfig[ vrf ]
      afClearLeaseIdInc( config, args, leaseIp )

def doEnterDhcpServerSubnetClientClass( mode, args ):
   nonGlobalClientClassHandler( mode, args, mode.subnetConfig )

def doEnterDhcpServerRangeClientClass( mode, args ):
   nonGlobalClientClassHandler( mode, args, mode.rangeConfig )

def setDhcpServerRangeClientClassIpAddrBase( mode, args, errorString ):
   ipAddrDefault = args[ 'IP_DEFAULT' ]
   ipAddr = args[ 'IP_ADDR' ]
   subnetRange = mode.range

   # verify the ipv4-address is from the correct range
   if ipAddr != ipAddrDefault and not rangeContains( subnetRange, ipAddr ):
      mode.addError( errorString.format( subnetRange.start, subnetRange.end ) )
      return

   config = mode.clientClassConfig
   config.assignedIp = ipAddr

def setDhcpServerRangeClientClassIpv4Addr( mode, args ):
   errorString = "ipv4-address must be a valid IPv4 address from range {} {}"
   setDhcpServerRangeClientClassIpAddrBase( mode, args, errorString )

def setDhcpServerRangeClientClassIpv6Addr( mode, args ):
   errorString = "ipv6-address must be a valid IPv6 address from range {} {}"
   setDhcpServerRangeClientClassIpAddrBase( mode, args, errorString )

def doEnterDhcpServerReservationsMode( mode, args ):
   reservationMode = args[ 'reservationMode' ]
   childMode = mode.childMode( reservationMode,
                               subnetConfig=mode.subnetConfig )
   mode.session_.gotoChildMode( childMode )

def noEnterDhcpServerReservationsMode( mode, args ):
   mode.subnetConfig.reservationsMacAddr.clear()
   mode.subnetConfig.reservationLayer2Intf.clear()
   mode.subnetConfig.reservationCircuitIdRemoteId.clear()

def doEnterDhcpServerReservationsMacAddressMode( mode, args ):
   macAddr = args[ 'MAC_ADDR' ]
   macAddressMode = args[ 'macAddressMode' ]
   mode.subnetConfig.reservationsMacAddr.newMember( macAddr )
   childMode = mode.childMode( macAddressMode, subnetConfig=mode.subnetConfig,
                               macAddr=macAddr )
   mode.session_.gotoChildMode( childMode )

def noEnterDhcpServerReservationsMacAddressMode( mode, args ):
   macAddr = args[ 'MAC_ADDR' ]
   del mode.subnetConfig.reservationsMacAddr[ macAddr ]

def doEnterDhcpServerReservationsAristaBaseMode( mode, args ):
   port = args[ 'PORT' ]
   vlan = args[ 'VLAN' ]
   macAddr = args[ 'MAC' ]
   reservation = tacLayer2Info( port, vlan, macAddr )
   mode.subnetConfig.reservationLayer2Intf.newMember( reservation )
   childMode = mode.childMode( args[ 'mode' ],
         subnetConfig=mode.subnetConfig, port=port, vlan=vlan, macAddr=macAddr )
   mode.session_.gotoChildMode( childMode )

def noEnterDhcpServerReservationsAristaBaseMode( mode, args ):
   port = args[ 'PORT' ]
   vlan = args[ 'VLAN' ]
   macAddr = args[ 'MAC' ]
   reservation = tacLayer2Info( port, vlan, macAddr )
   del mode.subnetConfig.reservationLayer2Intf[ reservation ]

def doEnterDhcpServerReservationsInfoOptionMode( mode, args ):
   mode.subnetConfig.reservationCircuitIdRemoteId.newMember( args[ 'infoOpt' ] )
   childMode = mode.childMode( DhcpServerReservationsInfoOptionMode,
         subnetConfig=mode.subnetConfig, infoOpt=args[ 'infoOpt' ] )
   mode.session_.gotoChildMode( childMode )

def noEnterDhcpServerReservationsInfoOptionMode( mode, args ):
   del mode.subnetConfig.reservationCircuitIdRemoteId[ args[ 'infoOpt' ] ]

def setDhcpServerReservationsIpAddress( mode, args ):
   ipAddrDefault = Arnet.IpAddr( '0.0.0.0' )
   ipAddr = args[ 'IP_ADDR' ]
   prefix = mode.subnetConfig.subnetId
   # verify the ipv4-address is from the current subnet
   if ipAddr != ipAddrDefault and not prefix.contains( ipAddr ):
      errorString = "ipv4-address must be a valid IPv4 address from subnet {}"
      mode.addError( errorString.format( prefix ) )
      return
   mode.reservationConfig.ipAddr = ipAddr

def setDhcpServerReservationsHostname( mode, args ):
   hostname = args.get( 'HOSTNAME', '' )
   mode.reservationConfig.hostname = hostname

def setDhcpServerReservationsIpAddressV6( mode, args ):
   ipAddrDefault = Arnet.Ip6Addr( '::' )
   ipAddr = args[ 'IP_ADDR' ]
   prefix = mode.subnetConfig.subnetId
   # verify the ipv6-address is from the current subnet
   if ipAddr != ipAddrDefault and not prefix.contains( ipAddr ):
      errorString = "ipv6-address must be a valid IPv6 address from subnet {}"
      mode.addError( errorString.format( prefix ) )
      return
   mode.reservationConfig.ipAddr = ipAddr

# -------------------------------------------------------------------------------
# The "[no|default] option ipv4 CODE [always-send] type string data STRING"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
def setDhcpServerGlobalV4StringArbitraryOption( mode, args ):
   if not mode.dhcpServerConfig.globalArbitraryOptionIpv4:
      mode.dhcpServerConfig.globalArbitraryOptionIpv4 = ()
   stringArbitraryOptionIs( mode.dhcpServerConfig.globalArbitraryOptionIpv4, args )

def noDhcpServerGlobalV4StringArbitraryOption( mode, args ):
   stringArbitraryOptionDel( mode.dhcpServerConfig.globalArbitraryOptionIpv4, args )

# -------------------------------------------------------------------------------
# The "[no|default] option ipv4 CODE [always-send] type fqdn data FQDN"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
def setDhcpServerGlobalV4FqdnArbitraryOption( mode, args ):
   if not mode.dhcpServerConfig.globalArbitraryOptionIpv4:
      mode.dhcpServerConfig.globalArbitraryOptionIpv4 = ()
   domain = args[ 'FQDN' ]
   if not validateHostname( domain ):
      mode.addError( f"Invalid domain name {domain}" )
      return
   fqdnArbitraryOptionIs( mode.dhcpServerConfig.globalArbitraryOptionIpv4, args )

def noDhcpServerGlobalV4FqdnArbitraryOption( mode, args ):
   fqdnArbitraryOptionDel( mode.dhcpServerConfig.globalArbitraryOptionIpv4, args )

# -------------------------------------------------------------------------------
# The "[no|default] option ipv4 CODE [always-send] type hex data HEX"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
def setDhcpServerGlobalV4HexArbitraryOption( mode, args ):
   if not mode.dhcpServerConfig.globalArbitraryOptionIpv4:
      mode.dhcpServerConfig.globalArbitraryOptionIpv4 = ()
   hexArbitraryOptionIs( mode.dhcpServerConfig.globalArbitraryOptionIpv4, args )

def noDhcpServerGlobalV4HexArbitraryOption( mode, args ):
   hexArbitraryOptionDel( mode.dhcpServerConfig.globalArbitraryOptionIpv4, args )

# -------------------------------------------------------------------------------
# The "[no|default] option ipv4 CODE [always-send] type ipv4-address data DATA"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
def setDhcpServerGlobalIpv4AddrArbitraryOption( mode, args ):
   if not mode.dhcpServerConfig.globalArbitraryOptionIpv4:
      mode.dhcpServerConfig.globalArbitraryOptionIpv4 = ()
   ipAddrArbitraryOptionIs( mode.dhcpServerConfig.globalArbitraryOptionIpv4, args )

def noDhcpServerGlobalIpv4AddrArbitraryOption( mode, args ):
   ipAddrArbitraryOptionDel( mode.dhcpServerConfig.globalArbitraryOptionIpv4, args )

# -------------------------------------------------------------------------------
# The "[no|default] option ipv6 CODE [always-send] type string data STRING"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
def setDhcpServerGlobalV6StringArbitraryOption( mode, args ):
   if not mode.dhcpServerConfig.globalArbitraryOptionIpv6:
      mode.dhcpServerConfig.globalArbitraryOptionIpv6 = ()
   stringArbitraryOptionIs( mode.dhcpServerConfig.globalArbitraryOptionIpv6, args )

def noDhcpServerGlobalV6StringArbitraryOption( mode, args ):
   stringArbitraryOptionDel( mode.dhcpServerConfig.globalArbitraryOptionIpv6, args )

# -------------------------------------------------------------------------------
# The "[no|default] option ipv6 CODE [always-send] type fqdn data FQDN"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
def setDhcpServerGlobalV6FqdnArbitraryOption( mode, args ):
   if not mode.dhcpServerConfig.globalArbitraryOptionIpv6:
      mode.dhcpServerConfig.globalArbitraryOptionIpv6 = ()
   domain = args[ 'FQDN' ]
   if not validateHostname( domain ):
      mode.addError( f"Invalid domain name {domain}" )
      return
   fqdnArbitraryOptionIs( mode.dhcpServerConfig.globalArbitraryOptionIpv6, args )

def noDhcpServerGlobalV6FqdnArbitraryOption( mode, args ):
   fqdnArbitraryOptionDel( mode.dhcpServerConfig.globalArbitraryOptionIpv6, args )

# -------------------------------------------------------------------------------
# The "[no|default] option ipv6 CODE [always-send] type hex data HEX"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
def setDhcpServerGlobalV6HexArbitraryOption( mode, args ):
   if not mode.dhcpServerConfig.globalArbitraryOptionIpv6:
      mode.dhcpServerConfig.globalArbitraryOptionIpv6 = ()
   hexArbitraryOptionIs( mode.dhcpServerConfig.globalArbitraryOptionIpv6, args )

def noDhcpServerGlobalV6HexArbitraryOption( mode, args ):
   hexArbitraryOptionDel( mode.dhcpServerConfig.globalArbitraryOptionIpv6, args )

# -------------------------------------------------------------------------------
# The "[no|default] option ipv6 CODE [always-send] type ipv6-address data DATA"
# command, in "conf-dhcp-server" mode.
# -------------------------------------------------------------------------------
def setDhcpServerGlobalIpv6AddrArbitraryOption( mode, args ):
   if not mode.dhcpServerConfig.globalArbitraryOptionIpv6:
      mode.dhcpServerConfig.globalArbitraryOptionIpv6 = ()
   ipAddrArbitraryOptionIs( mode.dhcpServerConfig.globalArbitraryOptionIpv6, args )

def noDhcpServerGlobalIpv6AddrArbitraryOption( mode, args ):
   ipAddrArbitraryOptionDel( mode.dhcpServerConfig.globalArbitraryOptionIpv6, args )

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type string data STRING"
# command, in "config-dhcp-cls-v4-assignments" mode.
# -------------------------------------------------------------------------------
def setDhcpServerClientClassStringArbitraryOption( mode, args ):
   if not mode.clientClassConfig.arbitraryOptionConfig:
      mode.clientClassConfig.arbitraryOptionConfig = ()
   stringArbitraryOptionIs( mode.clientClassConfig.arbitraryOptionConfig, args )

def noDhcpServerClientClassStringArbitraryOption( mode, args ):
   stringArbitraryOptionDel( mode.clientClassConfig.arbitraryOptionConfig, args )

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type ipv4-address data IP_ADDRS"
# command, in "config-dhcp-cls-v4-assignments" mode.
# -------------------------------------------------------------------------------
def setDhcpServerClientClassIpAddrArbitraryOption( mode, args ):
   if not mode.clientClassConfig.arbitraryOptionConfig:
      mode.clientClassConfig.arbitraryOptionConfig = ()
   ipAddrArbitraryOptionIs( mode.clientClassConfig.arbitraryOptionConfig, args )

def noDhcpServerClientClassIpAddrArbitraryOption( mode, args ):
   ipAddrArbitraryOptionDel( mode.clientClassConfig.arbitraryOptionConfig, args )

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type fqdn data FQDN"
# command, in "config-dhcp-cls-v4-assignments" mode.
# -------------------------------------------------------------------------------
def setDhcpServerClientClassFqdnArbitraryOption( mode, args ):
   if not mode.clientClassConfig.arbitraryOptionConfig:
      mode.clientClassConfig.arbitraryOptionConfig = ()
   fqdnArbitraryOptionIs( mode.clientClassConfig.arbitraryOptionConfig, args )

def noDhcpServerClientClassFqdnArbitraryOption( mode, args ):
   fqdnArbitraryOptionDel( mode.clientClassConfig.arbitraryOptionConfig, args )

# -------------------------------------------------------------------------------
# The "[no|default] option CODE [always-send] type hex data HEX"
# command, in "config-dhcp-cls-v4-assignments" mode.
# -------------------------------------------------------------------------------
def setDhcpServerClientClassHexArbitraryOption( mode, args ):
   if not mode.clientClassConfig.arbitraryOptionConfig:
      mode.clientClassConfig.arbitraryOptionConfig = ()
   hexArbitraryOptionIs( mode.clientClassConfig.arbitraryOptionConfig, args )

def noDhcpServerClientClassHexArbitraryOption( mode, args ):
   hexArbitraryOptionDel( mode.clientClassConfig.arbitraryOptionConfig, args )

def setDhcpServerClientClassMatchStringArbitraryOption( mode, args ):
   if not mode.matchCriteria.arbitraryOptionConfig:
      mode.matchCriteria.arbitraryOptionConfig = ()
   stringArbitraryOptionIs( mode.matchCriteria.arbitraryOptionConfig, args )

def noDhcpServerClientClassMatchStringArbitraryOption( mode, args ):
   stringArbitraryOptionDel( mode.matchCriteria.arbitraryOptionConfig, args )

def setDhcpServerClientClassMatchHexArbitraryOption( mode, args ):
   if not mode.matchCriteria.arbitraryOptionConfig:
      mode.matchCriteria.arbitraryOptionConfig = ()
   hexArbitraryOptionIs( mode.matchCriteria.arbitraryOptionConfig, args )

def noDhcpServerClientClassMatchHexArbitraryOption( mode, args ):
   hexArbitraryOptionDel( mode.matchCriteria.arbitraryOptionConfig, args )

def setDhcpServerClientClassMatchIpAddrArbitraryOption( mode, args ):
   if not mode.matchCriteria.arbitraryOptionConfig:
      mode.matchCriteria.arbitraryOptionConfig = ()
   ipAddrArbitraryOptionIs( mode.matchCriteria.arbitraryOptionConfig, args )

def noDhcpServerClientClassMatchIpAddrArbitraryOption( mode, args ):
   ipAddrArbitraryOptionDel( mode.matchCriteria.arbitraryOptionConfig, args )

def canSetVrf( intfName, oldVrf, newVrf, vrfDelete ):
   changed = False
   for config in dhcpServerVrfConfig.vrfConfig.values():
      hasIntf = ( intfName in config.interfacesIpv4 or
                  intfName in config.interfacesIpv6 )
      changed = changed or hasIntf
      del config.interfacesIpv4[ intfName ]
      del config.interfacesIpv6[ intfName ]
   if changed:
      msg = f'DHCP server has been deconfigured on interface {intfName}'
   else:
      msg = None
   return ( True, msg )

def canDeleteVrf( vrfName ):
   if vrfName not in dhcpServerVrfConfig.vrfConfig:
      return ( True, None )
   config = dhcpServerVrfConfig.vrfConfig[ vrfName ]
   changed = len( config.interfacesIpv4 ) > 0 or len( config.interfacesIpv6 ) > 0
   config.interfacesIpv4.clear()
   config.interfacesIpv6.clear()
   if changed:
      msg = ( 'DHCP server has been deconfigured '
              'from all interfaces in VRF {}' ).format( vrfName )
   else:
      msg = None
   return ( True, msg )

# When running "default interface <interface>", we need to deleteconfig
class DhcpServerIntf( CliPlugin.IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intfName = self.intf_.name
      for vrf in dhcpServerVrfConfig.vrfConfig:
         del dhcpServerVrfConfig.vrfConfig[ vrf ].interfacesIpv4[ intfName ]
         del dhcpServerVrfConfig.vrfConfig[ vrf ].interfacesIpv6[ intfName ]

def Plugin( entityManager ):
   global dhcpServerVrfConfig
   global dhcpServerVrfStatus
   global ipConfig
   ipConfig = LazyMount.mount( entityManager, 'ip/config', 'Ip::Config', 'r' )
   dhcpServerVrfConfig = ConfigMount.mount( entityManager, 'dhcpServer/vrf/config',
                                            'DhcpServer::VrfConfigDir', 'w' )
   dhcpServerVrfStatus = LazyMount.mount( entityManager, 'dhcpServer/vrf/status',
                                          'DhcpServer::VrfStatusDir', 'r' )
   IraIpIntfCli.canSetVrfHook.addExtension( canSetVrf )
   IraVrfCli.canDeleteVrfHook.addExtension( canDeleteVrf )
   CliPlugin.IntfCli.Intf.registerDependentClass( DhcpServerIntf )
