#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import CliCommand
import BasicCli
from CliDynamicSymbol import CliDynamicPlugin
from CliMode.DhcpServer import (
   DhcpServerGlobalClientClassBaseMode,
   DhcpServerSubnetClientClassBaseMode,
   DhcpServerRangeClientClassBaseMode,
)

# CliDynamic Plugin
assignDynamicSubmodes = CliDynamicPlugin( "DhcpServerClientAssignMode" )
matchDynamicSubmodes = CliDynamicPlugin( "DhcpServerClientMatchMode" )

def clearClientClassOptions( clientClassConfig, serverConfig, af ):
   clientClassConfig.domainName = ''
   if clientClassConfig.privateOptionConfig:
      clientClassConfig.privateOptionConfig.stringPrivateOption.clear()
      clientClassConfig.privateOptionConfig.ipAddrPrivateOption.clear()
      clientClassConfig.privateOptionConfig = None
   clientClassConfig.dnsServers.clear()
   if af == 'ipv4':
      clientClassConfig.leaseTime = serverConfig.leaseTimeIpv4Default
      clientClassConfig.defaultGateway = Arnet.IpAddr( '0.0.0.0' )
      clientClassConfig.tftpServerOption66 = clientClassConfig.tftpServerDefault
      clientClassConfig.tftpBootFileName = clientClassConfig.tftpBootFileNameDefault
   else:
      clientClassConfig.leaseTime = serverConfig.leaseTimeIpv6Default

def clearClientClassMatchConfig( clientClassConfig ):
   clientClassConfig.matchCriteriaModeConfig = None

class DhcpServerClientClassV4Mode( DhcpServerGlobalClientClassBaseMode,
                                   BasicCli.ConfigModeBase ):
   name = "DHCP Server Client Class V4"

   def __init__( self, parent, session, clientClassName ):
      self.vrf = parent.vrf
      param = ( clientClassName, self.vrf, 'ipv4' )
      DhcpServerGlobalClientClassBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig
      self.clientClassConfig = parent.dhcpServerConfig.\
            clientClassConfigIpv4.newMember( clientClassName )

class DhcpServerClientClassV6Mode( DhcpServerGlobalClientClassBaseMode,
                                   BasicCli.ConfigModeBase ):
   name = "DHCP Server Client Class V6"

   def __init__( self, parent, session, clientClassName ):
      self.vrf = parent.vrf
      param = ( clientClassName, self.vrf, 'ipv6' )
      DhcpServerGlobalClientClassBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig
      self.clientClassConfig = parent.dhcpServerConfig.\
            clientClassConfigIpv6.newMember( clientClassName )

# subnet's mode client class mode
class DhcpServerSubnetV4ClientClassMode( DhcpServerSubnetClientClassBaseMode,
                                         BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv4 Subnet's Client Class"

   def __init__( self, parent, session, clientClassName ):
      self.vrf = parent.vrf
      param = ( parent.subnetConfig, clientClassName, self.vrf, 'ipv4' )
      DhcpServerSubnetClientClassBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig

class DhcpServerSubnetV6ClientClassMode( DhcpServerSubnetClientClassBaseMode,
                                         BasicCli.ConfigModeBase ):
   name = "DHCP Server IPv6 Subnet's Client Class"

   def __init__( self, parent, session, clientClassName ):
      self.vrf = parent.vrf
      param = ( parent.subnetConfig, clientClassName, self.vrf, 'ipv6' )
      DhcpServerSubnetClientClassBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig

# Range's mode client class mode
class DhcpServerV4RangeClientClassMode( DhcpServerRangeClientClassBaseMode,
                                        BasicCli.ConfigModeBase ):
   name = "DHCP Server Ipv4 Range's Client Class"

   def __init__( self, parent, session, clientClassName ):
      self.vrf = parent.vrf
      param = ( parent.subnetConfig, parent.rangeConfig, clientClassName,
                self.vrf, 'ipv4' )
      DhcpServerRangeClientClassBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig

class DhcpServerV6RangeClientClassMode( DhcpServerRangeClientClassBaseMode,
                                        BasicCli.ConfigModeBase ):
   name = "DHCP Server Ipv6 Range's Client Class"

   def __init__( self, parent, session, clientClassName ):
      self.vrf = parent.vrf
      param = ( parent.subnetConfig, parent.rangeConfig, clientClassName,
                self.vrf, 'ipv6' )
      DhcpServerRangeClientClassBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig

class MatchExpr( CliCommand.CliExpression ):
   expression = 'match [ any ]'
   data = {
      'match': 'Specify match criteria for the client class',
      'any': "Requires clients' packets to match any of the match criteria to be "
             "included in this client class",
      }

class EnterDhcpServerAssignModeBase( CliCommand.CliCommandClass ):
   syntax = 'assignments'
   noOrDefaultSyntax = syntax

   @staticmethod
   def adapterBase( ipv4, args, v4ChildMode, v6ChildMode ):
      if ipv4:
         args[ 'af' ] = 'ipv4'
         args[ 'clientClassAssignAfMode' ] = v4ChildMode
      else:
         args[ 'af' ] = 'ipv6'
         args[ 'clientClassAssignAfMode' ] = v6ChildMode

   @staticmethod
   def handlerBase( mode, args, getOrCreateChildModeFunc ):
      childMode = getOrCreateChildModeFunc( mode, args )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandlerBase( mode, args ):
      clearClientClassOptions( mode.clientClassConfig, mode.dhcpServerConfig,
                               args[ 'af' ] )

class EnterDhcpServerMatchModeBase( CliCommand.CliCommandClass ):
   syntax = 'MATCH'
   noOrDefaultSyntax = syntax

   @staticmethod
   def adapterBase( mode, args, ipv4, v4MatchMode, v6MatchMode ):
      if ipv4:
         args[ 'af' ] = 'ipv4'
         args[ 'clientClassMatchMode' ] = v4MatchMode
      else:
         args[ 'af' ] = 'ipv6'
         args[ 'clientClassMatchMode' ] = v6MatchMode

   @staticmethod
   def handlerBase( mode, args, getOrCreateChildModeFunc ):
      matchAny = 'any' in args
      childMode = getOrCreateChildModeFunc( mode, args, matchAny )
      if not childMode.clientClassConfig.matchCriteriaModeConfig:
         childMode.clientClassConfig.matchCriteriaModeConfig = ()
      childMode.matchCriteria = childMode.clientClassConfig.matchCriteriaModeConfig
      childMode.matchCriteria.matchAny = matchAny
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandlerBase( mode, args, getOrCreateChildModeFunc ):
      clientClassConfig = mode.clientClassConfig
      matchCriteriaModeConfig = clientClassConfig.matchCriteriaModeConfig
      if matchCriteriaModeConfig:
         matchAny = 'any' in args
         childMode = getOrCreateChildModeFunc( mode, args, matchAny )
         childMode.removeComment()

         childMode.matchCriteria = None
         clearClientClassMatchConfig( clientClassConfig )

class EnterDhcpServerClientClassAssignMode( EnterDhcpServerAssignModeBase ):
   data = {
      'assignments': 'Assign DHCP options'
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      ipv4 = isinstance( mode, DhcpServerClientClassV4Mode )
      EnterDhcpServerAssignModeBase.adapterBase(
            ipv4, args,
            assignDynamicSubmodes.DhcpServerClientClassAssignV4Mode,
            assignDynamicSubmodes.DhcpServerClientClassAssignV6Mode )

   handler = "DhcpServerCliHandler.doEnterDhcpServerClientClassAssignMode"

   noOrDefaultHandler = "DhcpServerCliHandler.noEnterDhcpServerClientClassAssignMode"

class EnterDhcpServerClientClassMatchMode( EnterDhcpServerMatchModeBase ):
   data = {
      'MATCH': MatchExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      ipv4 = isinstance( mode, DhcpServerClientClassV4Mode )
      EnterDhcpServerClientClassMatchMode.adapterBase(
            mode, args, ipv4,
            matchDynamicSubmodes.DhcpServerClientClassV4MatchMode,
            matchDynamicSubmodes.DhcpServerClientClassV6MatchMode )

   handler = "DhcpServerCliHandler.doEnterDhcpServerClientClassMatchMode"

   noOrDefaultHandler = "DhcpServerCliHandler.noEnterDhcpServerClientClassMatchMode"

class EnterDhcpServerSubnetClientClassAssignMode( EnterDhcpServerAssignModeBase ):
   data = {
      'assignments': 'Assign DHCP options'
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      ipv4 = isinstance( mode, DhcpServerSubnetV4ClientClassMode )
      EnterDhcpServerAssignModeBase.adapterBase(
            ipv4, args,
            assignDynamicSubmodes.DhcpServerSubnetClientClassV4AssignMode,
            assignDynamicSubmodes.DhcpServerSubnetClientClassV6AssignMode )

   handler = "DhcpServerCliHandler.doEnterDhcpServerSubnetClientClassAssignMode"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noEnterDhcpServerSubnetClientClassAssignMode" )

class EnterDhcpServerRangeClientClassAssignMode( EnterDhcpServerAssignModeBase ):
   data = {
      'assignments': 'Assign DHCP options and IP addresses'
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      ipv4 = isinstance( mode, DhcpServerV4RangeClientClassMode )
      EnterDhcpServerAssignModeBase.adapterBase(
            ipv4, args,
            assignDynamicSubmodes.DhcpServerRangeClientClassV4AssignMode,
            assignDynamicSubmodes.DhcpServerRangeClientClassV6AssignMode )

   handler = "DhcpServerCliHandler.doEnterDhcpServerRangeClientClassAssignMode"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noEnterDhcpServerRangeClientClassAssignMode" )

class EnterDhcpServerSubnetClientClassMatchMode( EnterDhcpServerMatchModeBase ):
   data = {
      'MATCH': MatchExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      ipv4 = isinstance( mode, DhcpServerSubnetV4ClientClassMode )
      EnterDhcpServerClientClassMatchMode.adapterBase(
            mode, args, ipv4,
            matchDynamicSubmodes.DhcpServerSubnetClientClassV4MatchMode,
            matchDynamicSubmodes.DhcpServerSubnetClientClassV6MatchMode )

   handler = "DhcpServerCliHandler.doEnterDhcpServerSubnetClientClassMatchMode"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noEnterDhcpServerSubnetClientClassMatchMode" )

class EnterDhcpServerRangeClientClassMatchMode( EnterDhcpServerMatchModeBase ):
   data = {
      'MATCH': MatchExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      ipv4 = isinstance( mode, DhcpServerV4RangeClientClassMode )
      EnterDhcpServerMatchModeBase.adapterBase(
            mode, args, ipv4,
            matchDynamicSubmodes.DhcpServerRangeClientClassV4MatchMode,
            matchDynamicSubmodes.DhcpServerRangeClientClassV6MatchMode )

   handler = "DhcpServerCliHandler.doEnterDhcpServerRangeClientClassMatchMode"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noEnterDhcpServerRangeClientClassMatchMode" )

# Add Assign Mode
DhcpServerClientClassV4Mode.addCommandClass( EnterDhcpServerClientClassAssignMode )
DhcpServerClientClassV6Mode.addCommandClass( EnterDhcpServerClientClassAssignMode )

# Add Match Mode
DhcpServerClientClassV4Mode.addCommandClass( EnterDhcpServerClientClassMatchMode )
DhcpServerClientClassV6Mode.addCommandClass( EnterDhcpServerClientClassMatchMode )

# Subnet assign mode
for ClientClassMode in ( DhcpServerSubnetV4ClientClassMode,
                         DhcpServerSubnetV6ClientClassMode ):
   ClientClassMode.addCommandClass(
         EnterDhcpServerSubnetClientClassAssignMode )

# Subnet match mode
for ClientClassMode in ( DhcpServerSubnetV4ClientClassMode,
                         DhcpServerSubnetV6ClientClassMode ):
   ClientClassMode.addCommandClass( EnterDhcpServerSubnetClientClassMatchMode )

# Range Assign Mode
for ClientClassMode in ( DhcpServerV4RangeClientClassMode,
                         DhcpServerV6RangeClientClassMode ):
   ClientClassMode.addCommandClass( EnterDhcpServerRangeClientClassAssignMode )

# Range Match Mode
for ClientClassMode in ( DhcpServerV4RangeClientClassMode,
                         DhcpServerV6RangeClientClassMode ):
   ClientClassMode.addCommandClass( EnterDhcpServerRangeClientClassMatchMode )
